/***********************************************************
 Copyright (C) 2012 VeriSign, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 ***********************************************************/
package com.verisign.epp.codec.launch;

import java.util.Date;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Class to hold the launch notice information including the 
 * notice identifier, the expiration (not after) date and time, and 
 * the date and time the claims notice was accepted.  
 */
public class EPPLaunchNotice implements EPPCodecComponent {
	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPLaunchNotice.class
			.getName(), EPPCatFactory.getInstance().getFactory());

	/**
	 * Constant for the local name
	 */
	public static final String ELM_LOCALNAME = "notice";

	/**
	 * Constant for the tag name
	 */
	public static final String ELM_NAME = EPPLaunchExtFactory.NS_PREFIX + ":"
			+ ELM_LOCALNAME;

	/**
	 * Element local name for the notice identifier
	 */
	private static final String ELM_NOTICE_ID = "noticeID";

	/**
	 * Element local name for the generated date
	 */
	private static final String ELM_NOT_AFTER_DATE = "notAfter";

	/**
	 * Element local name for the accepted date
	 */
	private static final String ELM_ACCEPTED_DATE = "acceptedDate";

	/**
	 * OPTIONAL &quot;validatorID&quot; attribute name that is used to define the 
	 * Validator Identifier of the Trademark Validator.  
	 */
	private final static String ATTR_VALIDATOR_ID = "validatorID";

	
	/**
	 * Unique notice identifier generated by the source of the claims notice
	 * information like the Claims Notice Information Service (CNIS).
	 */
	String noticeId;
	
	/**
	 * OPTIONAL &quot;validatorID&quot; attribute that is used to define the 
	 * Validator Identifier of the Trademark Validator.  
	 */
	private String validatorId = null;	

	/**
	 * Contains the date and time of the expiration of the claims notice.
	 */
	Date notAfterDate;

	/**
	 * Contains the date and time that the claims notice was displayed or
	 * accepted.
	 */
	Date acceptedDate;

	/**
	 * Default constructor for <code>EPPLaunchNotice</code>.
	 */
	public EPPLaunchNotice() {
	}

	/**
	 * Constructor that takes all of he attributes of the
	 * <code>EPPLaunchNotice</code>.
	 * 
	 * @param aNoticeId
	 *            Unique notice identifier of the claims notice
	 * @param aNotAfterDate
	 *            Date and time of the expiration of the claims notice
	 * @param aAcceptedDate
	 *            Date and time when the notice was displayed or accepted
	 */
	public EPPLaunchNotice(String aNoticeId, Date aNotAfterDate,
			Date aAcceptedDate) {
		this.noticeId = aNoticeId;
		this.notAfterDate = aNotAfterDate;
		this.acceptedDate = aAcceptedDate;
	}
	
	/**
	 * Constructor that takes all of he attributes of the
	 * <code>EPPLaunchNotice</code>.
	 * 
	 * @param aNoticeId
	 *            Unique notice identifier of the claims notice
	 * @param aNotAfterDate
	 *            Date and time of the expiration of the claims notice
	 * @param aAcceptedDate
	 *            Date and time when the notice was displayed or accepted
	 * @param aValidatorId
	 *            Identifier of the Trademark Validator used to query for 
	 *            the Claims Notice information.
	 */
	public EPPLaunchNotice(String aNoticeId, Date aNotAfterDate,
			Date aAcceptedDate, String aValidatorId) {
		this(aNoticeId, aNotAfterDate, aAcceptedDate);
		this.validatorId = aValidatorId;
	}
	

	/**
	 * Clone <code>EPPLaunchNotice</code>.
	 * 
	 * @return clone of <code>EPPLaunchNotice</code>
	 * @exception CloneNotSupportedException
	 *                standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPLaunchNotice clone = (EPPLaunchNotice) super.clone();

		return clone;
	}

	/**
	 * Sets all this instance's data in the given XML document
	 * 
	 * @param aDocument
	 *            a DOM Document to attach data to.
	 * @return The root element of this component.
	 * 
	 * @throws EPPEncodeException
	 *             Thrown if any errors prevent encoding.
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {
		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null"
					+ " on in EPPLaunchNotice.encode(Document)");
		}

		if (this.noticeId == null) {
			throw new EPPEncodeException("notice identifier is not set.");
		}
		if (this.notAfterDate == null) {
			throw new EPPEncodeException("not after date is not set.");
		}
		if (this.acceptedDate == null) {
			throw new EPPEncodeException("accepted date is not set.");
		}
		Element root = aDocument.createElementNS(EPPLaunchExtFactory.NS,
				ELM_NAME);

		// Notice Identifier
		Element noticeIdElm = aDocument.createElementNS(EPPLaunchExtFactory.NS,
				EPPLaunchExtFactory.NS_PREFIX + ":" + ELM_NOTICE_ID);
		root.appendChild(noticeIdElm);
		Text textNode = aDocument.createTextNode(this.noticeId);
		noticeIdElm.appendChild(textNode);
		
		// Validator Id
		if (this.validatorId != null) {
			noticeIdElm.setAttribute(ATTR_VALIDATOR_ID, this.validatorId);
		}


		// Not After Date
		EPPUtil.encodeTimeInstant(aDocument, root, this.notAfterDate,
				EPPLaunchExtFactory.NS, EPPLaunchExtFactory.NS_PREFIX + ":"
						+ ELM_NOT_AFTER_DATE);

		// Accepted Date
		EPPUtil.encodeTimeInstant(aDocument, root, this.acceptedDate,
				EPPLaunchExtFactory.NS, EPPLaunchExtFactory.NS_PREFIX + ":"
						+ ELM_ACCEPTED_DATE);

		return root;
	}

	/**
	 * Decode the <code>EPPLaunchNotice</code> component
	 * 
	 * @param aElement
	 *            Root element of the <code>EPPLaunchNotice</code>
	 * @throws EPPDecodeException
	 *             Error decoding the <code>EPPLaunchNotice</code>
	 */
	public void decode(Element aElement) throws EPPDecodeException {

		// Notice Identifier
		this.noticeId = EPPUtil.decodeString(aElement, EPPLaunchExtFactory.NS,
				ELM_NOTICE_ID);

		Element currElm = EPPUtil.getElementByTagNameNS(aElement,
				EPPLaunchExtFactory.NS, ELM_NOTICE_ID);
		if (currElm != null && currElm.hasAttribute(ATTR_VALIDATOR_ID)) {
			this.validatorId = currElm.getAttribute(ATTR_VALIDATOR_ID);
			if (this.validatorId != null && this.validatorId.length() == 0) {
				this.validatorId = null;
			}
		}
		else {
			this.validatorId = null;
		}
		
		
		// Not After Date
		this.notAfterDate = EPPUtil.decodeTimeInstant(aElement,
				EPPLaunchExtFactory.NS, ELM_NOT_AFTER_DATE);

		// Accepted Date
		this.acceptedDate = EPPUtil.decodeTimeInstant(aElement,
				EPPLaunchExtFactory.NS, ELM_ACCEPTED_DATE);
	}

	/**
	 * implements a deep <code>EPPLaunchNotice</code> compare.
	 * 
	 * @param aObject
	 *            <code>EPPLaunchNotice</code> instance to compare with
	 * 
	 * @return true if equal false otherwise
	 */
	public boolean equals(Object aObject) {

		if (!(aObject instanceof EPPLaunchNotice)) {
			cat.error("EPPLaunchNotice.equals(): aObject is not an EPPLaunchNotice");
			return false;
		}

		EPPLaunchNotice other = (EPPLaunchNotice) aObject;

		// Notice Identifier
		if (!EqualityUtil.equals(this.noticeId, other.noticeId)) {
			cat.error("EPPLaunchNotice.equals(): noticeId not equal");
			return false;
		}
		
		// Validator Id
		if (!EqualityUtil.equals(this.validatorId, other.validatorId)) {
			cat.error("EPPLaunchNotice.equals(): validatorId not equal");
			return false;
		}
		

		// Not After Date
		if (!EqualityUtil.equals(this.notAfterDate, other.notAfterDate)) {
			cat.error("EPPLaunchNotice.equals(): notAfterDate not equal");
			return false;
		}

		// Accepted Date
		if (!EqualityUtil.equals(this.acceptedDate, other.acceptedDate)) {
			cat.error("EPPLaunchNotice.equals(): acceptedDate not equal");
			return false;
		}

		return true;
	}

	/**
	 * Gets the unique notice identifier of the claims notice.
	 * 
	 * @return Unique notice identifier if set; <code>null</code> otherwise.
	 */
	public String getNoticeId() {
		return this.noticeId;
	}

	/**
	 * Sets the unique notice identifier of the claims notice.
	 * 
	 * @param aNoticeId
	 *            Unique notice identifier
	 */
	public void setNoticeId(String aNoticeId) {
		this.noticeId = aNoticeId;
	}
	
	/**
	 * Gets the OPTIONAL Validator Identifier, which is the unique identifier 
	 * for the Trademark Validator used for the Trademark Claims Notice 
	 * information.  If undefined, the ICANN TMCH is the Validator or 
	 * if the value is &quot;tmch&quot;.
	 * 
	 * @return The Validator Identifier if defined; <code>null</code> otherwise 
	 * to indicate to use the ICANN TMCH.
	 */
	public String getValidatorId() {
		return this.validatorId;
	}

	/**
	 * Sets the OPTIONAL Validator Identifier, which is the unique identifier 
	 * for the Trademark Validator used for the Trademark Claims Notice 
	 * information.  If undefined, the ICANN TMCH is the Validator or 
	 * if the value is &quot;tmch&quot;.
	 * 
	 * @param aValidatorId Validator Identifier, where <code>null</code> or a value 
	 * of &quot;tmch&quot; can be specified to indicate the use of the ICANN TMCH.
	 */
	public void setValidatorId(String aValidatorId) {
		this.validatorId = aValidatorId;
	}
	
	/**
	 * Is the Validator Identifier defined?
	 * 
	 * @return <code>true</code> if the Validator Identifier is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasValidatorId() {
		return (this.validatorId != null ? true : false);
	}
	

	/**
	 * Gets the date and time of expiration of the claims notice.
	 * 
	 * @return Date and time of the expiration of the claims notice if set;
	 *         <code>null</code> otherwise.
	 */
	public Date getNotAfterDate() {
		return this.notAfterDate;
	}

	/**
	 * Sets the date and time of expiration of the claims notice.
	 * 
	 * @param aNotAfterDate
	 *            Date and time of expiration of the claims notice.
	 */
	public void setNotAfterDate(Date aNotAfterDate) {
		this.notAfterDate = aNotAfterDate;
	}

	/**
	 * Gets the date and time that the claims notice was displayed or accepted.
	 * 
	 * @return Date and time that the claims notice was displayed or accepted if
	 *         set; <code>null</code> otherwise.
	 */
	public Date getAcceptedDate() {
		return this.acceptedDate;
	}

	/**
	 * Sets the date and time that the claims notice was displayed or accepted.
	 * 
	 * @param aAcceptedDate
	 *            Date and time that the claims notice was displayed or
	 *            accepted.
	 */
	public void setAcceptedDate(Date aAcceptedDate) {
		this.acceptedDate = aAcceptedDate;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful;
	 *         <code>ERROR</code> otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

}
