/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.interfaces.validate.v02;

import java.util.Random;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.validate.v02.EPPValidateAddress;
import com.verisign.epp.codec.validate.v02.EPPValidateAuthInfo;
import com.verisign.epp.codec.validate.v02.EPPValidateCheckResp;
import com.verisign.epp.codec.validate.v02.EPPValidateContact;
import com.verisign.epp.codec.validate.v02.EPPValidateKeyValue;
import com.verisign.epp.codec.validate.v02.EPPValidatePostalDefinition;
import com.verisign.epp.interfaces.EPPApplicationSingle;
import com.verisign.epp.interfaces.EPPCommandException;
import com.verisign.epp.interfaces.EPPSession;
import com.verisign.epp.transport.EPPClientCon;
import com.verisign.epp.util.Environment;
import com.verisign.epp.util.TestThread;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Is a unit test of the <code>EPPValidate</code> class. The unit test will
 * initialize a session with an EPP Server, will invoke <code>EPPValidate</code>
 * check operation, and will end a session with an EPP Server. The configuration
 * file used by the unit test defaults to epp.config, but can be changed by
 * passing the file path as the first command line argument. The unit test can
 * be run in multiple threads by setting the "threads" system property. For
 * example, the unit test can be run in 2 threads with the configuration file
 * ../../epp.config with the following command: <br>
 * <br>
 * java com.verisign.epp.interfaces.EPPValidateTst -Dthreads=2 ../../epp.config
 * <br>
 * <br>
 */
public class EPPValidateTst {
  /**
   * Handle to the Singleton EPP Application instance (
   * <code>EPPApplicationSingle</code>)
   */
  private static EPPApplicationSingle app = EPPApplicationSingle.getInstance();

  /** Name of configuration file to use for test (default = epp.config). */
  private static String configFileName = "epp.config";

  /** Logging category */
  private static Logger cat = LoggerFactory.getLogger(EPPValidateTst.class);

  /** EPP Validate associated with test */
  private EPPValidate validate = null;

  /** EPP Session associated with test */
  private EPPSession session = null;

  /** Connection to the EPP Server. */
  private EPPClientCon connection = null;

  /** Current test iteration */
  private int iteration = 0;

  /**
   * Random instance for the generation of unique objects (hosts, IP addresses,
   * etc.).
   */
  private Random rd = new Random(System.currentTimeMillis());

  /**
   * JUNIT test method to implement the <code>EPPValidateTst TestCase</code>.
   * Each sub-test will be invoked in order to satisfy testing the EPPValidate
   * interface.
   */
  @Test
  public void testValidate() {
    int numIterations = 1;

    String iterationsStr = System.getProperty("iterations");

    if (iterationsStr != null) {
      numIterations = Integer.parseInt(iterationsStr);
    }

    for (this.iteration = 0; (numIterations == 0) || (this.iteration < numIterations); this.iteration++) {
      printStart("Test Suite");

      validateCheck();

      printEnd("Test Suite");
    }
  }

  /**
   * Unit test of <code>EPPValidate.sendCheck</code>.
   */
  private void validateCheck() {
    printStart("validateCheck");

    EPPValidateCheckResp response;

    try {
      // Check contacts (registrant, tech, admin, and billing) for the COM
      // TLD
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("validateCheck: Check contacts (registrant, tech, admin, and billing) for the COM TLD");

      this.validate.setTransId("ABC-12345");

      // Contact #1 - sh8013 with full info
      EPPValidateContact theContact = new EPPValidateContact("sh8013", "registrant", "COM");

      EPPValidateAddress theAddress = new EPPValidateAddress();
      theAddress.setStreets("123 Example Dr.", "Suite 100");
      theAddress.setCity("Dulles");
      theAddress.setStateProvince("VA");
      theAddress.setPostalCode("20166-6503");
      theAddress.setCountry("US");
      EPPValidatePostalDefinition thePostalDef = new EPPValidatePostalDefinition(
            EPPValidatePostalDefinition.Type.INT, "John Doe", theAddress);
      thePostalDef.setOrg("Example Inc.");
      theContact.addPostalInfo(thePostalDef);

      theContact.setVoice("+1.7035555555");
      theContact.setFax("+1.7035555556");
      theContact.setEmail("jdoe@example.com");
      theContact.setAuthInfo(new EPPValidateAuthInfo("2fooBAR"));

      theContact.addKeyValue(new EPPValidateKeyValue("VAT", "1234567890"));

      this.validate.addContact(theContact);

      // Contact #2 - existing sh8012 for tech contact
      this.validate.addContact(new EPPValidateContact("sh8012", "tech", "COM"));

      // Contact #3 - sh8014 for admin contact
      theContact = new EPPValidateContact("sh8014", "admin", "COM");

      theAddress = new EPPValidateAddress();
      theAddress.setStreets("123 Example Dr.", "Suite 100");
      theAddress.setCity("Dulles");
      theAddress.setStateProvince("VA");
      theAddress.setPostalCode("20166-6503");
      theAddress.setCountry("US");
      thePostalDef = new EPPValidatePostalDefinition(EPPValidatePostalDefinition.Type.INT, "John Doe", theAddress);
      thePostalDef.setOrg("Example Inc.");
      theContact.addPostalInfo(thePostalDef);

      theContact.setVoice("+1.7035555555");
      theContact.setFax("+1.7035555556");
      theContact.setEmail("jdoe@example.com");
      theContact.setAuthInfo(new EPPValidateAuthInfo("2fooBAR"));

      this.validate.addContact(theContact);

      // Contact #4 - sh8014 for billing contact
      try {
        theContact = (EPPValidateContact) theContact.clone();
        theContact.setContactType("billing");
        this.validate.addContact(theContact);
      }
      catch (CloneNotSupportedException e) {
        Assert.fail("CloneNotSupportedException cloning sh8014 for billing contact: " + e);
      }

      response = this.validate.sendCheck();

      System.out.println("Response Type = " + response.getType());

      System.out.println("Response.TransId.ServerTransId = " + response.getTransId().getServerTransId());

      System.out.println("Response.TransId.ServerTransId = " + response.getTransId().getClientTransId());

      // Output all of the response attributes
      System.out.println("\nvalidateCheck: Response = [" + response + "]");
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("validateCheck");
  }

  /**
   * Unit test of <code>EPPSession.initSession</code>. The session attribute is
   * initialized with the attributes defined in the EPP sample files.
   */
  private void initSession() {
    printStart("initSession");

    // Set attributes for initSession
    this.session.setClientID(Environment.getProperty("EPP.Test.clientId", "ClientX"));
    this.session.setPassword(Environment.getProperty("EPP.Test.password", "foo-BAR2"));
    this.session.setTransId("ABC-12345-XYZ");
    this.session.setVersion("1.0");
    this.session.setLang("en");

    // Initialize the session
    try {
      this.session.initSession();
    }

    catch (EPPCommandException e) {
      EPPResponse response = this.session.getResponse();

      // Is a server specified error?
      if ((response != null) && (!response.isSuccess())) {
        Assert.fail("Server Error : " + response);
      }
      else {
        e.printStackTrace();

        Assert.fail("initSession Error : " + e);
      }
    }

    printEnd("initSession");
  }

  /**
   * Unit test of <code>EPPSession.endSession</code>. The session with the EPP
   * Server will be terminated.
   */
  private void endSession() {
    printStart("endSession");

    this.session.setTransId("ABC-12345-XYZ");

    // End the session
    try {
      this.session.endSession();
    }

    catch (EPPCommandException e) {
      EPPResponse response = this.session.getResponse();

      // Is a server specified error?
      if ((response != null) && (!response.isSuccess())) {
        Assert.fail("Server Error : " + response);
      }

      else // Other error
      {
        e.printStackTrace();

        Assert.fail("initSession Error : " + e);
      }
    }

    printEnd("endSession");
  }

  /**
   * JUNIT <code>setUp</code> method, which sets the default client Id to
   * "theRegistrar".
   */
  @Before
  public void setUp() {
    try {
      String theSessionClassName = System.getProperty("EPP.SessionClass");

      if (theSessionClassName != null) {
        try {
          Class theSessionClass = Class.forName(theSessionClassName);

          if (!EPPSession.class.isAssignableFrom((theSessionClass))) {
            Assert.fail(theSessionClassName + " is not a subclass of EPPSession");
          }

          this.session = (EPPSession) theSessionClass.getDeclaredConstructor().newInstance();
        }
        catch (Exception ex) {
          Assert.fail("Exception instantiating EPP.SessionClass value " + theSessionClassName + ": " + ex);
        }
      }
      else {
        this.session = new EPPSession();
      }

    }

    catch (Exception e) {
      e.printStackTrace();

      Assert.fail("Error initializing the session: " + e);
    }

    initSession();

    this.validate = new EPPValidate(this.session);
  }

  /**
   * JUNIT <code>tearDown</code>, which currently does nothing.
   */
  @After
  public void tearDown() {
    endSession();
  }

  /**
   * JUNIT <code>suite</code> static method, which returns the tests associated
   * with <code>EPPValidateTst</code>.
   */
  @BeforeClass
  public static void beforeClass() {
    String theConfigFileName = System.getProperty("EPP.ConfigFile");
    if (theConfigFileName != null) {
      configFileName = theConfigFileName;
    }

    try {
      app.initialize(configFileName);
    }

    catch (EPPCommandException e) {
      e.printStackTrace();

      Assert.fail("Error initializing the EPP Application: " + e);
    }
  }

  /**
   * Handle an <code>EPPCommandException</code>, which can be either a server
   * generated error or a general exception. If the exception was caused by a
   * server error, "Server Error :&lt;Response XML&gt;" will be specified. If
   * the exception was caused by a general algorithm error, "General Error
   * :&lt;Exception Description&gt;" will be specified.
   *
   * @param aException
   *           Exception thrown during test
   */
  public void handleException(Exception aException) {
    EPPResponse response = this.session.getResponse();

    aException.printStackTrace();

    // Is a server specified error?
    if ((response != null) && (!response.isSuccess())) {
      Assert.fail("Server Error : " + response);
    }

    else {
      Assert.fail("General Error : " + aException);
    }
  }

  /**
   * Makes a unique contact name using the current time.
   *
   * @return Unique contact name <code>String</code>
   */
  public String makeContactName() {
    long tm = System.currentTimeMillis();

    return new String("Con" + String.valueOf(tm + this.rd.nextInt(5)).substring(7));
  }

  /**
   * Print the start of a test with the <code>Thread</code> name if the current
   * thread is a <code>TestThread</code>.
   *
   * @param aTest
   *           name for the test
   */
  private void printStart(String aTest) {
    if (Thread.currentThread() instanceof TestThread) {
      System.out.print(Thread.currentThread().getName() + ", iteration " + this.iteration + ": ");

      cat.info(Thread.currentThread().getName() + ", iteration " + this.iteration + ": " + aTest + " Start");
    }

    System.out.println("Start of " + aTest);

    System.out.println("****************************************************************\n");
  }

  /**
   * Print the end of a test with the <code>Thread</code> name if the current
   * thread is a <code>TestThread</code>.
   *
   * @param aTest
   *           name for the test
   */
  private void printEnd(String aTest) {
    System.out.println("****************************************************************");

    if (Thread.currentThread() instanceof TestThread) {
      System.out.print(Thread.currentThread().getName() + ", iteration " + this.iteration + ": ");

      cat.info(Thread.currentThread().getName() + ", iteration " + this.iteration + ": " + aTest + " End");
    }

    System.out.println("End of " + aTest);

    System.out.println("\n");
  }

  /**
   * Print message
   *
   * @param aMsg
   *           message to print
   */
  private void printMsg(String aMsg) {
    if (Thread.currentThread() instanceof TestThread) {
      System.out.println(Thread.currentThread().getName() + ", iteration " + this.iteration + ": " + aMsg);

      cat.info(Thread.currentThread().getName() + ", iteration " + this.iteration + ": " + aMsg);
    }

    else {
      System.out.println(aMsg);

      cat.info(aMsg);
    }
  }

  /**
   * Print error message
   *
   * @param aMsg
   *           errpr message to print
   */
  private void printError(String aMsg) {
    if (Thread.currentThread() instanceof TestThread) {
      System.err.println(Thread.currentThread().getName() + ", iteration " + this.iteration + ": " + aMsg);

      cat.error(Thread.currentThread().getName() + ", iteration " + this.iteration + ": " + aMsg);
    }

    else {
      System.err.println(aMsg);

      cat.error(aMsg);
    }
  }
}
