/***********************************************************
Copyright (C) 2020 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.serverstub.unhandlednamespaces.v1_0;

import java.util.GregorianCalendar;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.changepoll.EPPChangeCaseId;
import com.verisign.epp.codec.changepoll.EPPChangeData;
import com.verisign.epp.codec.changepoll.EPPChangeOperation;
import com.verisign.epp.codec.domain.EPPDomainContact;
import com.verisign.epp.codec.domain.EPPDomainCreateCmd;
import com.verisign.epp.codec.domain.EPPDomainInfoCmd;
import com.verisign.epp.codec.domain.EPPDomainInfoResp;
import com.verisign.epp.codec.domain.EPPDomainMapFactory;
import com.verisign.epp.codec.domain.EPPDomainStatus;
import com.verisign.epp.codec.domain.EPPDomainTransferResp;
import com.verisign.epp.codec.gen.EPPAuthInfo;
import com.verisign.epp.codec.gen.EPPMsgQueue;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPResult;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.rgpext.EPPRgpExtInfData;
import com.verisign.epp.codec.rgpext.EPPRgpExtStatus;
import com.verisign.epp.codec.secdnsext.v11.EPPSecDNSAlgorithm;
import com.verisign.epp.codec.secdnsext.v11.EPPSecDNSExtDsData;
import com.verisign.epp.codec.secdnsext.v11.EPPSecDNSExtInfData;
import com.verisign.epp.codec.unhandlednamespaces.v1_0.EPPUnhandledNamespacesResponseFilter;
import com.verisign.epp.exception.EPPException;
import com.verisign.epp.framework.EPPEvent;
import com.verisign.epp.framework.EPPEventResponse;
import com.verisign.epp.framework.EPPPollQueueException;
import com.verisign.epp.framework.EPPPollQueueMgr;
import com.verisign.epp.serverstub.DomainHandler;
import com.verisign.epp.serverstub.SessionData;

/**
 * The {@code UnhandledNamespacesDomainHandler} class is a concrete
 * implementation of the abstract {@code UnhandledNamespacesDomainHandler}
 * class. It defines the Server's behavior with implementing the Unhandled
 * Namespaces {@code RFC 9038}.
 *
 * @see com.verisign.epp.framework.EPPEvent
 * @see com.verisign.epp.framework.EPPEventResponse
 */
public class UnhandledNamespacesDomainHandler extends DomainHandler {

  /** Logger */
  private static Logger cat = LoggerFactory.getLogger(UnhandledNamespacesDomainHandler.class);

  /**
   * Unhandled namespace filter used to move unhandled namespace extensions
   * into &lt;extValue&gt; elements.
   */
  EPPUnhandledNamespacesResponseFilter unhandledNamespaceFilter;

  /**
   * Constructs an instance of UnhandledNamespacesDomainHandler
   */
  public UnhandledNamespacesDomainHandler() {
    this.unhandledNamespaceFilter = new EPPUnhandledNamespacesResponseFilter();

  }

  /**
   * Invoked when a Domain Create command is received. This overridden method
   * will call the {@link DomainHandler}, but will add the two poll messages if
   * the domain is "un-poll-messages.com". The poll messages are based on the
   * examples in draft-ietf-regext-unhandled-namespaces, which include:<br>
   * <ol>
   * <li>Domain transfer auto approve poll message that can be filtered based
   * on support for the domain mapping.</li>
   * <li>Domain change poll that can be filtered based on support for the
   * domain mapping and/or the change poll extension.</li>
   * </ol>
   * It is up to the client to consume the poll messages with a different set
   * of login services to exercise the filtering of the poll messages.
   *
   * @param aEvent
   *           The <code>EPPEvent</code> that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           <code>EPPDomaindHandler</code>
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  @Override
  public EPPEventResponse doDomainCreate(EPPEvent aEvent, Object aData) {

    EPPDomainCreateCmd theCommand = (EPPDomainCreateCmd) aEvent.getMessage();

    cat.debug("doDomainCreate: command = [" + theCommand + "]");

    EPPTransId theTransId = new EPPTransId(theCommand.getTransId(), "54321-XYZ");

    // Domain name "un-poll-messages.com" to insert the poll messages?
    if (theCommand.getName().equalsIgnoreCase("un-poll-messages.com")) {

      // Add change poll message
      EPPDomainInfoResp theChangePollMsg = new EPPDomainInfoResp();

      EPPMsgQueue theMsgQueue = new EPPMsgQueue();
      theMsgQueue.setMsg("Registry initiated update of domain.");
      theChangePollMsg.setMsgQueue(theMsgQueue);
      theChangePollMsg.setName(theCommand.getName());
      theChangePollMsg.setRoid("EXAMPLE1-REP");
      theChangePollMsg.setRegistrant("jd1234");
      theChangePollMsg.addContact(new EPPDomainContact("sh8013", EPPDomainContact.TYPE_ADMINISTRATIVE));
      theChangePollMsg.addContact(new EPPDomainContact("sh8013", EPPDomainContact.TYPE_TECHNICAL));
      theChangePollMsg.setClientId("ClientX");
      theChangePollMsg.setCreatedBy("ClientY");
      theChangePollMsg.setLastUpdatedBy("ClientX");
      theChangePollMsg.setCreatedDate(new GregorianCalendar(2012, 5, 3).getTime());
      theChangePollMsg.setLastUpdatedDate(new GregorianCalendar(2013, 11, 22).getTime());
      theChangePollMsg.setExpirationDate(new GregorianCalendar(2014, 5, 3).getTime());
      theChangePollMsg.setTransId(new EPPTransId(theCommand.getTransId(), "54322-XYZ"));
      theChangePollMsg.setResult(EPPResult.SUCCESS);
      theChangePollMsg.addStatus(new EPPDomainStatus(EPPDomainStatus.ELM_STATUS_SERVER_UPDATE_PROHIBITED));
      theChangePollMsg.addStatus(new EPPDomainStatus(EPPDomainStatus.ELM_STATUS_SERVER_DELETE_PROHIBITED));
      theChangePollMsg.addStatus(new EPPDomainStatus(EPPDomainStatus.ELM_STATUS_SERVER_TRANSFER_PROHIBITED));

      // Add Change Poll Extension
      EPPChangeData changeData = new EPPChangeData(new EPPChangeOperation(EPPChangeOperation.OPERATION_UPDATE),
            new GregorianCalendar(2013, 11, 22).getTime(), "12345-XYZ", "URS Admin", EPPChangeData.STATE_AFTER,
            new EPPChangeCaseId("urs123", EPPChangeCaseId.TYPE_URS), "URS Lock", null);

      theChangePollMsg.addExtension(changeData);

      try {
        EPPPollQueueMgr.getInstance().put(null, EPPDomainMapFactory.NS, theChangePollMsg, null);
      }
      catch (EPPPollQueueException ex) {
        cat.error("doDomainCreate: Error adding change poll message [" + theChangePollMsg + "]: " + ex);

        EPPResult theResult = new EPPResult(EPPResult.COMMAND_FAILED);
        EPPResponse theResponse = new EPPResponse(theTransId, theResult);

        return new EPPEventResponse(theResponse);
      }

      // Add transfer poll message
      EPPDomainTransferResp theTransMsg = new EPPDomainTransferResp();

      theMsgQueue = new EPPMsgQueue();
      theMsgQueue.setMsg("Transfer Auto Approved.");
      theTransMsg.setActionClient("ClientY");
      theTransMsg.setActionDate(new GregorianCalendar(2018, 8, 24).getTime());
      theTransMsg.setExpirationDate(new GregorianCalendar(2018, 8, 24).getTime());
      theTransMsg.setName(theCommand.getName());
      theTransMsg.setRequestClient("ClientX");
      theTransMsg.setRequestDate(new GregorianCalendar(2018, 8, 24).getTime());
      theTransMsg.setTransferStatus(EPPResponse.TRANSFER_SERVER_APPROVED);

      try {
        EPPPollQueueMgr.getInstance().put(null, EPPDomainMapFactory.NS, theTransMsg, null);
      }
      catch (EPPPollQueueException ex) {
        cat.error("doDomainCreate: Error adding transfer server approve poll message [" + theTransMsg + "]: " + ex);

        EPPResult theResult = new EPPResult(EPPResult.COMMAND_FAILED);
        EPPResponse theResponse = new EPPResponse(theTransId, theResult);

        return new EPPEventResponse(theResponse);
      }

    }

    EPPEventResponse theEventResponse = super.doDomainCreate(aEvent, aData);

    return theEventResponse;
  }

  /**
   * Invoked when a Domain Info command is received. This overridden method
   * will return the domain info response from
   * draft-ietf-regext-unhandled-namespaces with filtering of the DNSSEC
   * extension in the response if not supported based on the login extension
   * services for two domain names "un-no-dnssec.com" and "un-no-rgp.com". The
   * two domain names will drive the inclusion of the desired extension that
   * will get filtered if the login services don't include support for the
   * refernce extension (dnssec or rgp).
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPDomaindHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  @Override
  public EPPEventResponse doDomainInfo(EPPEvent aEvent, Object aData) {
    SessionData sessionData = (SessionData) aData;

    EPPDomainInfoCmd theCommand = (EPPDomainInfoCmd) aEvent.getMessage();

    cat.debug("doDomainInfo: command = [" + theCommand + "]");

    EPPTransId theTransId = new EPPTransId(theCommand.getTransId(), "54322-XYZ");

    // Create domain info response per draft-ietf-regext-unhandled-namespaces

    if (theCommand.getName().equalsIgnoreCase("un-no-dnssec.com")
          || theCommand.getName().equalsIgnoreCase("un-no-rgp.com")) {

      EPPDomainInfoResp theResponse = new EPPDomainInfoResp();

      theResponse.setName(theCommand.getName());
      theResponse.setRoid("EXAMPLE1-REP");
      theResponse.setRegistrant("jd1234");
      theResponse.addContact(new EPPDomainContact("sh8013", EPPDomainContact.TYPE_ADMINISTRATIVE));
      theResponse.addContact(new EPPDomainContact("sh8013", EPPDomainContact.TYPE_TECHNICAL));
      theResponse.addNs("ns1.example.com");
      theResponse.addNs("ns2.example.com");
      theResponse.addHost("ns1.example.com");
      theResponse.addHost("ns2.example.com");
      theResponse.setClientId("ClientX");
      theResponse.setCreatedBy("ClientY");
      theResponse.setLastUpdatedBy("ClientX");
      theResponse.setCreatedDate(new GregorianCalendar(1999, 4, 3).getTime());
      theResponse.setLastUpdatedDate(new GregorianCalendar(1999, 12, 3).getTime());
      theResponse.setLastTransferDate(new GregorianCalendar(2000, 4, 8).getTime());
      theResponse.setExpirationDate(new GregorianCalendar(2005, 4, 3).getTime());
      theResponse.setAuthInfo(new EPPAuthInfo("2fooBAR"));
      theResponse.setTransId(theTransId);
      theResponse.setResult(EPPResult.SUCCESS);

      if (theCommand.getName().equalsIgnoreCase("un-no-dnssec.com")) {

        theResponse.addStatus(new EPPDomainStatus(EPPDomainStatus.ELM_STATUS_OK));

        EPPSecDNSExtInfData dnssecData = new EPPSecDNSExtInfData();
        EPPSecDNSExtDsData dsData = new EPPSecDNSExtDsData(12345, EPPSecDNSAlgorithm.DSA,
              EPPSecDNSExtDsData.SHA1_DIGEST_TYPE, "49FD46E6C4B45C55D4AC");
        dnssecData.appendDsData(dsData);
        theResponse.addExtension(dnssecData);
      }
      else { // theMessage.getName().equalsIgnoreCase("un-no-rgp.com")
        // Set EPP status to pendingDelete
        theResponse.addStatus(new EPPDomainStatus(EPPDomainStatus.ELM_STATUS_PENDING_DELETE));

        EPPRgpExtInfData rgpExt = new EPPRgpExtInfData();

        EPPRgpExtStatus rgpStatus = new EPPRgpExtStatus();
        rgpStatus.setStatus(EPPRgpExtStatus.REDEMPTION_PERIOD);
        rgpExt.addStatus(rgpStatus);

        theResponse.addExtension(rgpExt);
      }

      // Filter the domain info response based on the login extension
      // services
      try

      {
        theResponse = (EPPDomainInfoResp) this.unhandledNamespaceFilter.filter(theResponse,
              sessionData.getLoginCmd().getExtensionServices(), aData);
      }
      catch (EPPException ex) {
        cat.error(
              "UnhandledNamespacesDomainHandler.doDomainInfo(): Exception unhandled namespaces filtering response ["
                    + theResponse + "]: " + ex);

        EPPResult theResult = new EPPResult(EPPResult.COMMAND_FAILED);
        EPPResponse theErrResponse = new EPPResponse(theTransId, theResult);

        return new EPPEventResponse(theErrResponse);
      }

      return new EPPEventResponse(theResponse);
    }
    else {
      return super.doDomainInfo(aEvent, aData);
    }

  }
}
