/***********************************************************
Copyright (C) 2023 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.serverstub;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.host.EPPHostCreateCmd;
import com.verisign.epp.codec.host.EPPHostInfoCmd;
import com.verisign.epp.codec.host.EPPHostUpdateCmd;
import com.verisign.epp.framework.EPPEvent;
import com.verisign.epp.framework.EPPEventResponse;
import com.verisign.epp.serverstub.v1_0.TtlV1_0HostHandler;

/**
 * The {@code TtlHostHandler} class is a concrete implementation of the
 * abstract {@code com.verisign.epp.framework.EPPHostHandler} class. It
 * handles accepting commands in a server that supports more than one version of
 * the TTL extension. The version specific {@code TtlSubHostHandler} is
 * called based on the TTL extension version passed or the set of login services
 * provided by the client.
 */
public class TtlHostHandler extends HostHandler {

  /** Used for logging */
  private static Logger cat = LoggerFactory.getLogger(TtlHostHandler.class);

  // Sub-handler for handling version 1.0 extension messages.
  TtlV1_0HostHandler v1_0SubHandler = new TtlV1_0HostHandler();


  /**
   * Invoked when a Host Info command is received.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPHostdHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doHostInfo(EPPEvent aEvent, Object aData) {
    SessionData theSessionData = (SessionData) aData;
    EPPHostInfoCmd theCommand = (EPPHostInfoCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doHostInfo(aEvent, aData);
    
    // v1_0 of TTL info?
    if (theSessionData.getLoginCmd().hasExtensionService(com.verisign.epp.codec.ttl.v1_0.EPPTtlExtFactory.NS)) {
      this.v1_0SubHandler.doHostInfo(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);      
    }

    return theEventResponse;
  }

  /**
   * Invoked when a Host Create command is received.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPHostdHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doHostCreate(EPPEvent aEvent, Object aData) {
    EPPHostCreateCmd theCommand = (EPPHostCreateCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doHostCreate(aEvent, aData);

    // v1_0 of TTL create?
    if (theCommand.hasExtension(com.verisign.epp.codec.ttl.v1_0.EPPTtlCreate.class)) {
      this.v1_0SubHandler.doHostCreate(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);
    }
    else {
      cat.info("doHostCreate: TTL create = null");
      System.out.println("doHostCreate: TTL create = null");
    }

    return theEventResponse;
  }


  /**
   * Invoked when a Host Update command is received.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPHostdHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doHostUpdate(EPPEvent aEvent, Object aData) {
    EPPHostUpdateCmd theCommand = (EPPHostUpdateCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doHostUpdate(aEvent, aData);

    // v1_0 of TTL update?
    if (theCommand.hasExtension(com.verisign.epp.codec.ttl.v1_0.EPPTtlUpdate.class)) {
      this.v1_0SubHandler.doHostUpdate(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);
    }
    else {
      cat.info("doHostUpdate: TTL update = null");
      System.out.println("doHostUpdate: TTL update = null");
    }

    return theEventResponse;
  }

}
