/***********************************************************
Copyright (C) 2023 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.serverstub;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.domain.EPPDomainCreateCmd;
import com.verisign.epp.codec.domain.EPPDomainInfoCmd;
import com.verisign.epp.codec.domain.EPPDomainUpdateCmd;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.framework.EPPEvent;
import com.verisign.epp.framework.EPPEventResponse;
import com.verisign.epp.serverstub.v1_0.TtlV1_0DomainHandler;

/**
 * The {@code TtlDomainHandler} class is a concrete implementation of the
 * abstract {@code com.verisign.epp.framework.EPPDomainHandler} class. It
 * handles accepting commands in a server that supports more than one version of
 * the TTL extension. The version specific {@code TtlSubDomainHandler} is
 * called based on the TTL extension version passed or the set of login services
 * provided by the client.
 */
public class TtlDomainHandler extends DomainHandler {

  /** Used for logging */
  private static Logger cat = LoggerFactory.getLogger(TtlDomainHandler.class);

  // Sub-handler for handling version 1.0 extension messages.
  TtlV1_0DomainHandler v1_0SubHandler = new TtlV1_0DomainHandler();


  /**
   * Invoked when a Domain Info command is received.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPDomaindHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doDomainInfo(EPPEvent aEvent, Object aData) {
    SessionData theSessionData = (SessionData) aData;
    EPPDomainInfoCmd theCommand = (EPPDomainInfoCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doDomainInfo(aEvent, aData);
    
    // v1_0 of TTL info?
    if (theSessionData.getLoginCmd().hasExtensionService(com.verisign.epp.codec.ttl.v1_0.EPPTtlExtFactory.NS)) {
      this.v1_0SubHandler.doDomainInfo(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);      
    }

    return theEventResponse;
  }

  /**
   * Invoked when a Domain Create command is received.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPDomaindHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doDomainCreate(EPPEvent aEvent, Object aData) {
    EPPDomainCreateCmd theCommand = (EPPDomainCreateCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doDomainCreate(aEvent, aData);

    // v1_0 of TTL create?
    if (theCommand.hasExtension(com.verisign.epp.codec.ttl.v1_0.EPPTtlCreate.class)) {
      this.v1_0SubHandler.doDomainCreate(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);
    }
    else {
      cat.info("doDomainCreate: TTL create = null");
      System.out.println("doDomainCreate: TTL create = null");
    }

    return theEventResponse;
  }


  /**
   * Invoked when a Domain Update command is received.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPDomaindHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doDomainUpdate(EPPEvent aEvent, Object aData) {
    EPPDomainUpdateCmd theCommand = (EPPDomainUpdateCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doDomainUpdate(aEvent, aData);

    // v1_0 of TTL update?
    if (theCommand.hasExtension(com.verisign.epp.codec.ttl.v1_0.EPPTtlUpdate.class)) {
      this.v1_0SubHandler.doDomainUpdate(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);
    }
    else {
      cat.info("doDomainUpdate: TTL update = null");
      System.out.println("doDomainUpdate: TTL update = null");
    }

    return theEventResponse;
  }

}
