/***********************************************************
Copyright (C) 2023 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 ***********************************************************/
package com.verisign.epp.codec.ttl.v1_0;

import java.security.InvalidParameterException;

/**
 * {@code RRecType} specifies the DNS resource record type to which the TTL
 * value pertains. This attribute MUST have one of the following values: "NS",
 * "DS", "DNAME", "A", "AAAA" or "custom" , which include the following possible
 * resource record values:<br>
 * <ul>
 * <li>{@code NS}</li>
 * <li>{@code DS}</li>
 * <li>{@code DNAME}</li>
 * <li>{@code A}</li>
 * <li>{@code AAAA}</li>
 * <li>{@code CUSTOM} - Custom resource record that must be defined using a
 * separate attribute</li>
 * </ul>
 */
public enum RRecType {
  NS("NS"), DS("DS"), DNAME("DNAME"), A("A"), AAAA("AAAA"), CUSTOM("custom");

  /**
   * Resource record type string representation
   */
  private final String rrecStr;

  /**
   * Define the resource record type string value for the enumerated value.
   * 
   * @param aRRecStr
   *           resource record type enumerated value string
   */
  RRecType(String aRRecStr) {
    this.rrecStr = aRRecStr;
  }

  /**
   * Gets the {@code RRecType} enumerated instance given the matching string.
   * 
   * @param aString
   *           {@code RRecType} enumerated string to convert to an enumerated
   *           {@code RRecType} instance.
   * 
   * @return Enumerated {@code RRecType} value matching the {@code String}.
   * 
   * @throws InvalidParameterException
   *            If {@code aString} does not match an enumerated
   *            {@code RRecType} string value.
   */
  public static RRecType getRRecType(String aString) {
    if (aString.equals(NS.rrecStr)) {
      return NS;
    }
    else if (aString.equals(DS.rrecStr)) {
      return DS;
    }
    else if (aString.equals(DNAME.rrecStr)) {
      return DNAME;
    }
    else if (aString.equals(A.rrecStr)) {
      return A;
    }
    else if (aString.equals(AAAA.rrecStr)) {
      return AAAA;
    }
    else if (aString.equals(CUSTOM.rrecStr)) {
      return CUSTOM;
    }
    else {
      throw new InvalidParameterException("RRecType enum value of " + aString + " is not valid.");
    }
  }

  /**
   * Convert the enumerated {@code RRecType} instance value to a
   * {@code String}.
   */
  public String toString() {
    return this.rrecStr;
  }

}
