/***********************************************************
Copyright (C) 2023 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.ttl.v1_0;

// W3C Imports
import java.util.LinkedHashSet;
import java.util.Set;

import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
//----------------------------------------------
// Imports
//----------------------------------------------
// SDK Imports
import com.verisign.epp.codec.gen.EPPExtFactory;
import com.verisign.epp.codec.gen.EPPProtocolExtension;
import com.verisign.epp.codec.gen.EPPService;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Factory for the TTL Extension objects. The TTL Extension only supports
 * command, response extensions with {@code createExtension}, but currently
 * does not support protocol extensions with
 * {@code createProtocolExtension}. Calling
 * {@code createProtocolExtension} will result in an exception.
 */
public class EPPTtlExtFactory extends EPPExtFactory {
  /** XML Namespace */
  public static final String NS = "urn:ietf:params:xml:ns:epp:ttl-1.0";

  /** XML Nameespace Prefix */
  public static final String NS_PREFIX = "ttl";

  /**
   * Service object associated with {@code EPPTtlExtFactory}. The service
   * object is used when creating the Greeting or the Login.
   */
  private EPPService service;

  /**
   * Default constructor for {@code EPPTtlExtFactory}.
   */
  public EPPTtlExtFactory() {
    service = new EPPService(NS_PREFIX, NS);
    service.setServiceType(EPPService.EXT_SERVICE);
  }

  /**
   * Creates a concrete {@code EPPCodecComponent} from an XML element
   * that represents an EPP extension.
   *
   * @param aExtensionElm
   *            Extension XML element.
   *
   * @return Concrete {@code EPPCodecComponent} associated with the
   *         extension XML element.
   *
   * @exception EPPCodecException
   *                Error creating the concrete {@code EPPCodecComponent}
   */
  public EPPCodecComponent createExtension(Element aExtensionElm)
      throws EPPCodecException {
    String name = aExtensionElm.getLocalName();

    if (!aExtensionElm.getNamespaceURI().equals(NS)) {
      throw new EPPCodecException("Invalid extension type " + name);
    }

    if (name.equals(EPPUtil.getLocalName(EPPTtlInfo.ELM_NAME))) {
      return new EPPTtlInfo();
    }
    else if (name.equals(EPPUtil.getLocalName(EPPTtlInfData.ELM_NAME))) {
      return new EPPTtlInfData();
    }
    else if (name.equals(EPPUtil.getLocalName(EPPTtlCreate.ELM_NAME))) {
      return new EPPTtlCreate();
    }
    else if (name.equals(EPPUtil.getLocalName(EPPTtlUpdate.ELM_NAME))) {
      return new EPPTtlUpdate();
    }
    else {
      throw new EPPCodecException("Invalid extension element " + name);
    }
  }

  /**
   * Creates a concrete {@code EPPProtocolExtension} from an XML element
   * that represents an EPP protocol extension.
   *
   * @param aExtensionElm
   *            extension XML element.
   *
   * @return Concrete {@code EPPProtocolExtension} associated with the
   *         extension XML element.
   *
   * @exception EPPCodecException
   *                Error creating the concrete
   *                {@code EPPProtocolExtension}
   */
  public EPPProtocolExtension createProtocolExtension(Element aExtensionElm)
      throws EPPCodecException {
    throw new EPPCodecException(
        "EPPTtlExtFactory.createProtocolExtension: Protocol extensions not supported");
  }

  /**
   * Gets the service information associated with the concrete
   * {@code EPPExtFactory}. The service information is used by
   * {@code EPPFactory} for extracting the XML namespace associated with
   * the extension factory.
   *
   * @return service description associated with the concrete
   *         {@code EPPExtFactory}.
   */
  public EPPService getService() {
    return service;
  }

  /**
   * Gets the list of XML schemas that need to be pre-loaded into the XML
   * Parser.
   *
   * @return {@code Set} of {@code String} XML Schema names that
   *         should be pre-loaded in the XML Parser.
   * 
   * @see com.verisign.epp.codec.gen.EPPMapFactory#getXmlSchemas()
   */
  public Set getXmlSchemas() {
    Set theSchemas = new LinkedHashSet();
    theSchemas.add("ttl-1.0.xsd");
    return theSchemas;
  }

}
