/***********************************************************
Copyright (C) 2023 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.ttl.v1_0;

import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPEncodeException;

/**
 * TTL Create Extension that enables a client to pass the TTL values with a
 * create command.
 */
public class EPPTtlCreate extends EPPTtlExtension {

  private static final long serialVersionUID = 1L;

  /**
   * XML local name for {@code EPPTtlCreate}.
   */
  public static final String ELM_LOCALNAME = "create";

  /**
   * XML root tag for {@code EPPTtlCreate}.
   */
  public static final String ELM_NAME = EPPTtlExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * Returns the local name &quot;create&quot; for the create extension.
   * 
   * @return Local name &quot;create&quot; for the create extension.
   */
  protected String getLocalName() {
    return ELM_LOCALNAME;
  }

  /**
   * Default constructor for {@code EPPTtlCreate}.
   */
  public EPPTtlCreate() {
  }

  /**
   * Constructor for {@code EPPTtlCreate} that takes a single TTL.
   * 
   * @param aTtl
   *           A single TTL value
   */
  public EPPTtlCreate(EPPTtl aTtl) {
    this.addTtl(aTtl);
  }

  /**
   * Constructor for {@code EPPTtlCreate} that a list of TTLs.
   * 
   * @param aTtls
   *           A list of TTLs
   */
  public EPPTtlCreate(List<EPPTtl> aTtls) {
    this.setTtls(aTtls);
  }

  /**
   * Encode of a {@code EPPTtlCreate} instance that includes checks of the
   * TTLs.
   * 
   * @param aDocument
   *           DOM Document, which acts is an Element factory
   * 
   * @return Element Root element associated with the object
   * 
   * @exception EPPEncodeException
   *               Error encoding {@code EPPTtlExtension}
   */
  public Element encode(Document aDocument) throws EPPEncodeException {

    // Clear any set policy TTL values
    if (super.hasTtls()) {

      for (EPPTtl ttl : super.getTtls()) {
        ttl.unsetPolicyValues();
      }
    }

    return super.encode(aDocument);
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   * 
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPTtlExtFactory.NS;
  }

}
