/***********************************************************
Copyright (C) 2023 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 ***********************************************************/
package com.verisign.epp.codec.ttl.v1_0;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPTtl} is used to define TTL values for the DNS resource records
 * associated with domain and host objects.
 */
public class EPPTtl implements EPPCodecComponent {

  private static final long serialVersionUID = 1L;

  /**
   * Category for logging
   */
  private static Logger cat = LoggerFactory.getLogger(EPPTtl.class);

  /**
   * Constant for the phase local name
   */
  public static final String ELM_LOCALNAME = "ttl";

  /**
   * Constant for the phase qualified name (prefix and local name)
   */
  public static final String ELM_NAME = EPPTtlExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * Attribute name for the DNS record type to which the TTL value pertains.
   * This attribute MUST have one of the following values: "NS", "DS", "DNAME",
   * "A", "AAAA" or "custom"
   */
  private static final String ATTR_FOR = "for";

  /**
   * OPTIONAL "custom" attribute that must be set when the "for" attribute is
   * set to {@code RRecType#CUSTOM}.
   */
  private final static String ATTR_CUSTOM = "custom";

  /**
   * OPTIONAL "min" attribute that indicates the lowest value that may be set.
   */
  private final static String ATTR_MIN = "min";

  /**
   * OPTIONAL "max" attribute that indicates the highest value that may be set.
   */
  private final static String ATTR_MAX = "max";

  /**
   * OPTIONAL "default" attribute that indicates the default value.
   */
  private final static String ATTR_DEFAULT = "default";

  /**
   * DNS record type to which the TTL value pertains. This attribute MUST have
   * one of the following values: "NS", "DS", "DNAME", "A", "AAAA" or "custom"
   */
  private RRecType rrecType;

  /**
   * OPTIONAL "custom" resource record type value.
   */
  private String custom;

  /**
   * Lowest value that may be set. This attribute MUST only be set in the info
   * response.
   */
  private Integer minValue;

  /**
   * Highest value that may be set. This attribute MUST only be set in the info
   * response.
   */
  private Integer maxValue;

  /**
   * Default value set by the server. This attribute MUST only be set in the
   * info response.
   */
  private Integer defaultValue;

  /**
   * Value of the TTL, which is non-negative between 0 and 2147483647. The
   * value can be set to {@code null} to indicate that it's not explicitly set.
   */
  private Integer value;

  /**
   * Default constructor. By default the value is {@code null}, which will
   * unset the TTL setting.
   */
  public EPPTtl() {
  }

  /**
   * Create {@code EPPTtl} instance with the TTL resource record type and with
   * a {@code null} TTL value.
   * 
   * @param aRRecType
   *           Required resource record type
   */
  public EPPTtl(RRecType aRRecType) {
    this.setRRecType(aRRecType);
  }

  /**
   * Create {@code EPPTtl} instance with a custom TTL resource record type and
   * with a {@code null} TTL value.
   * 
   * @param aRRecType
   *           Required resource record type
   * @param aCustom
   *           Custom resource record type
   */
  public EPPTtl(RRecType aRRecType, String aCustom) {
    this.setRRecType(aRRecType);
    this.setCustom(aCustom);
  }

  /**
   * Create {@code EPPTtl} instance with the TTL resource record type and the
   * TTL value.
   * 
   * @param aRRecType
   *           Required resource record type
   * @param aValue
   *           The TTL value. Set to {@code null} to unset the explicit TTL
   *           value.
   */
  public EPPTtl(RRecType aRRecType, Integer aValue) {
    this.setRRecType(aRRecType);
    this.setValue(aValue);
  }

  /**
   * Create {@code EPPTtl} instance with a custom TTL resource record type and
   * the TTL value.
   * 
   * @param aRRecType
   *           Required resource record type
   * @param aCustom
   *           Custom resource record type
   * @param aValue
   *           The TTL value. Set to {@code null} to unset the explicit TTL
   *           value.
   */
  public EPPTtl(RRecType aRRecType, String aCustom, Integer aValue) {
    this.setRRecType(aRRecType);
    this.setCustom(aCustom);
    this.setValue(aValue);
  }

  /**
   * Create {@code EPPTtl} instance with all of the attributes.
   * 
   * @param aRRecType
   *           Required resource record type
   * @param aCustom
   *           Custom resource record type. Use {@code null} if undefined.
   * @param aMinValue
   *           Min TTL value. Use {@code null} if undefined.
   * @param aMaxValue
   *           Max TTL value. Use {@code null} if undefined.
   * @param aDefaultValue
   *           Default TTL value. Use {@code null} if undefined.
   * @param aValue
   *           The ttl value. Set to {@code null} to unset the explicit TTL
   *           value.
   */
  public EPPTtl(RRecType aRRecType, String aCustom, Integer aMinValue, Integer aMaxValue, Integer aDefaultValue,
        Integer aValue) {
    this.setRRecType(aRRecType);
    this.setCustom(aCustom);
    this.setMinValue(aMinValue);
    this.setMaxValue(aMaxValue);
    this.setDefaultValue(aDefaultValue);
    this.setValue(aValue);
  }

  /**
   * Gets the resource record type.
   *
   * @return Resource record type if defined; {@code null} otherwise.
   */
  public RRecType getRRecType() {
    return this.rrecType;
  }

  /**
   * Sets resource record type.
   *
   * @param aRRecType
   *           Sets resource record type
   */
  public void setRRecType(RRecType aRRecType) {
    this.rrecType = aRRecType;
    cat.debug("setRRecType: rrecType = " + this.rrecType);
  }

  /**
   * Is the custom resource record type attribute defined?
   * 
   * @return {@code true} if the custom resource record type attribute is
   *         defined; {@code false} otherwise.
   */
  public boolean hasCustom() {
    return (this.custom != null ? true : false);
  }

  /**
   * Gets the custom resource record type attribute value.
   * 
   * @return Custom resource record type attribute if defined; {@code null}
   *         otherwise.
   */
  public String getCustom() {
    return this.custom;
  }

  /**
   * Sets the custom resource record type attribute value.
   * 
   * @param aCustom
   *           Custom resource record type value. Unset with {@code null}.
   */
  public void setCustom(String aCustom) {
    this.custom = aCustom;
    cat.debug("setCustom: custom = " + this.custom);
  }

  /**
   * Is the min TTL value defined?
   * 
   * @return {@code true} if the min TTL value is defined; {@code false}
   *         otherwise.
   */
  public boolean hasMinValue() {
    return (this.minValue != null ? true : false);
  }

  /**
   * Gets the min TTL value.
   * 
   * @return Min TTL value if defined; {@code null} otherwise.
   */
  public Integer getMinValue() {
    return this.minValue;
  }

  /**
   * Sets the TTL min value.
   * 
   * @param aMinValue
   *           Min TTL value. Unset with {@code null}.
   */
  public void setMinValue(Integer aMinValue) {
    this.minValue = aMinValue;
    cat.debug("setMinValue: minValue = " + this.minValue);
  }

  /**
   * Is the max TTL value defined?
   * 
   * @return {@code true} if the max TTL value is defined; {@code false}
   *         otherwise.
   */
  public boolean hasMaxValue() {
    return (this.maxValue != null ? true : false);
  }

  /**
   * Gets the max TTL value.
   * 
   * @return Max TTL value if defined; {@code null} otherwise.
   */
  public Integer getMaxValue() {
    return this.maxValue;
  }

  /**
   * Sets the TTL max value.
   * 
   * @param aMaxValue
   *           Max TTL value. Unset with {@code null}.
   */
  public void setMaxValue(Integer aMaxValue) {
    this.maxValue = aMaxValue;
    cat.debug("setMaxValue: maxValue = " + this.maxValue);
  }

  /**
   * Is the default TTL value defined?
   * 
   * @return {@code true} if the default TTL value is defined; {@code false}
   *         otherwise.
   */
  public boolean hasDefaultValue() {
    return (this.defaultValue != null ? true : false);
  }

  /**
   * Gets the default TTL value.
   * 
   * @return Default TTL value if defined; {@code null} otherwise.
   */
  public Integer getDefaultValue() {
    return this.defaultValue;
  }

  /**
   * Sets the TTL default value.
   * 
   * @param aDefaultValue
   *           Default TTL value. Unset with {@code null}.
   */
  public void setDefaultValue(Integer aDefaultValue) {
    this.defaultValue = aDefaultValue;
    cat.debug("setDefaultValue: defaultValue = " + this.defaultValue);
  }

  /**
   * Are all the policy TTL values ({@code minValue}, {@code maxValue},
   * {@code defaultValue}) defined?
   * 
   * @return {@code true} if all of the policy TTL values are defined;
   *         {@code false} otherwise.
   */
  public boolean hasPolicyValues() {
    return (this.hasMinValue() && this.hasMaxValue() && this.hasDefaultValue() ? true : false);
  }

  /**
   * Sets the policy TTL values ({@code minValue}, {@code maxValue},
   * {@code defaultValue}).
   * 
   * @param aMinValue
   *           Min TTL value. Use {@code null} if undefined.
   * @param aMaxValue
   *           Max TTL value. Use {@code null} if undefined.
   * @param aDefaultValue
   *           Default TTL value. Use {@code null} if undefined.
   */
  public void setPolicyValues(Integer aMinValue, Integer aMaxValue, Integer aDefaultValue) {
    this.setMinValue(aMinValue);
    this.setMaxValue(aMaxValue);
    this.setDefaultValue(aDefaultValue);
  }

  /**
   * Unsets the policy TTL values ({@code minValue}, {@code maxValue},
   * {@code defaultValue}).
   */
  public void unsetPolicyValues() {
    this.setMinValue(null);
    this.setMaxValue(null);
    this.setDefaultValue(null);
  }

  /**
   * Is the TTL value defined?
   * 
   * @return {@code true} if the TTL value is defined; {@code false} otherwise.
   */
  public boolean hasValue() {
    return (this.value != null ? true : false);
  }

  /**
   * Gets the TTL value.
   * 
   * @return TTL value if defined; {@code null} otherwise.
   */
  public Integer getValue() {
    return this.value;
  }

  /**
   * Sets the TTL value.
   * 
   * @param aValue
   *           TTL value. Unset with {@code null}.
   */
  public void setValue(Integer aValue) {
    this.value = aValue;
    cat.debug("setValue: value = " + this.value);
  }

  /**
   * Clone {@code EPPTtl} instance.
   *
   * @return clone of {@code EPPTtl}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  public Object clone() throws CloneNotSupportedException {
    EPPTtl clone = null;

    clone = (EPPTtl) super.clone();

    return clone;
  }

  /**
   * Decode the {@code EPPTtl} element aElement DOM Element tree.
   *
   * @param aElement
   *           - Root DOM Element to decode {@code EPPTtl} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  public void decode(Element aElement) throws EPPDecodeException {
    // RRecType
    String theRRecType = EPPUtil.decodeStringAttr(aElement, ATTR_FOR);
    if (theRRecType != null) {
      this.setRRecType(rrecType = RRecType.getRRecType(theRRecType));
    }
    else {
      this.setRRecType(null);
    }

    // Custom
    this.setCustom(EPPUtil.decodeStringAttr(aElement, ATTR_CUSTOM));

    String theStrAttr;

    // Min Value
    theStrAttr = EPPUtil.decodeStringAttr(aElement, ATTR_MIN);
    if (theStrAttr != null) {
      this.setMinValue(Integer.decode(theStrAttr));
    }
    else {
      this.setMinValue(null);
    }

    // Max Value
    theStrAttr = EPPUtil.decodeStringAttr(aElement, ATTR_MAX);
    if (theStrAttr != null) {
      this.setMaxValue(Integer.decode(theStrAttr));
    }
    else {
      this.setMaxValue(null);
    }

    // Default Value
    theStrAttr = EPPUtil.decodeStringAttr(aElement, ATTR_DEFAULT);
    if (theStrAttr != null) {
      this.setDefaultValue(Integer.decode(theStrAttr));
    }
    else {
      this.setDefaultValue(null);
    }

    // Value
    String theValueStr = EPPUtil.getTextContent(aElement, true);
    if (!theValueStr.isEmpty()) {
      this.setValue(Integer.decode(theValueStr));
    }
    else {
      this.setValue(null);
    }
  }

  /**
   * Encode a DOM Element tree from the attributes of the {@code EPPTtl}
   * instance.
   *
   * @param aDocument
   *           - DOM Document that is being built. Used as an Element factory.
   *
   * @return Element - Root DOM Element representing the {@code EPPTtl}
   *         instance.
   *
   * @exception EPPEncodeException
   *               - Unable to encode {@code EPPTtl} instance.
   */
  public Element encode(Document aDocument) throws EPPEncodeException {

    if (aDocument == null) {
      throw new EPPEncodeException("aDocument is null in EPPTtl.encode(Document)");
    }
    if (this.rrecType == null) {
      throw new EPPEncodeException("rrecType is null in EPPTtl.encode(Document)");
    }

    Element root = aDocument.createElementNS(EPPTtlExtFactory.NS, ELM_NAME);

    // RRecType
    root.setAttribute(ATTR_FOR, this.rrecType.toString());

    // Custom
    if (this.hasCustom()) {
      root.setAttribute(ATTR_CUSTOM, this.custom);
    }

    // Min Value
    if (this.hasMinValue()) {
      root.setAttribute(ATTR_MIN, this.minValue.toString());
    }

    // Max Value
    if (this.hasMaxValue()) {
      root.setAttribute(ATTR_MAX, this.maxValue.toString());
    }

    // Default Value
    if (this.hasDefaultValue()) {
      root.setAttribute(ATTR_DEFAULT, this.defaultValue.toString());
    }

    // Value
    if (this.hasValue()) {
      Text theValue = aDocument.createTextNode(this.value.toString());
      root.appendChild(theValue);
    }

    return root;
  }

  /**
   * implements a deep {@code EPPTtl} compare.
   *
   * @param aObject
   *           {@code EPPTtl} instance to compare with
   *
   * @return {@code true} if equal; {@code false} otherwise
   */
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPTtl)) {
      return false;
    }

    EPPTtl other = (EPPTtl) aObject;

    // RRecType
    if (!EqualityUtil.equals(this.rrecType, other.rrecType)) {
      cat.error("EPPTtl.equals(): rrecType not equal");
      return false;
    }

    // Custom
    if (!EqualityUtil.equals(this.custom, other.custom)) {
      cat.error("EPPTtl.equals(): custom not equal");
      return false;
    }

    // Min Value
    if (!EqualityUtil.equals(this.minValue, other.minValue)) {
      cat.error("EPPTtl.equals(): minValue not equal");
      return false;
    }

    // Max Value
    if (!EqualityUtil.equals(this.maxValue, other.maxValue)) {
      cat.error("EPPTtl.equals(): maxValue not equal");
      return false;
    }

    // Default Value
    if (!EqualityUtil.equals(this.defaultValue, other.defaultValue)) {
      cat.error("EPPTtl.equals(): defaultValue not equal");
      return false;
    }

    // Value
    if (!EqualityUtil.equals(this.value, other.value)) {
      cat.error("EPPTtl.equals(): value not equal");
      return false;
    }

    return true;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   * 
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPTtlExtFactory.NS;
  }

}
