/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Defines the Key Data Interface, as defined in RFC 5910, policies. The
 * &lt;registry:keyDataInterface&gt; element contains the following child
 * elements: <br>
 * <br>
 * <ul>
 * <li>&lt;registry:min&gt; - the minimum number of keys associated with the
 * domain object. Use {@link #getMin()} and {@link #setMin(Integer)} to get and
 * set the element.</li>
 * <li>&lt;registry:max&gt; - the maximum number of keys associated with the
 * domain object. Use {@link #getMax()} and {@link #setMax(Integer)} to get and
 * set the element.</li>
 * <li>&lt;registry:flags&gt; - Zero or more &lt;registry:flags&gt; elements
 * that define the supported flags field values, as described in section 2.1.1
 * of RFC 4034. Use {@link #getFlags()} and {@link #setFlags(List)} to get and
 * set the element. Use {@link #addFlags(Integer)} to add an flags field
 * {@code Integer} to an existing {@code List}.</li>
 * <li>&lt;registry:protocol&gt; - Zero or more &lt;registry:protocol&gt;
 * elements that define the supported protocols as described in section 2.1.2 of
 * RFC 4034. Use {@link #getProtocols()} and {@link #setProtocols(List)} to get
 * and set the element. Use {@link #addProtocol(Integer)} to add an algorithm
 * {@code Integer} to an existing {@code List}.</li>
 * <li>&lt;registry:alg&gt; - Zero or more &lt;registry:alg&gt; elements that
 * define the supported algorithms as described in section 2.1.3 of RFC 4034.
 * Use {@link #getAlgorithms()} and {@link #setAlgorithms(List)} to get and set
 * the element. Use {@link #addAlgorithm(Integer)} to add an algorithm
 * {@code Integer} to an existing {@code List}.</li>
 * </ul>
 *
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryDNSSEC
 */
public class EPPRegistryKey extends EPPRegistryAbstractMinMax {

  /**
   * Logger
   */
  private static Logger cat = LoggerFactory.getLogger(EPPRegistryKey.class);

  /**
   * XML local name for {@code EPPRegistryKey}.
   */
  public static final String ELM_LOCALNAME = "keyDataInterface";

  /**
   * XML root tag for {@code EPPRegistryKey}.
   */
  public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * XML Element Name of {@code flags} attribute.
   */
  public static final String ELM_FLAGS = "flags";

  /**
   * XML Element Name of {@code protocol} attribute.
   */
  public static final String ELM_PROTOCOL = "protocol";

  /**
   * XML Element Name of {@code algorithms} attribute.
   */
  public static final String ELM_ALGORITHM = "alg";

  /**
   * {@code List} of zero or more &lt;registry:flags&gt; elements that define
   * the supported flags field values, as described in section 2.1.1 of
   * [RFC4034].
   */
  private List<Integer> flags = new ArrayList<Integer>();

  /**
   * {@code List} of zero or more &lt;registry:protocol&gt; elements that
   * define the supported protocols, as described in section 2.1.2 of
   * [RFC4034].
   */
  private List<Integer> protocols = new ArrayList<Integer>();

  /**
   * {@code List} of algorithms in {@code String}
   */
  private List<Integer> algorithms = new ArrayList<Integer>();

  /**
   * Default constructor. Must call {@link #setMin(Integer)} and
   * {@link #setMax(Integer)} before calling {@link #encode(Document)} method.
   */
  public EPPRegistryKey() {
  }

  /**
   * Constructs an instance with {@code min} and {@code max}.
   *
   * @param aMin
   *           minimum number of keys associated with the domain object
   * @param aMax
   *           maximum number of keys associated with the domain object
   */
  public EPPRegistryKey(Integer aMin, Integer aMax) {
    super(aMin, aMax);
  }

  /**
   * Constructs an instance with {@code min}, {@code max} and {@code List} of
   * {@code algorithms}.
   *
   * @param aMin
   *           minimum number of keys associated with the domain object
   * @param aMax
   *           maximum number of keys associated with the domain object
   * @param aFlags
   *           supported flags as described in section 2.1.1 of RFC 4034
   * @param aProtocols
   *           supported protocols as described in section 2.1.2 of RFC 4034
   * @param aAlgorithms
   *           supported algorithms as described in section 2.1.3 of RFC 4034
   */
  public EPPRegistryKey(Integer aMin, Integer aMax, List<Integer> aFlags, List<Integer> aProtocols,
        List<Integer> aAlgorithms) {
    super(aMin, aMax);
    this.setFlags(aFlags);
    this.setProtocols(aProtocols);
    this.setAlgorithms(aAlgorithms);
  }

  /**
   * Constructs an instance with {@code min}, {@code max} and {@code List} of
   * {@code algorithms}.
   *
   * @param aMin
   *           minimum number of keys associated with the domain object
   * @param aMax
   *           maximum number of keys associated with the domain object
   * @param aFlags
   *           supported flags as described in section 2.1.1 of RFC 4034
   * @param aProtocols
   *           supported protocols as described in section 2.1.2 of RFC 4034
   * @param aAlgorithms
   *           supported algorithms as described in section 2.1.3 of RFC 4034
   */
  public EPPRegistryKey(int aMin, int aMax, List<Integer> aFlags, List<Integer> aProtocols,
        List<Integer> aAlgorithms) {
    super(aMin, aMax);
    this.setFlags(aFlags);
    this.setProtocols(aProtocols);
    this.setAlgorithms(aAlgorithms);
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPRegistryKey} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the {@code EPPRegistryKey}
   *         instance.
   *
   * @exception EPPEncodeException
   *               - Unable to encode {@code EPPRegistryKey} instance.
   */
  @Override
  public Element encode(Document aDocument) throws EPPEncodeException {
    Element root = super.encode(aDocument);

    // Flags
    EPPUtil.encodeIntegerList(aDocument, root, this.flags, EPPRegistryMapFactory.NS,
          EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_FLAGS);

    // Protocols
    EPPUtil.encodeIntegerList(aDocument, root, this.protocols, EPPRegistryMapFactory.NS,
          EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_PROTOCOL);

    // Algorithms
    EPPUtil.encodeIntegerList(aDocument, root, this.algorithms, EPPRegistryMapFactory.NS,
          EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_ALGORITHM);

    return root;
  }

  /**
   * Decode the {@code EPPRegistryKey} attributes from the aElement DOM Element
   * tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPRegistryKey} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  public void decode(Element aElement) throws EPPDecodeException {
    super.decode(aElement);

    // Flags
    this.flags = EPPUtil.decodeIntegerList(aElement, EPPRegistryMapFactory.NS, ELM_FLAGS);

    // Protocols
    this.protocols = EPPUtil.decodeIntegerList(aElement, EPPRegistryMapFactory.NS, ELM_PROTOCOL);

    // Algorithms
    this.algorithms = EPPUtil.decodeIntegerList(aElement, EPPRegistryMapFactory.NS, ELM_ALGORITHM);
  }

  /**
   * Clone {@code EPPRegistryKey}.
   *
   * @return clone of {@code EPPRegistryKey}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPRegistryKey clone = (EPPRegistryKey) super.clone();

    // Flags
    if (this.flags != null) {
      clone.flags = (List) ((ArrayList) this.flags).clone();
    }

    // Protocols
    if (this.protocols != null) {
      clone.protocols = (List) ((ArrayList) this.protocols).clone();
    }

    // Algorithms
    if (this.algorithms != null) {
      clone.algorithms = (List) ((ArrayList) this.algorithms).clone();
    }
    return clone;
  }

  /**
   * implements a deep {@code EPPRegistryKey} compare.
   *
   * @param aObject
   *           {@code EPPRegistryKey} instance to compare with
   *
   * @return {@code true} if this object is the same as the aObject argument;
   *         {@code false} otherwise
   */
  @Override
  public boolean equals(Object aObject) {
    if (!super.equals(aObject)) {
      return false;
    }

    if (!(aObject instanceof EPPRegistryKey)) {
      return false;
    }

    EPPRegistryKey theComp = (EPPRegistryKey) aObject;

    // Flags
    if (!((this.flags == null) ? (theComp.flags == null) : EPPUtil.equalLists(this.flags, theComp.flags))) {
      cat.error("EPPRegistryKey.equals(): flags not equal");
      return false;
    }

    // Protocols
    if (!((this.protocols == null) ? (theComp.protocols == null) :
          EPPUtil.equalLists(this.protocols, theComp.protocols))) {
      cat.error("EPPRegistryKey.equals(): protocols not equal");
      return false;
    }

    // Algorithms
    if (!((this.algorithms == null) ? (theComp.algorithms == null) :
          EPPUtil.equalLists(this.algorithms, theComp.algorithms))) {
      cat.error("EPPRegistryKey.equals(): algorithms not equal");
      return false;
    }

    return true;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Is the flags defined?
   *
   * @return {@code true} if the flags is defined; {@code false} otherwise.
   */
  public boolean hasFlags() {
    return (this.flags != null && !this.flags.isEmpty() ? true : false);
  }

  /**
   * Gets the supported flags.
   *
   * @return {@code List} of supported flags as described in section 2.1.1 of
   *         RFC 4034
   */
  public List<Integer> getFlags() {
    return this.flags;
  }

  /**
   * Sets the supported flags.
   *
   * @param aFlags
   *           {@code List} of supported flags as described in section 2.1.1 of
   *           RFC 4034
   */
  public void setFlags(List<Integer> aFlags) {
    if (aFlags == null) {
      this.flags = new ArrayList<Integer>();
    }
    else {
      this.flags = aFlags;
    }
  }

  /**
   * Adds one flags field to existing supported flags.
   *
   * @param aFlags
   *           supported flags field as described in section 2.1.1 of RFC 4034
   */
  public void addFlags(Integer aFlags) {
    if (this.flags == null) {
      this.flags = new ArrayList<Integer>();
    }
    this.flags.add(aFlags);
  }

  /**
   * Is the protocols defined?
   *
   * @return {@code true} if the protocols is defined; {@code false} otherwise.
   */
  public boolean hasProtocols() {
    return (this.protocols != null && !this.protocols.isEmpty() ? true : false);
  }

  /**
   * Gets the supported protocols.
   *
   * @return {@code List} of supported protocols as described in section 2.1.2
   *         of RFC 4034
   */
  public List<Integer> getProtocols() {
    return this.protocols;
  }

  /**
   * Sets the supported protocols.
   *
   * @param aProtocol
   *           {@code List} of supported protocols as described in section
   *           2.1.2 of RFC 4034
   */
  public void setProtocols(List<Integer> aProtocol) {
    if (aProtocol == null) {
      this.protocols = new ArrayList<Integer>();
    }
    else {
      this.protocols = aProtocol;
    }
  }

  /**
   * Adds one protocol to existing the supported protocols.
   *
   * @param aProtocol
   *           supported protocol as described in section 2.1.2 of RFC 4034
   */
  public void addProtocol(Integer aProtocol) {
    if (this.protocols == null) {
      this.protocols = new ArrayList<Integer>();
    }
    this.protocols.add(aProtocol);
  }

  /**
   * Is the algorithms defined?
   *
   * @return {@code true} if the algorithms is defined; {@code false}
   *         otherwise.
   */
  public boolean hasAlgorithms() {
    return (this.algorithms != null && !this.algorithms.isEmpty() ? true : false);
  }

  /**
   * Gets the supported algorithms.
   *
   * @return {@code List} of supported algorithms as described in section 2.1.3
   *         of RFC 4034
   */
  public List<Integer> getAlgorithms() {
    return this.algorithms;
  }

  /**
   * Sets the supported algorithms.
   *
   * @param aAlgorithms
   *           {@code List} of supported algorithms as described in section
   *           2.1.3 of RFC 4034
   */
  public void setAlgorithms(List<Integer> aAlgorithms) {
    if (aAlgorithms == null) {
      this.algorithms = new ArrayList<Integer>();
    }
    else {
      this.algorithms = aAlgorithms;
    }
  }

  /**
   * Adds one algorithm to existing supported algorithms.
   *
   * @param aAlgorithm
   *           supported algorithm as described in section 2.1.3 of RFC 4034
   */
  public void addAlgorithm(Integer aAlgorithm) {
    if (this.algorithms == null) {
      this.algorithms = new ArrayList<Integer>();
    }
    this.algorithms.add(aAlgorithm);
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   *
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPRegistryMapFactory.NS;
  }

  /**
   * Gets the root element local name.
   *
   * @return Root element local name.
   */
  @Override
  protected String getRootName() {
    return ELM_LOCALNAME;
  }

  /**
   * Gets the minimum element local name.
   *
   * @return Minimum element local name.
   */
  @Override
  protected String getElmMin() {
    return ELM_MIN;
  }

  /**
   * Gets the maximum element local name.
   *
   * @return Maximum element local name.
   */
  @Override
  protected String getElmMax() {
    return ELM_MAX;
  }

  /**
   * Gets the {@link Logger} to use.
   *
   * @return {@link Logger} instance to use for logging.
   */
  @Override
  protected Logger getLogger() {
    return cat;
  }

}
