/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.util.EqualityUtil;

/**
 * Defines the grace periods by operation type. The required "command" attribute
 * defines the operation type with the sample values of "create", "renew",
 * "transfer", and "autoRenew". The &lt;registry:gracePeriod&gt; element
 * requires the "unit" attribute with the possible values of "d" for day, "h"
 * for hour, and "m" for minute.
 *
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryDomain
 */
public class EPPRegistryGracePeriod extends EPPRegistryPeriodType {

  /**
   * Logger
   */
      private static Logger cat = LoggerFactory.getLogger(EPPRegistryGracePeriod.class);
        

  /**
   * XML local name for {@code EPPRegistryGracePeriod}.
   */
  public static final String ELM_LOCALNAME = "gracePeriod";

  /**
   * XML attribute name for the {@code command} attribute.
   */
  public static final String ATTR_COMMAND = "command";

  /**
   * EPP transform command to which this period applies
   */
  protected String command = null;

  /**
   * Default constructor. Must call {@link #setNumber(Integer)},
   * {@link #setUnit(EPPRegistryPeriodType.Unit)} and {@link #setCommand(String)} before calling
   * {@link #encode(Document)} method.
   */
  public EPPRegistryGracePeriod() {
  }

  /**
   * Constructs an instance of {@code EPPRegistryGracePeriod} with
   * {@code command}, {@code number} and {@code unit}.
   *
   * @param aCommand
   *           EPP command to which this period applies
   * @param aNumber
   *           number must be &gt; 0
   * @param aUnit
   *           unit must be one of {@code Unit.d}, {@code Unit.h}, or
   *           {@code Unit.m}.
   */
  public EPPRegistryGracePeriod(String aCommand, Integer aNumber, Unit aUnit) {
    this.command = aCommand;
    super.number = aNumber;
    super.unit = aUnit;
  }

  /**
   * Constructs an instance of {@code EPPRegistryGracePeriod} with
   * {@code command}, {@code number} and {@code unit}.
   *
   * @param command
   *           EPP command to which this period applies
   * @param number
   *           number must be &gt; 0
   * @param unit
   *           unit must be one of {@code Unit.d}, {@code Unit.h}, or
   *           {@code Unit.m}.
   */
  public EPPRegistryGracePeriod(String command, int number, Unit unit) {
    this(command, Integer.valueOf(number), unit);
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPRegistryGracePeriod} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         {@code EPPRegistryGracePeriod} instance.
   *
   * @exception EPPEncodeException
   *               - Unable to encode {@code EPPRegistryGracePeriod} instance.
   */
  @Override
  public Element encode(Document aDocument) throws EPPEncodeException {
    Element root = super.encode(aDocument);
    root.setAttribute(ATTR_COMMAND, this.command);

    return root;
  }

  /**
   * Decode the {@code EPPRegistryGracePeriod} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPRegistryGracePeriod} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  public void decode(Element aElement) throws EPPDecodeException {
    super.decode(aElement);
    this.command = aElement.getAttribute(ATTR_COMMAND);
  }

  /**
   * Extra validation of the default period number and unit.
   *
   * @throws EPPCodecException
   *            Validation error
   */
  @Override
  void validateState() throws EPPCodecException {
    super.validateState();

    // Command
    if (this.command == null || this.command.trim().length() == 0) {
      throw new EPPCodecException(getRootName() + ": command is empty");
    }
  }

  /**
   * implements a deep {@code EPPRegistryGracePeriod} compare.
   *
   * @param aObject
   *           {@code EPPRegistryGracePeriod} instance to compare with
   *
   * @return {@code true} if this object is the same as the aObject argument;
   *         {@code false} otherwise
   */
  @Override
  public boolean equals(Object aObject) {

    if (!(aObject instanceof EPPRegistryGracePeriod)) {
      return false;
    }

    EPPRegistryGracePeriod theComp = (EPPRegistryGracePeriod) aObject;

    // Number and Unit
    if (!super.equals(aObject)) {
      return false;
    }

    // Command
    if (!EqualityUtil.equals(this.command, theComp.command)) {
      cat.error("EPPRegistryGracePeriod.equals(): command not equal");
      return false;
    }

    return true;
  }

  /**
   * Get the EPP transform command to which this period applies.
   *
   * @return EPP transform command to which this period applies
   */
  public String getCommand() {
    return this.command;
  }

  /**
   * Set the EPP transform command to which this period applies.
   *
   * @param command
   *           EPP transform command to which this period applies
   */
  public void setCommand(String command) {
    this.command = command;
  }

  /**
   * Extra validation rules on top of {@link EPPRegistryPeriodType}:
   * {@code number} must be greater than "0"; {@code unit} must be one of
   * {@code Unit.d}, {@code Unit.h}, or {@code Unit.m}.
   */
  @Override
  void extraValidate() throws EPPCodecException {
    int n = this.number.intValue();
    if (n <= 0) {
      throw new EPPCodecException(getRootName() + ": number should be greater than 0");
    }
    if (super.unit != Unit.d && super.unit != Unit.h && super.unit != Unit.m) {
      throw new EPPCodecException(getRootName() + ": invalid unit. Valid values: d/h/m");
    }
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   *
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPRegistryMapFactory.NS;
  }

  /**
   * Gets the root element local name.
   * 
   * @return Root element local name.
   */
  protected String getRootName() {
    return ELM_LOCALNAME;
  }
  
}
