/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Defines the DS Data Interface, as defined in RFC 5910, policies. The
 * &lt;registry:dsDataInterface&gt; element contains the following child
 * elements: <br>
 * <br>
 *
 * <ul>
 * <li>&lt;registry:min&gt; - the minimum number of DS associated with the
 * domain object. Use {@link #getMin()} and {@link #setMin(Integer)} to get and
 * set the element.</li>
 * <li>&lt;registry:max&gt; - the maximum number of DS associated with the
 * domain object. Use {@link #getMax()} and {@link #setMax(Integer)} to get and
 * set the element.</li>
 * <li>&lt;registry:alg&gt; - zero or more &lt;registry:alg&gt; elements that
 * define the supported algorithms as described in section 5.1.2 of RFC 4034.
 * Use {@link #getAlgorithms()} and {@link #setAlgorithms(List)} to get and set
 * the element.</li>
 * <li>&lt;registry:digestType&gt; - zero or more &lt;registry:digestType&gt;
 * elements that define the supported digest types as described in section 5.1.3
 * of RFC 4034. Use {@link #getDigestTypes()} and {@link #setDigestTypes(List)}
 * to get and set the element.</li>
 * </ul>
 *
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryDNSSEC
 */
public class EPPRegistryDS extends EPPRegistryAbstractMinMax {

  /**
   * Logger
   */
  private static Logger cat = LoggerFactory.getLogger(EPPRegistryDS.class);

  /**
   * XML local name for {@code EPPRegistryDS}.
   */
  public static final String ELM_LOCALNAME = "dsDataInterface";

  /**
   * XML root tag for {@code EPPRegistryDS}.
   */
  public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * XML Element Name of {@code algorithms} attribute.
   */
  public static final String ELM_ALGORITHM = "alg";

  /** XML tag name for the {@code digestTypeF} attribute. */
  public static final String ELM_DIGEST = "digestType";

  /**
   * {@code List} of algorithms in {@code String}
   */
  private List<Integer> algorithms = new ArrayList<Integer>();

  /** {@code List} of digest type in {@code String} */
  private List<Integer> digestTypes = new ArrayList<Integer>();

  /**
   * Constructs a new {@code EPPRegistryDS} object. All attributes are set to
   * default. Must call {@link #setMin(Integer)} and {@link #setMax(Integer)}
   * before calling {@link #encode(Document)} method.
   */
  public EPPRegistryDS() {
  }

  /**
   * Constructs a new {@code EPPRegistryDS} with a min and a max value.
   *
   * @param aMin
   *           minimum number of DS associated with the domain object
   * @param aMax
   *           maximum number of DS associated with the domain object
   */
  public EPPRegistryDS(Integer aMin, Integer aMax) {
    super(aMin, aMax);
  }

  /**
   * Constructs a new {@code EPPRegistryDS} with given values.
   *
   * @param aMin
   *           minimum number of DS associated with the domain object
   * @param aMax
   *           maximum number of DS associated with the domain object
   * @param aAlgorithms
   *           {@code List} of algorithm {@code String}
   * @param aDigestTypes
   *           {@code List} of digest type {@code String}
   */
  public EPPRegistryDS(Integer aMin, Integer aMax, List<Integer> aAlgorithms, List<Integer> aDigestTypes) {
    super(aMin, aMax);
    this.setAlgorithms(aAlgorithms);
    this.setDigestTypes(aDigestTypes);
  }

  /**
   * Constructs a new {@code EPPRegistryDS} with given values.
   *
   * @param aMin
   *           minimum number of DS associated with the domain object
   * @param aMax
   *           maximum number of DS associated with the domain object
   * @param aAlgorithms
   *           {@code List} of algorithm {@code String}
   * @param aDigestTypes
   *           {@code List} of digest type {@code String}
   */
  public EPPRegistryDS(int aMin, int aMax, List<Integer> aAlgorithms, List<Integer> aDigestTypes) {
    this(Integer.valueOf(aMin), Integer.valueOf(aMax), aAlgorithms, aDigestTypes);
  }

  /**
   * Encode a DOM Element tree from the attributes of the {@code EPPRegistryDS}
   * instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the {@code EPPRegistryDS}
   *         instance.
   *
   * @exception EPPEncodeException
   *               - Unable to encode {@code EPPRegistryDS} instance.
   */
  @Override
  public Element encode(Document aDocument) throws EPPEncodeException {
    Element root = super.encode(aDocument);

    // Algorithms
    EPPUtil.encodeIntegerList(aDocument, root, this.algorithms, EPPRegistryMapFactory.NS,
          EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_ALGORITHM);

    // Digests
    EPPUtil.encodeIntegerList(aDocument, root, this.digestTypes, EPPRegistryMapFactory.NS,
          EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_DIGEST);

    return root;
  }

  /**
   * Decode the {@code EPPRegistryDS} attributes from the aElement DOM Element
   * tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPRegistryDS} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  public void decode(Element aElement) throws EPPDecodeException {
    super.decode(aElement);

    // Algorithms
    this.algorithms = EPPUtil.decodeIntegerList(aElement, EPPRegistryMapFactory.NS, ELM_ALGORITHM);

    // Digests
    this.digestTypes = EPPUtil.decodeIntegerList(aElement, EPPRegistryMapFactory.NS, ELM_DIGEST);
  }

  /**
   * Clone {@code EPPRegistryDS}.
   *
   * @return clone of {@code EPPRegistryDS}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPRegistryDS clone = (EPPRegistryDS) super.clone();

    // Algorithms
    if (this.algorithms != null) {
      clone.algorithms = (List) ((ArrayList) this.algorithms).clone();
    }

    // Digests
    if (this.digestTypes != null) {
      clone.digestTypes = (List) ((ArrayList) this.digestTypes).clone();
    }

    return clone;
  }

  /**
   * implements a deep {@code EPPRegistryDS} compare.
   *
   * @param aObject
   *           {@code EPPRegistryDS} instance to compare with
   *
   * @return {@code true} if this object is the same as the aObject argument;
   *         {@code false} otherwise
   */
  @Override
  public boolean equals(Object aObject) {
    if (!super.equals(aObject)) {
      return false;
    }

    if (!(aObject instanceof EPPRegistryDS)) {
      return false;
    }

    EPPRegistryDS theComp = (EPPRegistryDS) aObject;

    // Algorithms
    if (!((this.algorithms == null) ? (theComp.algorithms == null) :
          EPPUtil.equalLists(this.algorithms, theComp.algorithms))) {
      cat.error("EPPRegistryDS.equals(): algorithms not equal");
      return false;
    }

    // Digests
    if (!((this.digestTypes == null) ? (theComp.digestTypes == null) :
          EPPUtil.equalLists(this.digestTypes, theComp.digestTypes))) {
      cat.error("EPPRegistryDS.equals(): digestTypes not equal");
      return false;
    }

    return true;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Is the algorithms defined?
   *
   * @return {@code true} if the algorithms is defined; {@code false}
   *         otherwise.
   */
  public boolean hasAlgorithms() {
    return (this.algorithms != null && !this.algorithms.isEmpty() ? true : false);
  }

  /**
   * Gets the supported algorithms.
   *
   * @return {@code List} of supported algorithms as described in section 5.1.2
   *         of RFC 4034
   */
  public List<Integer> getAlgorithms() {
    return this.algorithms;
  }

  /**
   * Sets the supported algorithms.
   *
   * @param aAlgorithms
   *           {@code List} of supported algorithms as described in section
   *           5.1.2 of RFC 4034
   */
  public void setAlgorithms(List<Integer> aAlgorithms) {
    if (aAlgorithms == null) {
      this.algorithms = new ArrayList<Integer>();
    }
    else {
      this.algorithms = aAlgorithms;
    }
  }

  /**
   * Adds one algorithm to existing supported algorithms.
   *
   * @param aAlgorithm
   *           supported algorithm as described in section 5.1.2 of RFC 4034
   */
  public void addAlgorithm(Integer aAlgorithm) {
    if (this.algorithms == null) {
      this.algorithms = new ArrayList<Integer>();
    }
    this.algorithms.add(aAlgorithm);
  }

  /**
   * Get digest types.
   *
   * @return {@code List} of digest type in {@code String}
   */
  public List<Integer> getDigestTypes() {
    return this.digestTypes;
  }

  /**
   * Set digest types.
   *
   * @param aDigestTypes
   *           {@code List} of digest type in {@code String}
   */
  public void setDigestTypes(List<Integer> aDigestTypes) {
    if (aDigestTypes == null) {
      this.digestTypes = new ArrayList<Integer>();
    }
    else {
      this.digestTypes = aDigestTypes;
    }
  }

  /**
   * Add one digest type to an existing list.
   *
   * @param digestType
   *           digest type in {@code String}
   */
  public void addDigestType(Integer digestType) {
    if (this.digestTypes == null) {
      this.digestTypes = new ArrayList<Integer>();
    }

    this.digestTypes.add(digestType);
  }

  /**
   * Gets the root element local name.
   *
   * @return Root element local name.
   */
  @Override
  protected String getRootName() {
    return ELM_LOCALNAME;
  }

  /**
   * Gets the minimum element local name.
   *
   * @return Minimum element local name.
   */
  @Override
  protected String getElmMin() {
    return ELM_MIN;
  }

  /**
   * Gets the maximum element local name.
   *
   * @return Maximum element local name.
   */
  @Override
  protected String getElmMax() {
    return ELM_MAX;
  }

  /**
   * Gets the {@link Logger} to use.
   *
   * @return {@link Logger} instance to use for logging.
   */
  @Override
  protected Logger getLogger() {
    return cat;
  }

}
