/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import java.security.InvalidParameterException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPRegistryBatchSchedule} defines a batch schedule that uses the
 * &lt;registry:schedule&gt; element, with the required "frequency" attribute
 * that defines the frequency of execution. The "frequency" attribute has the
 * possible values of "daily", "weekly", and "monthy". The time zone is defined
 * using the XML schema "time" type conventions of UTC and offsets from UTC, or
 * using the OPTIONAL "tz" attribute that defines the named time zone. For
 * example, the named Eastern time zone can be specified using the setting
 * "tz=EST5EDT".
 *
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryDomain
 */
public class EPPRegistryBatchSchedule implements EPPCodecComponent {

  /**
   * Logger
   */
      private static Logger cat = LoggerFactory.getLogger(EPPRegistryBatchSchedule.class);
        

  /**
   * Possible values for the {@code frequency} attribute.
   */
  public static enum Frequency {

    /**
     * The schedule will execute every day.
     */
    daily,

    /**
     * The schedule will execute every week.
     */
    weekly,

    /**
     * The schedule will execute every month.
     */
    monthly;
  }

  /**
   * Possible values for the {@code dayOfWeek} attribute.
   */
  public static enum DayOfWeek {

    SUNDAY("0"), MONDAY("1"), TUESDAY("2"), WEDNESDAY("3"), THURSDAY("4"), FRIDAY("5"), SATURDAY("6");

    private final String dayOfWeekStr;

    /**
     * Define the string value for the enumerated value.
     *
     * @param aDayOfWeekStr
     *           Enumerated value string
     */
    DayOfWeek(String aDayOfWeekStr) {
      this.dayOfWeekStr = aDayOfWeekStr;
    }

    /**
     * Get the type enumerated value given the matching string.
     *
     * @param aDayOfWeekStr
     *           {@code PostalInfoTypeSupport} enumerated string to convert to
     *           an enumerated {@code PostalInfoTypeSupport} instance.
     *
     * @return Enumerated {@code PostalInfoTypeSupport} value matching the
     *         {@code String}.
     *
     * @throws InvalidParameterException
     *            If {@code aDayOfWeekStr} does not match an enumerated
     *            {@code PostalInfoTypeSupport} string value.
     */
    public static DayOfWeek getDayOfWeek(String aDayOfWeekStr) {
      if (aDayOfWeekStr == null) {
        throw new InvalidParameterException("null DayOfWeek enum value is not valid.");
      }

      if (aDayOfWeekStr.equals(SUNDAY.dayOfWeekStr)) {
        return SUNDAY;
      }
      else if (aDayOfWeekStr.equals(MONDAY.dayOfWeekStr)) {
        return MONDAY;
      }
      else if (aDayOfWeekStr.equals(TUESDAY.dayOfWeekStr)) {
        return TUESDAY;
      }
      else if (aDayOfWeekStr.equals(WEDNESDAY.dayOfWeekStr)) {
        return WEDNESDAY;
      }
      else if (aDayOfWeekStr.equals(THURSDAY.dayOfWeekStr)) {
        return THURSDAY;
      }
      else if (aDayOfWeekStr.equals(FRIDAY.dayOfWeekStr)) {
        return FRIDAY;
      }
      else if (aDayOfWeekStr.equals(SATURDAY.dayOfWeekStr)) {
        return SATURDAY;
      }
      else {
        throw new InvalidParameterException("DayOfWeek enum value of " + aDayOfWeekStr + " is not valid.");
      }

    }

    /**
     * Convert the enumerated {@code DayOfWeek} value to a {@code String} .
     */
    @Override
    public String toString() {
      return this.dayOfWeekStr;
    }

  }

  /**
   * XML local name for {@code EPPRegistryBatchSchedule}.
   */
  public static final String ELM_LOCALNAME = "schedule";

  /**
   * XML root tag for {@code EPPRegistryBatchSchedule}.
   */
  public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * XML attribute name for the {@code frequency} attribute.
   */
  public static final String ATTR_FREQUENCY = "frequency";

  /**
   * XML attribute name for the {@code tz} attribute.
   */
  public static final String ATTR_TZ = "tz";

  /**
   * XML attribute name for the {@code dayOfWeek} attribute.
   */
  public static final String ATTR_DAY_OF_WEEK = "dayOfWeek";

  /**
   * XML attribute name for the {@code dayOfMonth} attribute.
   */
  public static final String ATTR_DAY_OF_MONTH = "dayOfMonth";

  /**
   * The frequency of the schedule
   */
  Frequency frequency = null;

  /**
   * The time of the schedule, which follows the XML schema "time" type format.
   * Examples include "14:00:00" for a time without a time zone offset,
   * "07:00:00-05:00" for a time with a 5 hour offset from UTC or EST time
   * zone, and "17:00:00Z" for 5 PM UTC.
   */
  String time = null;

  /**
   * Optional time zone of the {@code time} attribute using a named time zone,
   * such as "EST5EDT". The XML schema "time" type does not support named time
   * zones, so this attribute is used when a named time zone needs to be
   * specified.
   */
  String timeZone = null;

  /**
   * Day of week when the frequency is set to {@link Frequency#weekly}.
   */
  DayOfWeek dayOfWeek = null;

  /**
   * Day of month when the frequency is set to {@link Frequency#monthly}.
   */
  Integer dayOfMonth = null;

  /**
   * Default constructor. Must call {@link #setFrequency(Frequency)} and
   * {@link #setTime(String)} before calling the
   * {@link #encode(org.w3c.dom.Document)} method.
   */
  public EPPRegistryBatchSchedule() {
  }

  /**
   * Construct an instance of {@code EPPRegistryBatchSchedule} with the
   * required frequency and time.
   *
   * @param aFrequency
   *           The frequency of the schedule
   * @param aTime
   *           The time of the schedule following the XML schema "time" type
   *           format
   */
  public EPPRegistryBatchSchedule(Frequency aFrequency, String aTime) {
    this.frequency = aFrequency;
    this.time = aTime;
  }

  /**
   * Construct an instance of {@code EPPRegistryBatchSchedule} that is used for
   * a daily schedule.
   *
   * @param aTime
   *           The time of the schedule following the XML schema "time" type
   *           format
   * @param aTimeZone
   *           The named time zone of the time, represented by the
   *           {@code aTime} parameter. An example of a named time zone is
   *           "EST5EDT". Set to {@code null} if undefined.
   */
  public EPPRegistryBatchSchedule(String aTime, String aTimeZone) {
    this(Frequency.daily, aTime);
    this.timeZone = aTimeZone;
  }

  /**
   * Construct an instance of {@code EPPRegistryBatchSchedule} that is used for
   * a day of week schedule.
   *
   * @param aTime
   *           The time of the schedule following the XML schema "time" type
   *           format
   * @param aDayOfWeek A day of the week 
   * @param aTimeZone
   *           Optional named time zone of the time, represented by the
   *           {@code aTime} parameter. An example of a named time zone is
   *           "EST5EDT". Set to {@code null} if undefined.
   */
  public EPPRegistryBatchSchedule(String aTime, DayOfWeek aDayOfWeek, String aTimeZone) {
    this(Frequency.weekly, aTime);
    this.dayOfWeek = aDayOfWeek;
    this.timeZone = aTimeZone;
  }

  /**
   * Construct an instance of {@code EPPRegistryBatchSchedule} that is used for
   * a day of month schedule.
   *
   * @param aTime
   *           The time of the schedule following the XML schema "time" type
   *           format
   * @param aDayOfMonth
   *           The day of month in the range of 1 - 31. Execution will not
   *           occur in the current month if the {@code aDayOfMonth} value is
   *           out-of-range for the current month (e.g, 29 - 31).
   * @param aTimeZone
   *           Optional named time zone of the time, represented by the
   *           {@code aTime} parameter. An example of a named time zone is
   *           "EST5EDT". Set to {@code null} if undefined.
   */
  public EPPRegistryBatchSchedule(String aTime, Integer aDayOfMonth, String aTimeZone) {
    this(Frequency.monthly, aTime);
    this.dayOfMonth = aDayOfMonth;
    this.timeZone = aTimeZone;
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPRegistryDomain} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         {@code EPPRegistryDomain} instance.
   *
   * @exception EPPEncodeException
   *               - Unable to encode {@code EPPRegistryBatchSchedule}
   *               instance.
   */
  @Override
  public Element encode(Document aDocument) throws EPPEncodeException {
    // Validate required state
    if (this.frequency == null) {
      throw new EPPEncodeException("Invalid state on EPPRegistryBatchSchedule.encode: frequency is not set");
    }
    if (this.time == null) {
      throw new EPPEncodeException("Invalid state on EPPRegistryBatchSchedule.encode: time is not set");
    }
    if (this.frequency == Frequency.daily && (this.hasDayOfWeek() || this.hasDayOfMonth())) {
      throw new EPPEncodeException(
            "Invalid state on EPPRegistryBatchSchedule.encode: dayOfWeek or dayOfMonth set with daily frequency");
    }
    if (this.frequency == Frequency.weekly && !this.hasDayOfWeek()) {
      throw new EPPEncodeException(
            "Invalid state on EPPRegistryBatchSchedule.encode: dayOfWeek not set with weekly frequency");
    }
    if (this.frequency == Frequency.monthly && !this.hasDayOfMonth()) {
      throw new EPPEncodeException(
            "Invalid state on EPPRegistryBatchSchedule.encode: dayOfMonth not set with monthly frequency");
    }

    Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

    // Frequency
    root.setAttribute(ATTR_FREQUENCY, this.frequency.toString());

    // Time
    Text timeVal = aDocument.createTextNode(this.time);
    root.appendChild(timeVal);

    // Time Zone
    if (this.hasTimeZone()) {
      root.setAttribute(ATTR_TZ, this.timeZone);
    }

    // Day Of Week
    if (this.frequency == Frequency.weekly && this.hasDayOfWeek()) {
      root.setAttribute(ATTR_DAY_OF_WEEK, this.dayOfWeek.toString());
    }

    // Day Of Month
    if (this.frequency == Frequency.monthly && this.hasDayOfMonth()) {
      root.setAttribute(ATTR_DAY_OF_MONTH, this.dayOfMonth.toString());
    }

    return root;
  }

  /**
   * Decode the {@code EPPRegistryBatchSchedule} attributes from the aElement
   * DOM Element tree.
   * 
   * @param aElement
   *           Root DOM Element to decode {@code EPPRegistryBatchSchedule}
   *           from.
   * 
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  public void decode(Element aElement) throws EPPDecodeException {

    // Frequency
    String theFrequencyStr = EPPUtil.decodeStringAttr(aElement, ATTR_FREQUENCY);
    if (theFrequencyStr != null) {
      this.frequency = Frequency.valueOf(theFrequencyStr);
    }
    else {
      this.frequency = null;
    }

    // Time
    this.time = EPPUtil.decodeStringValue(aElement);

    // Time Zone
    this.timeZone = EPPUtil.decodeStringAttr(aElement, ATTR_TZ);

    // Day of Week
    String theDayOfWeekStr = EPPUtil.decodeStringAttr(aElement, ATTR_DAY_OF_WEEK);
    if (theDayOfWeekStr != null) {
      this.dayOfWeek = DayOfWeek.getDayOfWeek(theDayOfWeekStr);
    }
    else {
      this.dayOfWeek = null;
    }

    // Day of Month
    String DayOfMonthStr = EPPUtil.decodeStringAttr(aElement, ATTR_DAY_OF_MONTH);
    if (DayOfMonthStr != null) {
      this.dayOfMonth = Integer.parseInt(DayOfMonthStr);
    }
    else {
      this.dayOfMonth = null;
    }

  }

  /**
   * Clone {@code EPPRegistryBatchSchedule}.
   * 
   * @return clone of {@code EPPRegistryBatchSchedule}
   * 
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  public Object clone() throws CloneNotSupportedException {
    return super.clone();
  }

  /**
   * implements a deep {@code EPPRegistryBatchSchedule} compare.
   * 
   * @param aObject
   *           {@code EPPRegistryBatchSchedule} instance to compare with
   * 
   * @return {@code true} if this object is the same as the aObject argument;
   *         {@code false} otherwise
   */
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPRegistryBatchSchedule)) {
      return false;
    }

    EPPRegistryBatchSchedule theComp = (EPPRegistryBatchSchedule) aObject;

    // Frequency
    if (!EqualityUtil.equals(this.frequency, theComp.frequency)) {
      cat.error("EPPRegistryBatchSchedule.equals(): policy not equal");
      return false;
    }

    // Time
    if (!EqualityUtil.equals(this.time, theComp.time)) {
      cat.error("EPPRegistryBatchSchedule.equals(): time not equal");
      return false;
    }

    // Time Zone
    if (!EqualityUtil.equals(this.timeZone, theComp.timeZone)) {
      cat.error("EPPRegistryBatchSchedule.equals(): timeZone not equal");
      return false;
    }

    // Day of Week
    if (!EqualityUtil.equals(this.dayOfWeek, theComp.dayOfWeek)) {
      cat.error("EPPRegistryBatchSchedule.equals(): dayOfWeek not equal");
      return false;
    }

    // Day of Month
    if (!EqualityUtil.equals(this.dayOfMonth, theComp.dayOfMonth)) {
      cat.error("EPPRegistryBatchSchedule.equals(): dayOfMonth not equal");
      return false;
    }

    return true;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   * 
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Gets the schedule execution frequency.
   * 
   * @return The schedule execution frequency if defined; {@code null}
   *         otherwise.
   */
  public Frequency getFrequency() {
    return this.frequency;
  }

  /**
   * Sets the schedule execution frequency.
   * 
   * @param aFrequency
   *           The schedule execution frequency
   */
  public void setFrequency(Frequency aFrequency) {
    this.frequency = aFrequency;
  }

  /**
   * Gets the schedule execution time using the XML schema "time" type format.
   * Examples include "14:00:00" for a time without a time zone offset,
   * "07:00:00-05:00" for a time with a 5 hour offset from UTC or EST time
   * zone, and "17:00:00Z" for 5 PM UTC.
   * 
   * @return The schedule execution time if defined; {@code null} otherwise.
   */
  public String getTime() {
    return this.time;
  }

  /**
   * Sets the schedule execution time using the XML schema "time" type format.
   * Examples include "14:00:00" for a time without a time zone offset,
   * "07:00:00-05:00" for a time with a 5 hour offset from UTC or EST time
   * zone, and "17:00:00Z" for 5 PM UTC.
   * 
   * @param aTime
   *           The schedule execution time
   */
  public void setTime(String aTime) {
    this.time = aTime;
  }

  /**
   * Is the schedule execution named time zone defined?
   *
   * @return {@code true} if the schedule execution named time zone is defined;
   *         {@code false} otherwise.
   */
  public boolean hasTimeZone() {
    return (this.timeZone != null ? true : false);
  }

  /**
   * Gets the optional schedule execution named time zone, such as "EST5EDT".
   * 
   * @return The schedule execution named time zone if defined; {@code null}
   *         otherwise.
   */
  public String getTimeZone() {
    return this.timeZone;
  }

  /**
   * Sets the optional schedule execution named time zone, such as "EST5EDT".
   * 
   * @param aTimeZone
   *           The schedule execution named time zone
   */
  public void setTimeZone(String aTimeZone) {
    this.timeZone = aTimeZone;
  }

  /**
   * Is the day of week defined?
   *
   * @return {@code true} if the day of week defined; {@code false} otherwise.
   */
  public boolean hasDayOfWeek() {
    return (this.dayOfWeek != null ? true : false);
  }

  /**
   * Gets the optional day of week.
   * 
   * @return The day of week if defined; {@code null} otherwise.
   */
  public DayOfWeek getDayOfWeek() {
    return this.dayOfWeek;
  }

  /**
   * Sets the optional day of week.
   * 
   * @param aDayOfWeek
   *           The day of week. Set to {@code null} if undefined.
   */
  public void setDayOfWeek(DayOfWeek aDayOfWeek) {
    this.dayOfWeek = aDayOfWeek;
  }

  /**
   * Is the day of month defined?
   *
   * @return {@code true} if the day of month defined; {@code false} otherwise.
   */
  public boolean hasDayOfMonth() {
    return (this.dayOfMonth != null ? true : false);
  }

  /**
   * Gets the optional day of month.
   * 
   * @return The day of month if defined; {@code null} otherwise.
   */
  public Integer getDayOfMonth() {
    return this.dayOfMonth;
  }

  /**
   * Sets the optional day of month.
   * 
   * @param aDayOfMonth
   *           The day of month. Set to {@code null} if undefined.
   */
  public void setDayOfMonth(Integer aDayOfMonth) {
    this.dayOfMonth = aDayOfMonth;
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   *
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPRegistryMapFactory.NS;
  }

}
