/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.serverstub;

import java.util.Enumeration;
import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPFactory;
import com.verisign.epp.framework.EPPAssembler;
import com.verisign.epp.framework.EPPDispatcher;
import com.verisign.epp.framework.EPPPollHandler;
import com.verisign.epp.framework.EPPPollQueueMgr;
import com.verisign.epp.framework.EPPXMLAssembler;
import com.verisign.epp.transport.EPPConException;
import com.verisign.epp.transport.EPPServerCon;
import com.verisign.epp.transport.EPPSrvFactorySingle;
import com.verisign.epp.util.EPPEnv;
import com.verisign.epp.util.EPPEnvException;
import com.verisign.epp.util.EPPEnvSingle;
import com.verisign.epp.transport.server.EPPQuicClientConnectionHandler;

/**
 * The {@code QuicServer} class is responsible for reading the config file,
 * instantiating an implementation of a ServerSocket (Plain or SSL), and
 * specifying the {@code ServerEventHandler} class that will be instantiated
 * with each new client connection.
 */
public class QuicServer {
	/** Category for logging */
	private static Logger cat = LoggerFactory.getLogger(QuicServer.class);

	/**
	 * Construct a QuicServer instance
	 *
	 * @param configFileName
	 *           The name of the config file where EPP properites are located
	 */
	public QuicServer(String configFileName) {
		try {
			// Initialize environment settings
			EPPEnvSingle env = EPPEnvSingle.getInstance();

			env.initialize(configFileName);

			// Initialize the dispatcher
			initializeDispatcher();

			// Initialize the data source (in memory queue)
			EPPPollQueueMgr.getInstance().setDataSource(new PollDataSource());

			// Initialize the poll queue
			initializePollQueue();
		}
		catch (EPPEnvException e) {
			e.printStackTrace();
			System.exit(1);
		}
		catch (Exception e) {
			e.printStackTrace();
			System.exit(1);
		}
	}

	/**
	 * Initialize the poll handler based on the EPP.PollHandlers,
	 */
	public void initializePollQueue() {
		EPPPollQueueMgr thePollQueue = EPPPollQueueMgr.getInstance();

		try {
			Vector<String> handlerClasses = EPPEnv.getPollHandlers();

			if (handlerClasses != null) {
				Enumeration<String> e = handlerClasses.elements();

				while (e.hasMoreElements()) {
					String handlerClassName = e.nextElement();
					Class handlerClass = Class.forName(handlerClassName);

					// register poll handler
					thePollQueue.register((EPPPollHandler) handlerClass.getDeclaredConstructor().newInstance());
					cat.info("Successfully loaded poll handler: " + handlerClass.getName());
				}

			}

		}
		catch (Exception e) {
			cat.error("EPP QuicServer couldn't initialize the environment:" + e);
			System.exit(1);
		}
	}

	/**
	 * Initializes {@code QuicServer}'s dispatcher to handle the commands defined by
	 * the supported EPP mappings. At this point we also get the list of Command
	 * Response extensions class names from the epp.config file and add the
	 * CommandResponseextension classes to the EPPExtFaactory
	 */
	public void initializeDispatcher() {
		/**
		 * Register the EPP Event Handlers with the Dispatcher. The fully
		 * qualified class names of the event handlers should be listed in the
		 * conf file. EPPEnv had already read them into a Vector that we can get.
		 */
		EPPDispatcher theDispatcher = EPPDispatcher.getInstance();

		try {

			EPPFactory theFactory = EPPFactory.getInstance();

			/**
			 * Add the command response level extensions to the extension factories
			 * provided if any of the command response level extensions exist. This
			 * can be done by adding the command response level extensions to the
			 * EPPFactories method addExtFactory
			 */
			Vector<String> commandExts = EPPEnv.getCmdResponseExtensions();

			if ((commandExts != null) && commandExts.elements().hasMoreElements()) {
				for (int i = 0; i < commandExts.size(); i++) {
					String commandExtensionClassName = commandExts.elementAt(i);

					try {
						theFactory.addExtFactory(commandExtensionClassName);
					}
					catch (EPPCodecException ex) {
						cat.error(
						      "Couldn't load the Extension Factory" + "associated with the CommandResponseExtensions" + ex);

						System.exit(1);
					}

					cat.info("Successfully loaded Command Extension Class:" + commandExtensionClassName);
				}

			}

			/**
			 * Add the object mapping level extensions to the map factories
			 * provided if any of the object mapping level extensions exist. This
			 * can be done by adding the object mapping level extensions to the
			 * EPPFactories method addMapFactory
			 */
			Vector<String> theMapFactories = EPPEnv.getMapFactories();

			if ((theMapFactories != null) && theMapFactories.elements().hasMoreElements()) {
				for (int i = 0; i < theMapFactories.size(); i++) {
					String mapFactoryClassName = theMapFactories.elementAt(i);

					try {
						theFactory.addMapFactory(mapFactoryClassName);
					}
					catch (EPPCodecException ex) {
						cat.error("Couldn't load the Map Factory " + mapFactoryClassName + ":" + ex);

						System.exit(1);
					}

					cat.info("Successfully loaded Map Factory Class:" + mapFactoryClassName);
				}

			}

			Vector<String> theServerEventHandlers = EPPEnv.getServerEventHandlers();
			Enumeration<String> e = theServerEventHandlers.elements();

			while (e.hasMoreElements()) {
				String handlerClassName = e.nextElement();
				Class handlerClass = Class.forName(handlerClassName);
				theDispatcher.registerHandler(
				      (com.verisign.epp.framework.EPPEventHandler) handlerClass.getDeclaredConstructor().newInstance());

				cat.info("Successfully loaded QuicServer handler: " + handlerClass.getName());
			}

		}
		catch (EPPEnvException e) {
			cat.error("Couldn't initialize the environment", e);
			System.exit(1);
		}
		catch (InstantiationException e) {
			cat.error("Couldn't instantiate one of the specified QuicServer event handlers:", e);
			System.exit(1);
		}
		catch (IllegalAccessException e) {
			cat.error("Couldn't instantiate one of the specified QuicServer event handlers"
			      + "\n The class or initializer is not accessible:", e);
			System.exit(1);
		}
		catch (ClassNotFoundException e) {
			cat.error("Couldn't instantiate one of the event handlers listed in the conf file:", e);
			System.exit(1);
		}
		catch (Exception e) {
			cat.error("General exception in initializeDispatcher: ", e);
			System.exit(1);
		}

		/**
		 * See if a specific EPPAssembler has been specified. If so load it.
		 */
		try {
			String assemblerClassName = EPPEnv.getServerEPPAssembler();

			if ((assemblerClassName != null) && (assemblerClassName.length() > 0)) {
				cat.info("Found Assembler class in config: [" + assemblerClassName + "]");
				cat.info("Attempting to instantiate " + assemblerClassName + "...");

				Class assemblerClass = Class.forName(assemblerClassName);
				EPPAssembler assembler = (com.verisign.epp.framework.EPPAssembler) assemblerClass.getDeclaredConstructor()
				      .newInstance();

				theDispatcher.setAssembler(assembler);

				cat.info("Successfully set assembler " + assemblerClassName);
			}
			else {
				cat.info("No EPPServerAssembler specified. Using default... "
				      + "(com.verisign.epp.framework.EPPXMLAssembler)");

				theDispatcher.setAssembler(new EPPXMLAssembler());
			}
		}
		catch (Exception e) {
			cat.error("EoQ Server Initialization Exception: " + e);
			System.exit(1);
		}

		/**
		 * Register the ConnectionHandler so that the ServerStub's custom greeting
		 * is always sent to connecting clients.
		 */
		theDispatcher.registerConnectionHandler(new ConnectionHandler());
	}

	/**
	 * Runs the EPP over TCP (EoT) Stub QuicServer
	 */
	private void run() {
		// Initialize the QuicServer connection factory
		EPPSrvFactorySingle theFactory = EPPSrvFactorySingle.getInstance();

		cat.info("EoQ Server: Starting Server...");

		EPPServerCon theServer;
		try {
			theServer = theFactory.getEPPServer();
			theServer.RunServer(new EPPQuicClientConnectionHandler());
		}
		catch (Exception e) {
			cat.error("EPP QuicServer: Run Exception: " + e);
			e.printStackTrace();
			System.exit(1);
		}
	}

	/**
	 * Runs the EPP Stub QuicServer QuicServer with the usage:<br>
	 * java QuicServer [-help|&lt;config-file&gt;]<br>
	 *
	 * <pre>
	 *    -help Gets the help
	 *    &lt;config-file&gt; EPP configuration file, with default of "epp.config"
	 * </pre>
	 *
	 * @param args
	 *           Command line arguments
	 */
	public static void main(String[] args) {
		String theConfigFile = "epp.config";

		if (args.length == 1) {

			if (args[0].equalsIgnoreCase("-help")) {
				System.out.println("Usage: java QuicServer [-help|<config-file>]");
				System.out.println("Executes the EPP over QUIC (EoQ) Stub Server");
				System.out.println("\t-help gets this help");
				System.out.println("\t<config-file> EPP SDK configuration settings, with default of \"epp.config\"");
				System.exit(0);

			}
			theConfigFile = args[0];
		}
		else if (args.length > 1) {
			System.out.println("Usage: java QuicServer [-help|<config-file>]");
			System.out.println("Executes the EPP over QUIC (EoQ) Stub Server");
			System.out.println("\t-help gets this help");
			System.out.println("\t<config-file> EPP SDK configuration settings, with default of \"epp.config\"");
			System.exit(1);
		}

		QuicServer theServer = new QuicServer(theConfigFile);
		theServer.run();
	}

}
