
/***********************************************************
Copyright (C) 2024 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

Quic://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.pool;


import com.verisign.epp.interfaces.EPPCommandException;
import com.verisign.epp.interfaces.EPPQuicSession;
import com.verisign.epp.pool.EPPPooledSession;
import com.verisign.epp.transport.client.EPPSSLContext;

/**
 * Pooled generic {@code EPPQuicSession}. Timestamp attributes were added to
 * handle absolute session timeout and ensure that idle timeouts don't occur.
 */
public class EPPPooledGenericQuicSession extends EPPQuicSession implements EPPPooledSession {

	/**
	 * When pooled object was created
	 */
	private long createdTime = System.currentTimeMillis();

	/**
	 * Last time session was used or a keep alive was sent
	 */
	private long lastTouched = System.currentTimeMillis();

	/**
	 * Absolute timeout for this session.
	 */
	private long absoluteTimeout = EPPSessionPoolableFactory.TIMEOUT_UNSET;

	/**
	 * Default constructor for {@code EPPPooledGenericSession}.
	 * 
	 * @throws EPPCommandException
	 *            On error
	 */
	public EPPPooledGenericQuicSession() throws EPPCommandException {
		super();
	}

	/**
	 * Constructor that takes an explicit URL to connect to.
	 * 
	 * @param aUrl
	 *           URL to connect to
	 * 
	 * @throws EPPCommandException
	 *            Error connecting to server
	 */
	public EPPPooledGenericQuicSession(String aUrl) throws EPPCommandException {
		super(aUrl);
	}

	/**
	 * Constructor that takes an explicit URL to connect to and
	 * {@code EPPSSLContext} to use.
	 * 
	 * @param aUrl
	 *           URL to connect to
	 * @param aSSLContext
	 *           Optional specific SSL context to use. Set to {@code null} if
	 *           undefined.
	 * 
	 * @throws EPPCommandException
	 *            Error connecting to server
	 */
	public EPPPooledGenericQuicSession(String aUrl, EPPSSLContext aSSLContext) throws EPPCommandException {
		super(aUrl, aSSLContext);
	}

	/**
	 * Gets the time the pooled object was created.
	 * 
	 * @return Epoch time of creation
	 */
	public long getCreatedTime() {
		return createdTime;
	}

	/**
	 * Gets the last time the pooled object was touched.
	 * 
	 * @return Epoch time of touch
	 */
	public long getLastTouched() {
		return lastTouched;
	}

	/**
	 * Sets the last touched to the current time.
	 */
	public void touch() {
		this.lastTouched = System.currentTimeMillis();
	}

	/**
	 * Has absolute timeout set?
	 *
	 * @return {@code true} if absolute timeout are set; {@code false} otherwise.
	 */
	public boolean hasAbsoluteTimeout() {
		if (this.absoluteTimeout != EPPSessionPoolableFactory.TIMEOUT_UNSET) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Gets the session absolute timeout in milliseconds.
	 *
	 * @return Absolute timeout in milliseconds
	 */
	public long getAbsoluteTimeout() {
		return this.absoluteTimeout;
	}

	/**
	 * Sets the session absolute timeout in milliseconds.
	 *
	 * @param aAbsoluteTimeout
	 *           Absolute timeout in milliseconds.
	 */
	public void setAbsoluteTimeout(long aAbsoluteTimeout) {
		this.absoluteTimeout = aAbsoluteTimeout;
	}

}