/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.orgext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * <code>EPPOrgExtId</code> is used to define an organization identifier with a
 * required role. The organization identifier can be empty in the case of a
 * delete.
 */
public class EPPOrgExtId implements EPPCodecComponent {

  /**
   * Category for logging
   */
      private static Logger cat = LoggerFactory.getLogger(EPPOrgExtId.class);

  /**
   * XML local name for <code>EPPOrgExtId</code>.
   */
  public static final String ELM_LOCALNAME = "id";

  /**
   * XML root tag for <code>EPPOrgExtId</code>.
   */
  public static final String ELM_NAME = EPPOrgExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * Name of XML attribute for the &quot;role&quot; attribute.
   */
  private static final String ATTR_ROLE = "role";

  /**
   * XML local name for the org identifier element
   */
  private static final String ELM_ORG_ID = "id";

  /**
   * The role of the linked organization.
   */
  private String role;

  /**
   * The organization identifier of the linked organization, which can be
   * empty.
   */
  private String orgId;

  /**
   * Default constructor for <code>EPPOrgExtId</code>.
   */
  public EPPOrgExtId() {
  }

  /**
   * Constructor for <code>EPPOrgExtId</code> that takes the role.
   * 
   * @param aRole
   *           Organization role
   * 
   */
  public EPPOrgExtId(String aRole) {
    this.setRole(aRole);
  }

  /**
   * Constructor for <code>EPPOrgExtId</code> that takes the role and the
   * organization identifier value.
   * 
   * @param aRole
   *           Organization role
   * @param aOrgId
   *           Organization identifier
   * 
   */
  public EPPOrgExtId(String aRole, String aOrgId) {
    this.setRole(aRole);
    this.setOrgId(aOrgId);
  }

  /**
   * Is the organization identifier defined?
   * 
   * @return <code>true</code> if the organization identifier is defined;
   *         <code>false</code> otherwise.
   */
  public boolean hasOrgId() {
    return (this.orgId != null ? true : false);
  }

  /**
   * Gets the organization identifier.
   * 
   * @return The organization identifier if defined;<code>null</code>
   *         otherwise.
   */
  public String getOrgId() {
    return this.orgId;
  }

  /**
   * Sets the organization identifier.
   * 
   * @param aOrgId
   *           The organization identifier
   */
  public void setOrgId(String aOrgId) {
    this.orgId = aOrgId;
  }

  /**
   * Gets the organization role.
   * 
   * @return The organization role if defined;<code>null</code> otherwise.
   */
  public String getRole() {
    return this.role;
  }

  /**
   * Sets the organization role.
   * 
   * @param aRole
   *           The organization role
   */
  public void setRole(String aRole) {
    this.role = aRole;
  }

  /**
   * encode instance into a DOM element tree. A DOM Document is passed as an
   * argument and functions as a factory for DOM objects. The root element
   * associated with the instance is created and each instance attribute is
   * appended as a child node.
   * 
   * @param aDocument
   *           DOM Document, which acts is an Element factory
   * 
   * @return Element Root element associated with the object
   * 
   * @exception EPPEncodeException
   *               Error encoding <code>EPPOrgExtId</code>
   */
  public Element encode(Document aDocument) throws EPPEncodeException {

    // Check required attributes
    if (this.role == null) {
      throw new EPPEncodeException("Undefined role in EPPOrgExtId");
    }
    if (this.orgId == null) {
      throw new EPPEncodeException("Undefined orgId in EPPOrgExtId");
    }

    // Create root element
    Element root = aDocument.createElementNS(EPPOrgExtFactory.NS, ELM_NAME);

    // Org Identifier
    root.appendChild(aDocument.createTextNode(this.orgId));

    // Org Role
    root.setAttribute(ATTR_ROLE, this.role);

    return root;
  }

  /**
   * decode a DOM element tree to initialize the instance attributes. The
   * <code>aElement</code> argument represents the root DOM element and is used
   * to traverse the DOM nodes for instance attribute values.
   * 
   * @param aElement
   *           <code>Element</code> to decode
   * 
   * @exception EPPDecodeException
   *               Error decoding <code>Element</code>
   */
  public void decode(Element aElement) throws EPPDecodeException {

    // Org Identifier
    Node theTextNode = aElement.getFirstChild();
    if (theTextNode != null) {
      this.orgId = theTextNode.getNodeValue();
    }

    // Org Role
    String theRole = aElement.getAttribute(ATTR_ROLE);

    if (theRole != null && !theRole.isEmpty()) {
      this.role = theRole;
    }
    else {
      this.role = null;
    }
  }

  /**
   * clone an <code>EPPCodecComponent</code>.
   * 
   * @return clone of concrete <code>EPPOrgExt</code>
   * 
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  public Object clone() throws CloneNotSupportedException {
    EPPOrgExtId clone = (EPPOrgExtId) super.clone();

    return clone;
  }

  /**
   * Implementation of <code>Object.toString</code>, which will result in an
   * indented XML <code>String</code> representation of the concrete
   * <code>EPPCodecComponent</code>.
   * 
   * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
   *         otherwise.
   */
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Compare an instance of <code>EPPOrgExt</code> with this instance.
   * 
   * @param aObject
   *           Object to compare with.
   * 
   * @return <code>true</code> if equal; <code>false</code> otherwise.
   */
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPOrgExtId)) {
      return false;
    }

    EPPOrgExtId other = (EPPOrgExtId) aObject;

    // Org Identifier
    if (!EqualityUtil.equals(this.orgId, other.orgId)) {
      cat.error("EPPOrgExtId.equals(): orgId not equal");
      return false;
    }

    // Org Role
    if (!EqualityUtil.equals(this.role, other.role)) {
      cat.error("EPPOrgExtId.equals(): role not equal");
      return false;
    }

    return true;
  }

  /**
   * Returns the XML namespace associated with the
   * <code>EPPCodecComponent</code>.
   * 
   * @return XML namespace for the <code>EPPCodecComponent</code>.
   */
  @Override
  public String getNamespace() {
    return EPPOrgExtFactory.NS;
  }

}
