/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.org;

//----------------------------------------------
//
// imports...
//
//----------------------------------------------
// Java Core Imports
import java.util.Date;

// Log4j Imports
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
// W3C Imports
import org.w3c.dom.Element;
import org.w3c.dom.Text;

// EPP Imports
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Poll message used to indicate the result of a pending action. Pending action
 * can included domain create, update, renew, transfer, and delete.
 *
 * @see com.verisign.epp.codec.org.EPPOrgCreateCmd
 * @see com.verisign.epp.codec.org.EPPOrgUpdateCmd
 * @see com.verisign.epp.codec.org.EPPOrgDeleteCmd
 */
public class EPPOrgPendActionMsg extends EPPResponse {

  /**
   * Category for logging
   */
      private static Logger cat = LoggerFactory.getLogger(EPPOrgPendActionMsg.class);
        

  /**
   * XML local name for {@code EPPOrgPendActionMsg}.
   */
  public static final String ELM_LOCALNAME = "panData";

  /**
   * XML root tag for {@code EPPOrgPendActionMsg}.
   */
  public static final String ELM_NAME = EPPOrgMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * XML Element Name for the {@code orgId} attribute.
   */
  private final static String ELM_ORG_ID = "id";

  /**
   * XML Element Name for the {@code paTRID} attribute.
   */
  private final static String ELM_PATRID = "paTRID";

  /**
   * XML Element Name for the {@code paDate} attribute.
   */
  private final static String ELM_PENDING_DATE = "paDate";

  /**
   * XML Attribute Name for the pending action result
   */
  private final static String ATTR_RESULT = "paResult";

  /**
   * The org identifier of the org.
   */
  private String orgId = null;

  /**
   * Was the pending request successful? A value of {@code true} indicates that
   * the request has been approved and completed. A value of {@code false}
   * indicates that the request has been denied and the requested action has
   * not been taken. Default value is {@code false}.
   */
  private boolean paSuccess = false;

  /**
   * Pending transaction id contains the client transaction identifier and
   * server transaction identifier returned with the original response to
   * process the command.
   */
  EPPTransId pendingTransId = null;

  /**
   * Pending date and time describing when review of the requested action was
   * completed.
   */
  private Date pendingDate = null;

  /**
   * {@code EPPOrgPendActionMsg} default constructor. Must call required setter
   * methods before encode.
   */
  public EPPOrgPendActionMsg() {
  }

  /**
   * {@code EPPOrgPendActionMsg} constructor that takes all of the required
   * attributes.
   *
   * @param aTransId
   *           Poll command transaction id
   * @param aOrgId
   *           Org Id of pending action
   * @param aSuccess
   *           Was the pending action successful
   * @param aPendingTransId
   *           Pending action transaction id
   * @param aPendingDate
   *           Date of pending action completion
   */
  public EPPOrgPendActionMsg(EPPTransId aTransId, String aOrgId, boolean aSuccess, EPPTransId aPendingTransId,
        Date aPendingDate) {
    super(aTransId);

    this.orgId = aOrgId;
    this.paSuccess = aSuccess;
    this.setPendingTransId(aPendingTransId);
    this.pendingDate = aPendingDate;
  }

  /**
   * Gets the EPP command type associated with {@code EPPOrgPendActionMsg}.
   *
   * @return EPPOrgPendActionMsg.ELM_NAME
   */
  @Override
  public String getType() {
    return ELM_NAME;
  }

  /**
   * Gets the EPP command Namespace associated with {@code EPPOrgPendActionMsg}
   * .
   *
   * @return {@code EPPOrgMapFactory.NS}
   */
  @Override
  public String getNamespace() {
    return EPPOrgMapFactory.NS;
  }

  /**
   * Gets the the server-unique identifier of the organization object.
   *
   * @return The the server-unique identifier of the organization object if
   *         defined;<code>null</code> otherwise.
   */
  public String getOrgId() {
    return this.orgId;
  }

  /**
   * Sets the org identifier.
   *
   * @param aOrgId
   *           The org identifier
   */
  public void setOrgId(String aOrgId) {
    this.orgId = aOrgId;
  }

  /**
   * Gets the date that the pending action was completed.
   *
   * @return Returns the pendingDate.
   */
  public Date getPendingDate() {
    return this.pendingDate;
  }

  /**
   * Sets the date that the pending action was completed.
   *
   * @param aPendingDate
   *           The pendingDate to set.
   */
  public void setPendingDate(Date aPendingDate) {
    this.pendingDate = aPendingDate;
  }

  /**
   * Gets the pending transaction id.
   *
   * @return Returns the pendingTransId.
   */
  public EPPTransId getPendingTransId() {
    return this.pendingTransId;
  }

  /**
   * Sets the pending transaction id.
   *
   * @param pendingTransId
   *           The pendingTransId to set.
   */
  public void setPendingTransId(EPPTransId pendingTransId) {
    this.pendingTransId = pendingTransId;
    this.pendingTransId.setRootName(EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_PATRID);
  }

  /**
   * Was the pending action successful?
   *
   * @return Returns {@code true} if the pending action was successfully
   *         completed; {@code false} otherwise.
   */
  public boolean isPASuccess() {
    return this.paSuccess;
  }

  /**
   * Sets if the pending action was successful.
   *
   * @param aSuccess
   *           The success to set.
   */
  public void setPASuccess(boolean aSuccess) {
    this.paSuccess = aSuccess;
  }

  /**
   * Compare an instance of {@code EPPOrgPendActionMsg} with this instance.
   *
   * @param aObject
   *           Object to compare with.
   *
   * @return DOCUMENT ME!
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPOrgPendActionMsg)) {
      cat.error(
            "EPPOrgPendActionMsg.equals(): " + aObject.getClass().getName() + " not EPPOrgPendActionMsg instance");

      return false;
    }

    if (!super.equals(aObject)) {
      cat.error("EPPOrgPendActionMsg.equals(): super class not equal");

      return false;
    }

    EPPOrgPendActionMsg theComp = (EPPOrgPendActionMsg) aObject;

    // Org Identifier
    if (!EqualityUtil.equals(this.orgId, theComp.orgId)) {
      cat.error("EPPOrgPendActionMsg.equals(): orgId not equal");
      return false;
    }

    // PA Success
    if (!EqualityUtil.equals(this.paSuccess, theComp.paSuccess)) {
      cat.error("EPPOrgPendActionMsg.equals(): paSuccess not equal");
      return false;
    }

    // Pending Transaction Id
    if (!EqualityUtil.equals(this.pendingTransId, theComp.pendingTransId)) {
      cat.error("EPPOrgPendActionMsg.equals(): pendingTransId not equal");
      return false;
    }

    // Pending Date
    if (!EqualityUtil.equals(this.pendingDate, theComp.pendingDate)) {
      cat.error("EPPOrgPendActionMsg.equals(): pendingDate not equal");
      return false;
    }

    return true;
  }

  /**
   * Clone {@code EPPOrgPendActionMsg}.
   *
   * @return clone of {@code EPPOrgPendActionMsg}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPOrgPendActionMsg clone = (EPPOrgPendActionMsg) super.clone();

    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Validate the state of the {@code EPPOrgPendActionMsg} instance. A valid
   * state means that all of the required attributes have been set. If
   * validateState returns without an exception, the state is valid. If the
   * state is not valid, the EPPCodecException will contain a description of
   * the error. throws EPPCodecException State error. This will contain the
   * name of the attribute that is not valid.
   *
   * @throws EPPCodecException
   *            DOCUMENT ME!
   */
  void validateState() throws EPPCodecException {
    if (this.orgId == null) {
      cat.error("EPPOrgPendActionMsg.validateState(): required attribute orgId is not set");
      throw new EPPCodecException("required attribute orgId is not set");
    }

    if (this.pendingTransId == null) {
      cat.error("EPPOrgPendActionMsg.validateState(): required attribute pendingTransId is not set");
      throw new EPPCodecException("required attribute pendingTransId is not set");
    }

    if (this.pendingDate == null) {
      cat.error("EPPOrgPendActionMsg.validateState(): required attribute pendingDate is not set");
      throw new EPPCodecException("required attribute pendingDate is not set");
    }
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPOrgPendActionMsg} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         {@code EPPOrgPendActionMsg} instance.
   *
   * @exception EPPEncodeException
   *               Unable to encode {@code EPPOrgPendActionMsg} instance.
   */
  @Override
  protected Element doEncode(Document aDocument) throws EPPEncodeException {
    // Validate Attributes
    try {
      validateState();
    }
    catch (EPPCodecException e) {
      throw new EPPEncodeException("Invalid state on EPPOrgPendActionMsg.encode: " + e);
    }

    Element root = aDocument.createElementNS(EPPOrgMapFactory.NS, ELM_NAME);

    // Org Id
    Element nameElm = aDocument.createElementNS(EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_ORG_ID);
    root.appendChild(nameElm);

    // PA Success
    if (this.paSuccess) {
      nameElm.setAttribute(ATTR_RESULT, "1");
    }
    else {
      nameElm.setAttribute(ATTR_RESULT, "0");
    }

    Text textNode = aDocument.createTextNode(this.orgId);
    nameElm.appendChild(textNode);

    // Pending Transaction Id
    EPPUtil.encodeComp(aDocument, root, this.pendingTransId);

    // Pending Date
    EPPUtil.encodeTimeInstant(aDocument, root, this.pendingDate, EPPOrgMapFactory.NS,
          EPPOrgMapFactory.NS_PREFIX + ":" + ELM_PENDING_DATE);

    return root;
  }

  /**
   * Decode the {@code EPPOrgPendActionMsg} attributes from the
   * {@code aElement} DOM Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPOrgPendActionMsg} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode {@code aElement}
   */
  @Override
  protected void doDecode(Element aElement) throws EPPDecodeException {
    // Org Id
    Element theNameElm = EPPUtil.getElementByTagNameNS(aElement, EPPOrgMapFactory.NS, ELM_ORG_ID);
    this.orgId = theNameElm.getFirstChild().getNodeValue();

    // PA Success
    this.paSuccess = EPPUtil.decodeBooleanAttr(theNameElm, ATTR_RESULT);

    // Pending Transaction ID
    this.pendingTransId = (EPPTransId) EPPUtil.decodeComp(aElement, EPPOrgMapFactory.NS, ELM_PATRID,
          EPPTransId.class);

    // Pending Date
    this.pendingDate = EPPUtil.decodeTimeInstant(aElement, EPPOrgMapFactory.NS, ELM_PENDING_DATE);

    // Validate Attributes
    try {
      validateState();
    }
    catch (EPPCodecException e) {
      throw new EPPDecodeException("Invalid state on EPPOrgPendActionMsg.decode: " + e);
    }
  }

}
