/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.org;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * <code>EPPOrgCheckResult</code> is used to represent the for the check of
 * an individual org. The org information includes the org
 * identifier, whether the org identifier is available, and optionally the
 * reason that the org identifier is not available.
 */
public class EPPOrgCheckResult implements EPPCodecComponent {

  /**
   * Category for logging
   */
  private static Logger cat = LoggerFactory.getLogger(EPPOrgCheckResult.class);

  /**
   * XML local name for <code>EPPOrgCheckResult</code>.
   */
  public static final String ELM_LOCALNAME = "cd";

  /**
   * XML root tag for <code>EPPOrgCheckResult</code>.
   */
  public static final String ELM_NAME = EPPOrgMapFactory.NS_PREFIX + ":"
      + ELM_LOCALNAME;

  /**
   * XML local name for the org identifier element
   */
  private static final String ELM_ORG_ID = "id";

  /**
   * XML local name for the reason element
   */
  private static final String ELM_REASON = "reason";

  /**
   * XML local name for the avail attribute
   */
  private static final String ATTR_AVAIL = "avail";

  /**
   * XML local name for the reason lang attribute.
   */
  private static final String ATTR_LANG = "lang";

  /**
   * Value for the org identifier
   */
  private String orgId;

  /**
   * Is the <code>orgId</code> available?
   */
  private boolean available;

  /**
   * OPTIONAL reason element
   */
  private String reason;

  /**
   * OPTIONAL language of the reason.
   */
  private String reasonLang;

  /**
   * Default constructor for <code>EPPOrgCheckResult</code>.
   */
  public EPPOrgCheckResult() {
  }

  /**
   * Constructor for <code>EPPOrgCheckResult</code> that takes org
   * identifier and whether the org identifier is available.
   * 
   * @param aOrgId
   *            The org identifier
   * @param aAvailable
   *            Is the org identifier available?
   */
  public EPPOrgCheckResult(String aOrgId, boolean aAvailable) {
    this.setOrgId(aOrgId);
    this.setAvailable(aAvailable);
  }

  /**
   * Constructor for <code>EPPOrgCheckResult</code> that is used for
   * unavailable org identifiers providing the reason the org
   * identifier is not available.
   * 
   * @param aOrgId
   *            The org identifier
   * @param aReason
   *            Reason that the org identifier is not available.
   */
  public EPPOrgCheckResult(String aOrgId, String aReason) {
    this.setOrgId(aOrgId);
    this.setAvailable(false);
    this.setReason(aReason);
  }

  /**
   * Constructor for <code>EPPOrgCheckResult</code> that is used for
   * unavailable org identifiers providing the reason the org
   * identifier is not available with the reason language.
   * 
   * @param aOrgId
   *            The org identifier
   * @param aReason
   *            Reason that the org identifier is not available.
   * @param aReasonLang
   *            Reason language for reason value.
   */
  public EPPOrgCheckResult(String aOrgId, String aReason,
      String aReasonLang) {
    this.setOrgId(aOrgId);
    this.setAvailable(false);
    this.setReason(aReason);
    this.setReasonLang(aReasonLang);
  }

  /**
   * Gets the org identifier.
   * 
   * @return The org identifier if defined;<code>null</code> otherwise.
   */
  public String getOrgId() {
    return this.orgId;
  }

  /**
   * Sets the org identifier.
   * 
   * @param aOrgId
   *            The org identifier
   */
  public void setOrgId(String aOrgId) {
    this.orgId = aOrgId;
  }

  /**
   * Is the org identifier available?
   * 
   * @return <code>true</code> if the org identifier is available;
   *         <code>false</code> otherwise.
   */
  public boolean isAvailable() {
    return this.available;
  }

  /**
   * Sets the org identifier is available.
   * 
   * @param aAvailable
   *            <code>true</code> if the org identifier is available;
   *            <code>false</code> otherwise.
   */
  public void setAvailable(boolean aAvailable) {
    this.available = aAvailable;
  }

  /**
   * Is the reason defined?
   * 
   * @return <code>true</code> if the reason is defined; <code>false</code>
   *         otherwise.
   */
  public boolean hasReason() {
    return (this.reason != null ? true : false);
  }

  /**
   * Gets the reason that the org identifier is not available.
   * 
   * @return Reason that the org identifier is not available if defined;
   *         <code>null</code> otherwise.
   */
  public String getReason() {
    return this.reason;
  }

  /**
   * Sets the reason that the org identifier is not available.
   * 
   * @param aReason
   *            Reason that the org identifier is not available. Set to
   *            <code>null</code> to clear the reason.
   */
  public void setReason(String aReason) {
    this.reason = aReason;
  }

  /**
   * Is the reason language defined?
   * 
   * @return <code>true</code> if the reason language is defined;
   *         <code>false</code> otherwise.
   */
  public boolean hasReasonLang() {
    return (this.reasonLang != null ? true : false);
  }

  /**
   * Gets the reason language value.
   * 
   * @return Reason language if defined; <code>null</code> otherwise.
   */
  public String getReasonLang() {
    return this.reasonLang;
  }

  /**
   * Sets the reason language value.
   * 
   * @param aReasonLang
   *            Reason language for reason value.
   */
  public void setReasonLang(String aReasonLang) {
    this.reasonLang = aReasonLang;
  }

  /**
   * encode instance into a DOM element tree. A DOM Document is passed as an
   * argument and functions as a factory for DOM objects. The root element
   * associated with the instance is created and each instance attribute is
   * appended as a child node.
   * 
   * @param aDocument
   *            DOM Document, which acts is an Element factory
   * 
   * @return Element Root element associated with the object
   * 
   * @exception EPPEncodeException
   *                Error encoding <code>EPPOrgCheckResult</code>
   */
  public Element encode(Document aDocument) throws EPPEncodeException {

    // Check required attributes
    if (this.orgId == null) {
      throw new EPPEncodeException(
          "Undefined name in EPPOrgCheckResult");
    }

    // Domain
    Element root = aDocument.createElementNS(EPPOrgMapFactory.NS,
        ELM_NAME);

    // Org Identifier
    Element theElm = aDocument.createElementNS(EPPOrgMapFactory.NS,
        EPPOrgMapFactory.NS_PREFIX + ":" + ELM_ORG_ID);

    theElm.appendChild(aDocument.createTextNode(this.orgId));

    // Available
    EPPUtil.encodeBooleanAttr(theElm, ATTR_AVAIL, this.available);

    root.appendChild(theElm);

    // Reason and ReasonLang
    if (this.hasReason()) {
      Element reasonElm = aDocument.createElementNS(
          EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX
              + ":" + ELM_REASON);

      if (this.hasReasonLang()) {
        reasonElm.setAttribute(ATTR_LANG, this.reasonLang);
      }

      reasonElm.appendChild(aDocument.createTextNode(this.reason));

      root.appendChild(reasonElm);
    }

    return root;
  }

  /**
   * decode a DOM element tree to initialize the instance attributes. The
   * <code>aElement</code> argument represents the root DOM element and is
   * used to traverse the DOM nodes for instance attribute values.
   * 
   * @param aElement
   *            <code>Element</code> to decode
   * 
   * @exception EPPDecodeException
   *                Error decoding <code>Element</code>
   */
  public void decode(Element aElement) throws EPPDecodeException {

    // Org Identifier
    Element theNameElm = EPPUtil.getElementByTagNameNS(aElement,
        EPPOrgMapFactory.NS, ELM_ORG_ID);
    if (theNameElm != null) {
      Node textNode = theNameElm.getFirstChild();
      if (textNode != null) {
        this.orgId = textNode.getNodeValue();
      }
      else {
        this.orgId = null;
      }

      // Available
      this.available = EPPUtil.decodeBooleanAttr(theNameElm, ATTR_AVAIL);
    }

    // Reason and ReasonLang
    Element theReasonElm = EPPUtil.getElementByTagNameNS(aElement,
        EPPOrgMapFactory.NS, ELM_REASON);

    if (theReasonElm != null) {
      Node textNode = theReasonElm.getFirstChild();
      if (textNode != null) {
        this.reason = textNode.getNodeValue();

        String theReasonLang = theReasonElm.getAttribute(ATTR_LANG);
        if (theReasonLang != null && !theReasonLang.isEmpty()) {
          this.reasonLang = theReasonLang;
        }
        else {
          this.reasonLang = null;
        }

      }
      else {
        this.reason = null;
        this.reasonLang = null;
      }
    }
    else {
      this.reason = null;
      this.reasonLang = null;
    }
  }

  /**
   * clone an <code>EPPCodecComponent</code>.
   * 
   * @return clone of concrete <code>EPPOrgCheckResult</code>
   * 
   * @exception CloneNotSupportedException
   *                standard Object.clone exception
   */
  public Object clone() throws CloneNotSupportedException {
    EPPOrgCheckResult clone = (EPPOrgCheckResult) super.clone();

    return clone;
  }

  /**
   * Implementation of <code>Object.toString</code>, which will result in an
   * indented XML <code>String</code> representation of the concrete
   * <code>EPPCodecComponent</code>.
   * 
   * @return Indented XML <code>String</code> if successful;
   *         <code>ERROR</code> otherwise.
   */
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Compare an instance of <code>EPPOrgCheckResult</code> with this
   * instance.
   * 
   * @param aObject
   *            Object to compare with.
   * 
   * @return <code>true</code> if equal; <code>false</code> otherwise.
   */
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPOrgCheckResult)) {
      return false;
    }

    EPPOrgCheckResult other = (EPPOrgCheckResult) aObject;

    // Org Identifier
    if (!EqualityUtil.equals(this.orgId, other.orgId)) {
      cat.error("EPPOrgCheckResult.equals(): orgId not equal");
      return false;
    }

    // Available
    if (!EqualityUtil.equals(this.available, other.available)) {
      cat.error("EPPOrgCheckResult.equals(): available not equal");
      return false;
    }

    // Reason and ReasonLang
    if (!EqualityUtil.equals(this.reason, other.reason)) {
      cat.error("EPPOrgCheckResult.equals(): reason not equal");
      return false;
    }
    if (!EqualityUtil.equals(this.reasonLang, other.reasonLang)) {
      cat.error("EPPOrgCheckResult.equals(): reasonLang not equal");
      return false;
    }

    return true;
  }

  /**
   * Returns the XML namespace associated with the <code>EPPCodecComponent</code>.
   * 
   * @return XML namespace for the <code>EPPCodecComponent</code>. 
   */
  @Override
  public String getNamespace() {
    return EPPOrgMapFactory.NS;
  }
  
}
