/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.nameWatch;

import java.util.Date;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;


/**
 * Represents an EPP NameWatch &lt;nameWatch:renData&gt; response to a {@code 
 * EPPNameWatchRenewCmd}. When a &lt;renew&gt; command has been
 * processed successfully, the EPP &lt;resData&gt; element MUST contain a
 * child &lt;nameWatch:renData&gt; element that identifies the nameWatch
 * namespace and the location of the nameWatch schema.  The
 * &lt;nameWatch:name&gt; element contains the following child
 * elements:<br><br>
 * 
 * <ul>
 * <li>
 * A &lt;nameWatch:roid&gt; element that contains the fully qualified nameWatch
 * roid that has been created or whose validity period has been extended. Use
 * {@code getRoid} and {@code setRoid}     to get and set the
 * element.
 * </li>
 * <li>
 * An OPTIONAL &lt;nameWatch:exDate&gt; element that contains the end of the
 * nameWatch's validity period.  Use {@code getExpirationDate} and
 * {@code setExpirationDate} to get and set the element.
 * </li>
 * </ul>
 * 
 * <br><br>
 *
 * @author $Author: jim $
 * @version $Revision: 1.1.1.1 $
 *
 * @see com.verisign.epp.codec.nameWatch.EPPNameWatchRenewCmd
 */
public class EPPNameWatchRenewResp extends EPPResponse {
  /** XML Element Name of {@code EPPNameWatchRenewResp} root element. */
  final static String ELM_NAME = "nameWatch:renData";

  /** XML Element Name for the {@code name} attribute. */
  private final static String ELM_ROID = "nameWatch:roid";

  /** XML Element Name for the {@code expirationDate} attribute. */
  private final static String ELM_EXPIRATION_DATE = "nameWatch:exDate";

  /** NameWatch Roid of nameWatch to create. */
  private String roid = null;

  /** the end of the nameWatch's validity period. */
  private Date expirationDate = null;

  /**
   * {@code EPPNameWatchRenewResp} default constructor.  Must call
   * required setter methods before encode.     the defaults include the
   * following:     <br><br>
   * 
   * <ul>
   * <li>
   * name is set to {@code null}
   * </li>
   * <li>
   * expiration date is set to {@code null}
   * </li>
   * </ul>
   * 
   * <br>     The name must be set before invoking {@code encode}.
   */
  public EPPNameWatchRenewResp() {
  }

  // End EPPNameWatchRenewResp()

  /**
   * {@code EPPNameWatchRenewResp} constructor that takes the required
   * attribute values as parameters.
   *
   * @param aTransId Transaction Id associated with response.
   * @param aRoid NameWatch roid
   */
  public EPPNameWatchRenewResp(EPPTransId aTransId, String aRoid) {
    super(aTransId);

    roid = aRoid;
  }

  // End EPPNameWatchRenewResp.EPPNameWatchRenewResp(EPPTransId, String)

  /**
   * {@code EPPNameWatchRenewResp} constructor that takes the required
   * attribute values as parameters.
   *
   * @param aTransId transaction Id associated with response.
   * @param aRoid nameWatch roid
   * @param aExpirationDate expiration date of the nameWatch
   */
  public EPPNameWatchRenewResp(
                 EPPTransId aTransId, String aRoid,
                 Date aExpirationDate) {
    super(aTransId);

    roid         = aRoid;
    expirationDate     = aExpirationDate;
  }

  // End EPPNameWatchRenewResp.EPPNameWatchRenewResp(EPPTransId, String, String, Date)

  /**
   * Gets the EPP command type associated with
   * {@code EPPNameWatchRenewResp}.
   *
   * @return EPPNameWatchRenewResp.ELM_NAME
   */
  public String getType() {
    return ELM_NAME;
  }

  // End EPPNameWatchRenewResp.getType()

  /**
   * Gets the EPP command Namespace associated with
   * {@code EPPNameWatchRenewResp}.
   *
   * @return {@code EPPNameWatchMapFactory.NS}
   */
  public String getNamespace() {
    return EPPNameWatchMapFactory.NS;
  }

  // End EPPNameWatchRenewResp.getNamespace()

  /**
   * Compare an instance of {@code EPPNameWatchRenewResp} with this
   * instance.
   *
   * @param aObject Object to compare with.
   *
   * @return DOCUMENT ME!
   */
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPNameWatchRenewResp)) {
      return false;
    }

    if (!super.equals(aObject)) {
      return false;
    }

    EPPNameWatchRenewResp theComp = (EPPNameWatchRenewResp) aObject;

    // roid
    if (
      !(
          (roid == null) ? (theComp.roid == null)
                     : roid.equals(theComp.roid)
        )) {
      return false;
    }

    // Expiration Date
    if (
      !(
          (expirationDate == null) ? (
                           theComp.expirationDate == null
                         )
                         : expirationDate.equals(theComp.expirationDate)
        )) {
      return false;
    }

    return true;
  }

  // End EPPNameWatchRenewResp.equals(Object)

  /**
   * Clone {@code EPPNameWatchRenewResp}.
   *
   * @return clone of {@code EPPNameWatchRenewResp}
   *
   * @exception CloneNotSupportedException standard Object.clone exception
   */
  public Object clone() throws CloneNotSupportedException {
    EPPNameWatchRenewResp clone = (EPPNameWatchRenewResp) super.clone();

    return clone;
  }

  // End EPPNameWatchRenewResp.clone()

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful;
   *       {@code ERROR} otherwise.
   */
  public String toString() {
    return EPPUtil.toString(this);
  }

  // End EPPNameWatchRenewResp.toString()

  /**
   * Gets the nameWatch roid
   *
   * @return NameWatch Roid {@code String} instance if defined;
   *       {@code null} otherwise.
   */
  public String getRoid() {
    return roid;
  }

  // End EPPNameWatchRenewResp.getRoid()

  /**
   * Sets the nameWatch roid.
   *
   * @param aRoid NameWatch Roid
   */
  public void setRoid(String aRoid) {
    roid = aRoid;
  }

  // End EPPNameWatchRenewResp.setRoid(String)

  /**
   * Gets the expiration date and time of the nameWatch.
   *
   * @return Expiration date and time of the nameWatch if defined;
   *       {@code null} otherwise.
   */
  public Date getExpirationDate() {
    return expirationDate;
  }

  // End EPPNameWatchRenewResp.getExpirationDate()

  /**
   * Sets the expiration date and time of the nameWatch.
   *
   * @param aExpirationDate Expiration date and time of the nameWatch.
   */
  public void setExpirationDate(Date aExpirationDate) {
    expirationDate = aExpirationDate;
  }

  // End EPPNameWatchRenewResp.setExpirationDate(Date)

  /**
   * Validate the state of the {@code EPPNameWatchRenewResp} instance. A
   * valid state means that all of the required attributes have been set. If
   * validateState     returns without an exception, the state is valid. If
   * the state is not     valid, the EPPCodecException will contain a
   * description of the error.     throws     EPPCodecException    State
   * error.  This will contain the name of the     attribute that is not
   * valid.
   *
   * @throws EPPCodecException DOCUMENT ME!
   */
  void validateState() throws EPPCodecException {
    if (roid == null) {
      throw new EPPCodecException("roid required attribute is not set");
    }

    if (expirationDate == null) {
      throw new EPPCodecException("expirationDate required attribute is not set");
    }
  }

  // End EPPNameWatchRenewResp.validateState()

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPNameWatchRenewResp} instance.
   *
   * @param aDocument DOM Document that is being built.  Used as an Element
   *      factory.
   *
   * @return Element    Root DOM Element representing the
   *       {@code EPPNameWatchRenewResp} instance.
   *
   * @exception EPPEncodeException Unable to encode
   *        {@code EPPNameWatchRenewResp} instance.
   */
  protected Element doEncode(Document aDocument) throws EPPEncodeException {
    // Validate Attributes
    try {
      validateState();
    }
     catch (EPPCodecException e) {
      throw new EPPEncodeException("Invalid state on EPPNameWatchRenewResp.encode: "
                     + e);
    }

    Element root =
      aDocument.createElementNS(EPPNameWatchMapFactory.NS, ELM_NAME);

    root.setAttribute("xmlns:nameWatch", EPPNameWatchMapFactory.NS);

    // Roid
    EPPUtil.encodeString(
               aDocument, root, roid, EPPNameWatchMapFactory.NS,
               ELM_ROID);

    // Expiration Date
    EPPUtil.encodeTimeInstant(
                  aDocument, root, expirationDate,
                  EPPNameWatchMapFactory.NS, ELM_EXPIRATION_DATE);

    return root;
  }

  // End EPPNameWatchRenewResp.doEncode(Document)

  /**
   * Decode the {@code EPPNameWatchRenewResp} attributes from the
   * {@code aElement} DOM Element tree.
   *
   * @param aElement Root DOM Element to decode
   *      {@code EPPNameWatchRenewResp} from.
   *
   * @exception EPPDecodeException Unable to decode {@code aElement}
   */
  protected void doDecode(Element aElement) throws EPPDecodeException {
    // Roid
    roid =
      EPPUtil.decodeString(aElement, EPPNameWatchMapFactory.NS, ELM_ROID);

    // Expiration Date
    expirationDate =
      EPPUtil.decodeTimeInstant(
                    aElement, EPPNameWatchMapFactory.NS,
                    ELM_EXPIRATION_DATE);

    // Validate Attributes
    try {
      validateState();
    }
     catch (EPPCodecException e) {
      throw new EPPDecodeException("Invalid state on EPPNameWatchRenewResp.decode: "
                     + e);
    }
  }

  // End EPPNameWatchRenewResp.doDecode(Element)
}
