/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.launchpolicy.v01;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.registry.policy.EPPRegistryZoneInterface;
import com.verisign.epp.framework.EPPRegistryPolicyAdapter;


/**
 * {@code EPPLaunchPolicyAdapter} is used to adapt command EPP extensions (
 * {@link EPPLaunchPolicyCreate} and {@link EPPLaunchPolicyUpdate}) to its
 * contained {@link EPPLaunchPolicyZone}, and adapt a
 * {@link EPPLaunchPolicyZone} to a info response EPP extension (
 * {@link EPPLaunchPolicyInfData}).
 */
public class EPPLaunchPolicyAdapter extends EPPRegistryPolicyAdapter {

      private static Logger cat = LoggerFactory.getLogger(EPPLaunchPolicyAdapter.class);
        

  /**
   * Adapt a list of EPP extensions used with a registry create or update to a
   * matching launch phase policy extension.
   * 
   * @param aEppExtensions
   *           List of EPP extensions to adapt to a zone policy extension.
   * 
   * @return Matching launch phase policy extension if found; {@code null}
   *         otherwise.
   */
  @Override
  public EPPRegistryZoneInterface eppExtToZoneExt(List<EPPCodecComponent> aEppExtensions) {
    cat.debug("eppExtToZoneExt: enter");

    for (EPPCodecComponent theExt : aEppExtensions) {
      if (theExt instanceof EPPLaunchPolicyCreate) {
        cat.debug("eppExtToZoneExt: found " + EPPLaunchPolicyCreate.class.getName());
        return ((EPPLaunchPolicyCreate) theExt).getZone();
      }
      else if (theExt instanceof EPPLaunchPolicyUpdate) {
        cat.debug("eppExtToZoneExt: found " + EPPLaunchPolicyUpdate.class.getName());
        return ((EPPLaunchPolicyUpdate) theExt).getZone();
      }
    }

    cat.debug("eppExtToZoneExt: exit (not found)");
    return null;
  }

  /**
   * Adapt a list of policy extensions available for the zone to a matching EPP
   * extension for an info response.
   * 
   * @param aZoneExtensions
   *           List of zone policy extensions to adapt to an EPP extension for
   *           an info response.
   * @return Matching EPP extension for an info response if found; {@code null}
   *         otherwise.
   */
  @Override
  public EPPCodecComponent zoneExtToEppExt(List<EPPRegistryZoneInterface> aZoneExtensions) {
    cat.debug("zoneExtToEppExt: enter");

    for (EPPRegistryZoneInterface theExt : aZoneExtensions) {
      if (theExt instanceof EPPLaunchPolicyZone) {
        cat.debug("zoneExtToEppExt: found " + EPPLaunchPolicyZone.class.getName());
        return new EPPLaunchPolicyInfData((EPPLaunchPolicyZone) theExt);
      }
    }

    cat.debug("zoneExtToEppExt: exit (not found)");
    return null;
  }

}
