/***********************************************************
Copyright (C) 2024 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.serverstub;

import java.io.File;
import java.util.Enumeration;
import java.util.Vector;

import org.apache.catalina.Context;
import org.apache.catalina.connector.Connector;
import org.apache.catalina.startup.Tomcat;
import org.apache.coyote.http2.Http2Protocol;
import org.apache.tomcat.util.net.SSLHostConfig;
import org.apache.tomcat.util.net.SSLHostConfigCertificate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPFactory;
import com.verisign.epp.framework.EPPByteArrayDispatcher;
import com.verisign.epp.framework.EPPPollHandler;
import com.verisign.epp.framework.EPPPollQueueMgr;
import com.verisign.epp.framework.HttpEPPXMLAssembler;
import com.verisign.epp.transport.EPPSrvFactorySingle;
import com.verisign.epp.util.EPPEnv;
import com.verisign.epp.util.EPPEnvException;
import com.verisign.epp.util.EPPEnvSingle;

/**
 * EPP over HTTP (EoH) Stub Server main class that uses the embedded Tomcat
 * server.
 */
public class HttpServer {
	/** Category for logging */
	private static Logger cat = LoggerFactory.getLogger(HttpServer.class);

	/**
	 * Initialize using the EPPEnv class and the epp.config file
	 * 
	 * @param aConfigFile
	 *           EPP configuration file to use to initialize the EPP SDK
	 */
	public HttpServer(String aConfigFile) {
		try {
			// Initialize environment settings
			EPPEnvSingle env = EPPEnvSingle.getInstance();
			env.initialize(aConfigFile);

			EPPSrvFactorySingle.getInstance();

			// Initialize the dispatcher
			initializeDispatcher();

			// Initialize the data source (in memory queue)
			EPPPollQueueMgr.getInstance().setDataSource(new PollDataSource());

			// Initialize the poll queue
			initializePollQueue();
		}
		catch (EPPEnvException e) {
			e.printStackTrace();
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * Initialize the poll handler based on the EPP.PollHandlers,
	 */
	public void initializePollQueue() {
		cat.debug("initializePollQueue: enter");
		EPPPollQueueMgr thePollQueue = EPPPollQueueMgr.getInstance();

		try {
			Vector<String> handlerClasses = EPPEnv.getPollHandlers();

			if (handlerClasses != null) {
				for (String handlerClassName : handlerClasses) {
					cat.debug("initializePollQueue: Loading poll handler " + handlerClassName);					
					Class handlerClass = Class.forName(handlerClassName);

					// register poll handler
					thePollQueue.register((EPPPollHandler) handlerClass.getDeclaredConstructor().newInstance());

					cat.info("initializePollQueue: Successfully loaded poll handler " + handlerClass.getName());					
				}
			}
		}
		catch (Exception e) {
			cat.error("EoR Server: Exception initalizing the poll queue:" + e);
			e.printStackTrace();
			System.exit(1);
		}
		cat.debug("initializePollQueue: exit");
	}

	/**
	 * Initializes <code>Server</code>'s dispatcher to handle the commmands
	 * defined by the supported EPP mappings. At this Point we also get the list
	 * of Command Response extensions class names from the epp.config file and
	 * add the CommandResponseextension classes to the EPPExtFaactory
	 */
	public void initializeDispatcher() {
		/**
		 * Register the EPP Event Handlers with the Dispatcher. The fully
		 * qualified class names of the event handlers should be listed in the
		 * conf file. EPPEnv had already read them into a Vector that we can get.
		 */
		EPPByteArrayDispatcher theDispatcher = EPPByteArrayDispatcher.getInstance();

		try {
			Vector<String> handlerClasses = EPPEnv.getServerEventHandlers();
			Enumeration<String> e = handlerClasses.elements();

			while (e.hasMoreElements()) {
				String handlerClassName = e.nextElement();
				Class handlerClass = Class.forName(handlerClassName);
				theDispatcher.registerHandler(
				      (com.verisign.epp.framework.EPPEventHandler) handlerClass.getDeclaredConstructor().newInstance());

				cat.info("Successfully loaded server handler: " + handlerClass.getName());
			}

			/**
			 * Now add the CommandResponse level extensions to the extension
			 * factories provided if any of the commandresponselevel extensions
			 * exist. This can be done by adding the commandresponselevel
			 * extensions to the EPPFactories method addExtFactory
			 */
			EPPFactory factory = EPPFactory.getInstance();
			Vector<String> commandExts = EPPEnv.getCmdResponseExtensions();

			if ((commandExts != null) && commandExts.elements().hasMoreElements()) {
				for (int i = 0; i < commandExts.size(); i++) {
					String commandExtensionClassName = commandExts.elementAt(i);

					try {
						factory.addExtFactory(commandExtensionClassName);
					}

					catch (EPPCodecException ex) {
						cat.error(
						      "Couldn't load the Extension Factory " + "associated with the CommandResponseExtensions" + ex);
						ex.printStackTrace();
					}

					cat.info("Successfully loaded Command Extension Class:" + commandExtensionClassName);
				}
			}

			theDispatcher.setAssembler(new HttpEPPXMLAssembler());
		}
		catch (Exception e) {
			cat.error("EoH Server: Exception initalizing the dispatcher:" + e);
			e.printStackTrace();
			System.exit(1);
		}
	}

	/**
	 * Runs the EPP over HTTP (EPP) Stub Server
	 */
	private void run() {
		// Initialize the server connection factory
		EPPSrvFactorySingle theFactory = EPPSrvFactorySingle.getInstance();

		cat.info("EoH Server: Starting server...");

		try {
			Tomcat theHttpServer = new Tomcat();
			theHttpServer.setBaseDir(new File(".").getAbsolutePath());

			Connector theConnector = new Connector();

			theConnector.setPort(EPPEnv.getServerPort());
			cat.debug("Server Port = " + EPPEnv.getServerPort());

			// "http" or "https"
			theConnector.setScheme(EPPEnv.getHttpServerScheme());
			cat.debug("Server HTTP Protocol = " + EPPEnv.getHttpServerScheme());

			// Initialize HTTPS?
			if (EPPEnv.getHttpServerScheme().equalsIgnoreCase("https")) {
				cat.debug("Initializing HTTPS");
				System.setProperty("javax.net.debug", EPPEnv.getSSLDebug());
				theConnector.setProperty("SSLEnabled", "true");
				theConnector.setProperty("clientAuth", "true");
				theConnector.setSecure(true);

				SSLHostConfig theSslHostConfig = new SSLHostConfig();
				theSslHostConfig.setSslProtocol(EPPEnv.getSSLProtocol());
				theSslHostConfig.setEnabledCiphers(EPPEnv.getSSLEnabledCipherSuites());
				theSslHostConfig.setEnabledProtocols(EPPEnv.getSSLEnabledProtocols());
				theSslHostConfig.setTruststoreFile(EPPEnv.getSSLTrustStoreFileName());
				theSslHostConfig.setTruststorePassword(EPPEnv.getSSLTrustStorePassPhrase());
				theSslHostConfig.setTruststoreType(EPPEnv.getKeyStore());
				theSslHostConfig.setCertificateVerification("true");
				theSslHostConfig.setHostName("_default_");

				SSLHostConfigCertificate theSslHostConfigCertificate = new SSLHostConfigCertificate(theSslHostConfig,
				      SSLHostConfigCertificate.Type.UNDEFINED);
				theSslHostConfigCertificate.setCertificateKeystoreFile(EPPEnv.getSSLKeyFileName());
				theSslHostConfigCertificate.setCertificateKeystorePassword(EPPEnv.getSSLPassPhrase());
				theSslHostConfigCertificate.setCertificateKeystoreType(EPPEnv.getKeyStore());
				theSslHostConfigCertificate.setCertificateKeyPassword(EPPEnv.getSSLKeyPassPhrase());
				theSslHostConfig.addCertificate(theSslHostConfigCertificate);

				theConnector.addSslHostConfig(theSslHostConfig);
			}
			else {
				theConnector.setSecure(false);
			}

			// Add support for HTTP/2
			Http2Protocol http2Protocol = new Http2Protocol();
			theConnector.getProtocolHandler().addUpgradeProtocol(http2Protocol);

			// Set the path to empty
			String theContextPath = "";
			String theDocBase = new File(".").getAbsolutePath();

			Context theContext = theHttpServer.addContext(theContextPath, theDocBase);

			theHttpServer.addServlet(theContextPath, "EoHControllerServlet", ControllerServlet.class.getName());

			theContext.addServletMappingDecoded("/", "EoHControllerServlet");

			theHttpServer.start();
			theHttpServer.getService().addConnector(theConnector);
			theHttpServer.getServer().await();
		}
		catch (Exception e) {
			cat.error("EoH Server: Run Exception: " + e);
			e.printStackTrace();
			System.exit(1);
		}
	}

	/**
	 * Runs the Server
	 *
	 * @param args
	 *           The command line argument should be the epp.config file
	 */
	public static void main(String[] args) {
		String theConfigFile = "epp.config";

		if (args.length == 1) {

			if (args[0].equalsIgnoreCase("-help")) {
				System.out.println("Usage: java Server [-help|<config-file>]");
				System.out.println("Executes the EPP over HTTP (EoH) Stub Server");
				System.out.println("\t-help gets this help");
				System.out.println("\t<config-file> EPP SDK configuration settings, with default of \"epp.config\"");
				System.exit(0);

			}
			theConfigFile = args[0];
		}
		else if (args.length > 1) {
			System.out.println("Usage: java Server [-help|<config-file>]");
			System.out.println("Executes the EPP over HTTP (EoH) Stub Server");
			System.out.println("\t-help gets this help");
			System.out.println("\t<config-file> EPP SDK configuration settings, with default of \"epp.config\"");
			System.exit(1);
		}

		HttpServer theServer = new HttpServer(theConfigFile);
		theServer.run();
	}

}
