/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.framework;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPFactory;
import com.verisign.epp.codec.gen.EPPMessage;
import com.verisign.epp.codec.host.EPPHostCheckCmd;
import com.verisign.epp.codec.host.EPPHostCreateCmd;
import com.verisign.epp.codec.host.EPPHostDeleteCmd;
import com.verisign.epp.codec.host.EPPHostInfoCmd;
import com.verisign.epp.codec.host.EPPHostMapFactory;
import com.verisign.epp.codec.host.EPPHostUpdateCmd;

/**
 * The {@code EPPHostHandler} class provides an interface for handling Host EPP
 * Commands in a Server implementation. EPPEvents are handled by the
 * handleEvent() method here and routed to the appropriate abstract member
 * function. Subclasses should override the abstract methods to define the
 * desired behavior of a particular command when it is received. A subclassed
 * instance of {@code EPPHostHandler} should be registered with the
 * {@code EPPDispatcher} so that EEPEvents related to the Host Mapping will be
 * handled there.
 *
 * @see EPPEventHandler
 * @see EPPEvent
 */
public abstract class EPPHostHandler implements EPPEventHandler {
  /** The Namespace that this handler supports. In this case, Host. */
  private static final String NS = EPPHostMapFactory.NS;

  /** Category for logging */
  private static Logger cat = LoggerFactory.getLogger(EPPHostHandler.class);

  /**
   * Construct an instance of {@code EPPHostHandler}
   */
  public EPPHostHandler() {
    try {
      EPPFactory.getInstance().addMapFactory(EPPHostMapFactory.class.getName());
    }
    catch (EPPCodecException e) {
      cat.error("Couldn't load the Map Factory associated with the Host Mapping", e);
      System.exit(1);
    }
  }

  /**
   * Returns the Namespace that this handler supports. In this case, host.
   *
   * @return String The Namespace that this handler supports
   */
  public final String getNamespace() {
    return NS;
  }

  /**
   * This method receives an {@code EPPEvent} and routes it to the appropriate
   * abstract method.
   *
   * @param aEvent
   *           An {@code EPPEvent} that contains the {@code EPPCommand}
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPContactHandler} instance
   *
   * @return Response to the event
   *
   * @exception EPPEventException
   *               Thrown if an unrecognized {@code EPPEvent} is received
   */
  public final EPPEventResponse handleEvent(EPPEvent aEvent, Object aData) throws EPPEventException {
    try {
      this.preHandleEvent(aEvent, aData);
    }
    catch (EPPHandleEventException e) {
      return new EPPEventResponse(e.getResponse());
    }

    EPPMessage message = aEvent.getMessage();
    EPPEventResponse response;

    if (message instanceof EPPHostCreateCmd) {
      response = doHostCreate(aEvent, aData);
    }
    else if (message instanceof EPPHostDeleteCmd) {
      response = doHostDelete(aEvent, aData);
    }
    else if (message instanceof EPPHostInfoCmd) {
      response = doHostInfo(aEvent, aData);
    }
    else if (message instanceof EPPHostCheckCmd) {
      response = doHostCheck(aEvent, aData);
    }
    else if (message instanceof EPPHostUpdateCmd) {
      response = doHostUpdate(aEvent, aData);
    }
    else {
      throw new EPPEventException("In EPPHostHandler an event was sent that is not supported");
    }

    try {
      this.postHandleEvent(aEvent, aData);
    }
    catch (EPPHandleEventException e) {
      return new EPPEventResponse(e.getResponse());
    }

    return response;
  }

  /**
   * Handles any common behavior that all host commands need to execute before
   * they execute their command specific behavior. The default implementation
   * does nothing.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPDomainHandler}
   *
   * @exception EPPHandleEventException
   *               Thrown if an error condition occurs. It must contain an
   *               {@code EPPEventResponse}
   */
  protected void preHandleEvent(EPPEvent aEvent, Object aData) throws EPPHandleEventException {
  }

  /**
   * Handles any common behavior that all gen commands need to execute after
   * they execute their command specific behavior. The default implementation
   * does nothing
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPDomainHandler}
   *
   * @exception EPPHandleEventException
   *               Thrown if an error condition occurs. It must contain an
   *               {@code EPPEventResponse}
   */
  protected void postHandleEvent(EPPEvent aEvent, Object aData) throws EPPHandleEventException {
  }

  /**
   * Invoked when a Host Create command is received. Subclasses should define
   * the behavior when a Host Create command is received.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPDomainHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected abstract EPPEventResponse doHostCreate(EPPEvent aEvent, Object aData);

  /**
   * Invoked when a Host Info command is received. Subclasses should define the
   * behavior when a Host Info command is received.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPDomaindHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected abstract EPPEventResponse doHostDelete(EPPEvent aEvent, Object aData);

  /**
   * Invoked when a Host Info command is received. Subclasses should define the
   * behavior when a Host Info command is received.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPDomaindHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected abstract EPPEventResponse doHostInfo(EPPEvent aEvent, Object aData);

  /**
   * Invoked when a Host Check command is received. Subclasses should define
   * the behavior when a Host Check command is received.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPDomaindHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected abstract EPPEventResponse doHostCheck(EPPEvent aEvent, Object aData);

  /**
   * Invoked when a Host Update command is received. Subclasses should define
   * the behavior when a Host Update command is received.
   *
   * @param aEvent
   *           The {@code EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPDomaindHandler}
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected abstract EPPEventResponse doHostUpdate(EPPEvent aEvent, Object aData);
}
