/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.host;

import java.util.HashSet;
import java.util.Set;

import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPCommand;
import com.verisign.epp.codec.gen.EPPMapFactory;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPService;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * {@code EPPHostMapFactory} represents the {@code EPPCommand} and
 * {@code EPPResponse} factory for the EPP Host Mapping with the XML Namespace
 * URI "urn:ietf:params:xmlns:host". The fully qualified class name for
 * {@code EPPHostMapFactory} is included in a call to {@code EPPFactory.init} or
 * {@code EPPCodec.init}.
 *
 * @see com.verisign.epp.codec.gen.EPPCodec
 * @see com.verisign.epp.codec.gen.EPPFactory
 * @see com.verisign.epp.codec.gen.EPPCommand
 * @see com.verisign.epp.codec.gen.EPPResponse
 */
public class EPPHostMapFactory extends EPPMapFactory {
  /** Namespace URI associated with EPPHostMapFactory. */
  public static final String NS = "urn:ietf:params:xml:ns:host-1.0";

  /** Namespace prefix associated with EPPHostMapFactory. */
  public static final String NS_PREFIX = "host";

  /** Service description for {@code EPPHostMapFactory} */
  private EPPService service = null;

  /**
   * Allocates a new {@code EPPHostMapFactory}. The service attribute will be
   * initialized with the XML namespace information defined by the
   * {@code EPPHostMapFactory} constants:<br>
   * 
   * <ul>
   * <li>NS_PREFIX Namespace prefix of {@code EPPHostMapFactory}</li>
   * <li>NS Namespace URI of {@code EPPHostMapFactory}</li>
   * <li>NS_SCHEMA Namespace schema reference of {@code EPPHostMapFactory}</li>
   * </ul>
   */
  public EPPHostMapFactory() {
    service = new EPPService(NS_PREFIX, NS);
  }

  /**
   * creates a concrete {@code EPPCommand} from the passed in XML Element tree.
   * {@code aMapElement} must be the root node for the command extension. For
   * example, &lt;host:create&gt; must be the element passed for a Host Create
   * Command.
   *
   * @param aMapElement
   *           Mapping Extension EPP XML Element.
   *
   * @return Concrete {@code EPPCommand} instance associated with
   *         {@code aMapElement}.
   *
   * @exception EPPCodecException
   *               Error creating concrete {@code EPPCommand}
   */
  public EPPCommand createCommand(Element aMapElement) throws EPPCodecException {
    String name = aMapElement.getLocalName();

    if (!aMapElement.getNamespaceURI().equals(NS)) {
      throw new EPPCodecException("Invalid mapping type " + name);
    }

    if (name.equals(EPPUtil.getLocalName(EPPHostCheckCmd.ELM_NAME))) {
      return new EPPHostCheckCmd();
    }
    else if (name.equals(EPPUtil.getLocalName(EPPHostInfoCmd.ELM_NAME))) {
      return new EPPHostInfoCmd();
    }
    else if (name.equals(EPPUtil.getLocalName(EPPHostCreateCmd.ELM_NAME))) {
      return new EPPHostCreateCmd();
    }
    else if (name.equals(EPPUtil.getLocalName(EPPHostDeleteCmd.ELM_NAME))) {
      return new EPPHostDeleteCmd();
    }
    else if (name.equals(EPPUtil.getLocalName(EPPHostUpdateCmd.ELM_NAME))) {
      return new EPPHostUpdateCmd();
    }
    else {
      throw new EPPCodecException("Invalid command element " + name);
    }
  }

  /**
   * creates a concrete {@code EPPResponse} from the passed in XML Element
   * tree. {@code aMapElement} must be the root node for the command extension.
   * For example, &lt;host:info-data&gt; must be the element passed for a Host
   * Info Response.
   *
   * @param aMapElement
   *           Mapping Extension EPP XML Element.
   *
   * @return Concrete {@code EPPResponse} instance associated with
   *         {@code aMapElement}.
   *
   * @exception EPPCodecException
   *               Error creating concrete {@code EPPResponse}
   */
  public EPPResponse createResponse(Element aMapElement) throws EPPCodecException {
    String name = aMapElement.getLocalName();

    if (!aMapElement.getNamespaceURI().equals(NS)) {
      throw new EPPCodecException("Invalid mapping type " + name);
    }

    if (name.equals(EPPUtil.getLocalName(EPPHostCheckResp.ELM_NAME))) {
      return new EPPHostCheckResp();
    }
    else if (name.equals(EPPUtil.getLocalName(EPPHostInfoResp.ELM_NAME))) {
      return new EPPHostInfoResp();
    }
    else if (name.equals(EPPUtil.getLocalName(EPPHostCreateResp.ELM_NAME))) {
      return new EPPHostCreateResp();
    }
    else if (name.equals(EPPUtil.getLocalName(EPPHostPendActionMsg.ELM_NAME))) {
      return new EPPHostPendActionMsg();
    }
    else {
      throw new EPPCodecException("Invalid response element " + name);
    }
  }

  /**
   * Gets the {@code EPPService} associated with {@code EPPHostMapFactory}. The
   * {@code EPPService} is used by {@code EPPFactory} for distributing the
   * responsibility of creating concrete {@code EPPCommand} and
   * {@code EPPResponse} objects by XML namespace. The XML namespace is defined
   * in the returned {@code EPPService}.
   *
   * @return service description for the Host Command Mapping.
   */
  public EPPService getService() {
    return service;
  }

  /**
   * Gets the list of XML schemas that need to be pre-loaded into the XML
   * Parser.
   *
   * @return {@code Set} of {@code String} XML Schema names that should be
   *         pre-loaded in the XML Parser.
   * 
   * @see com.verisign.epp.codec.gen.EPPMapFactory#getXmlSchemas()
   */
  public Set<String> getXmlSchemas() {
    Set<String> theSchemas = new HashSet<String>();
    theSchemas.add("host-1.0.xsd");
    return theSchemas;
  }

}
