/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.host;

import java.util.Vector;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCheckCmd;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Represents an EPP Host &lt;check&gt; command, which is used to determine if a
 * host name is known to the server. The &lt;host:check&gt; element MUST contain
 * the following child elements:<br>
 * <br>
 * 
 * <ul>
 * <li>One or more &lt;host:name&gt; elements that contain the fully qualified
 * name of the queried hosts. Use {@code getNames} and {@code setNames} to get
 * and set the elements. Use {@code setNames} to set an individual name.</li>
 * </ul>
 * 
 * <br>
 * <br>
 * {@code EPPHostCheckResp} is the concrete {@code EPPReponse} associated with
 * {@code EPPHostCheckCmd}. <br>
 * <br>
 *
 * @see com.verisign.epp.codec.host.EPPHostCheckResp
 */
public class EPPHostCheckCmd extends EPPCheckCmd {
  /**
   * XML local name for {@code EPPHostCheckCmd}.
   */
  public static final String ELM_LOCALNAME = "check";

  /**
   * XML Element Name of {@code EPPHostCheckCmd} root element.
   */
  public static final String ELM_NAME = EPPHostMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /** XML Element Name for the {@code names} attribute. */
  private final static String ELM_HOST_NAME = "name";

  /**
   * Host Names to check. This is a {@code Vector} of {@code String} instances.
   */
  private Vector<String> names;

  /**
   * {@code EPPHostCheckCmd} default constructor. It will set the names
   * attribute to an empty {@code Vector}.
   */
  public EPPHostCheckCmd() {
    names = new Vector<String>();
  }

  /**
   * {@code EPPHostCheckCmd} constructor that will check an individual host
   * name.
   *
   * @param aTransId
   *           Transaction Id associated with command.
   * @param aName
   *           Host name to check
   */
  public EPPHostCheckCmd(String aTransId, String aName) {
    super(aTransId);

    names = new Vector<String>();
    names.addElement(aName);
  }

  /**
   * {@code EPPHostCheckCmd} constructor that will check a list of host names.
   *
   * @param aTransId
   *           Transaction Id associated with command.
   * @param aNames
   *           {@code Vector} of host name {@code String} instances.
   */
  public EPPHostCheckCmd(String aTransId, Vector<String> aNames) {
    super(aTransId);

    names = aNames;
  }

  /**
   * Gets the EPP command Namespace associated with {@code EPPHostCheckCmd}.
   *
   * @return {@code EPPHostMapFactory.NS}
   */
  public String getNamespace() {
    return EPPHostMapFactory.NS;
  }

  /**
   * Gets the key for the host object, which is the host name.
   * 
   * @return The host name if set with a single value; {@code null} otherwise.
   */
  @Override
  public String getKey() {
	  if (this.names != null && this.names.size() == 1) {
		  return (String) this.names.elementAt(0);
	  }
	  else {
		  return null;
	  }
  }
  
  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPHostCheckCmd} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the {@code EPPHostCheckCmd}
   *         instance.
   *
   * @exception EPPEncodeException
   *               Unable to encode {@code EPPHostCheckCmd} instance.
   */
  protected Element doEncode(Document aDocument) throws EPPEncodeException {
    if (names.size() == 0) {
      throw new EPPEncodeException("No host names specified in EPPHostCheckCmd");
    }

    Element root = aDocument.createElementNS(EPPHostMapFactory.NS, ELM_NAME);

    // Names
    EPPUtil.encodeVector(aDocument, root, names, EPPHostMapFactory.NS,
          EPPHostMapFactory.NS_PREFIX + ":" + ELM_HOST_NAME);

    return root;
  }

  /**
   * Decode the {@code EPPHostCheckCmd} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPHostCheckCmd} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  protected void doDecode(Element aElement) throws EPPDecodeException {
    // Host Names
    names = EPPUtil.decodeVector(aElement, EPPHostMapFactory.NS, ELM_HOST_NAME);

    if (names == null) {
      names = new Vector<String>();
    }
  }

  /**
   * Compare an instance of {@code EPPHostCheckCmd} with this instance.
   *
   * @param aObject
   *           Object to compare with.
   *
   * @return {@code true} if equal; {@code false} otherwise
   */
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPHostCheckCmd)) {
      return false;
    }

    if (!super.equals(aObject)) {
      return false;
    }

    EPPHostCheckCmd theMap = (EPPHostCheckCmd) aObject;

    // Host Names
    if (!EPPUtil.equalVectors(names, theMap.names)) {
      return false;
    }

    return true;
  }

  /**
   * Clone {@code EPPHostCheckCmd}.
   *
   * @return clone of {@code EPPHostCheckCmd}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  public Object clone() throws CloneNotSupportedException {
    EPPHostCheckCmd clone = (EPPHostCheckCmd) super.clone();

    clone.names = (Vector<String>) names.clone();

    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Sets host name to check.
   *
   * @param aName
   *           Name to check.
   */
  public void setName(String aName) {
    names = new Vector<String>();
    names.addElement(aName);
  }

  /**
   * Gets host names to check.
   *
   * @return Vector of host name {@code String}'s.
   */
  public Vector<String> getNames() {
    return names;
  }

  /**
   * Sets host names to check.
   *
   * @param aNames
   *           Names to check.
   */
  public void setNames(Vector<String> aNames) {
    names = aNames;
  }

}
