/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.util;

import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * This is a helper class provided to intercept Errors occurred internal to SAX
 * / Dom Parser. When an Error / fatal Error or a warning occurs these methods
 * are invoked.
 */
public class EPPXMLErrorHandler implements ErrorHandler {

	/**
	 * Logger for the sent and received packets. Used for the
	 * {@link EPPSendReceiveLogger#maskString(String)} method in
	 * {@link #getErrorString(SAXParseException)}.
	 */
	private static EPPSendReceiveLogger sendReceiveLogger = EPPEnv.getSendReceiveLogger();

	/**
	 * Receive notification of a recoverable error. This method will encode a
	 * {@link org.xml.sax.SAXException}.
	 *
	 * @param aException
	 *           Error exception information
	 *
	 * @exception SAXException
	 *               Exception associated with the error.
	 */
	public void error(SAXParseException aException) throws SAXException {
		throw new SAXException(getErrorString(aException));
	}

	/**
	 * Receive notification of a fatal error. This method will encode a
	 * {@link org.xml.sax.SAXException}.
	 *
	 * @param aException
	 *           Fatal error exception information
	 *
	 * @exception SAXException
	 *               Exception associated with the fatal error.
	 */
	public void fatalError(SAXParseException aException) throws SAXException {
		throw new SAXException(getErrorString(aException));
	}

	/**
	 * Receive notification of a recoverable warning. This method will encode a
	 * {@link org.xml.sax.SAXException}.
	 *
	 * @param aException
	 *           Warning exception information
	 *
	 * @exception SAXException
	 *               Exception associated with the warning.
	 */
	public void warning(SAXParseException aException) throws SAXException {
		throw new SAXException(getErrorString(aException));
	}

	/**
	 * Returns a more descriptive Error Message.
	 *
	 * @param aException
	 *           {@link SAXParseException} to get error string from
	 *
	 * @return Returns error string associated with the {@link SAXParseException}
	 */
	private String getErrorString(SAXParseException aException) {
		StringBuffer theError = new StringBuffer();

		String systemId = aException.getSystemId();

		if (systemId != null) {
			int index = systemId.lastIndexOf('/');

			if (index != -1) {
				systemId = systemId.substring(index + 1);
			}

			theError.append(systemId);
		}

		theError.append("Line: ");
		theError.append(aException.getLineNumber());
		theError.append(", Column: ");
		theError.append(aException.getColumnNumber());
		theError.append(", Message: ");
		theError.append(sendReceiveLogger.maskString(aException.getMessage()));

		return theError.toString();
	}
}
