/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.transport.server;

import java.io.IOException;
import java.net.ServerSocket;

// Log4j Imports
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.transport.EPPConException;
import com.verisign.epp.transport.EPPServerCon;
import com.verisign.epp.transport.EPPServerThread;
import com.verisign.epp.transport.ServerEventHandler;
import com.verisign.epp.util.EPPEnv;
import com.verisign.epp.util.EPPEnvException;

/**
 * EPP Plain TCP Server
 */
public class EPPPlainServer implements EPPServerCon {
	/** Category for logging */
	private static Logger cat = LoggerFactory.getLogger(EPPPlainServer.class);

	/** Server socket to listen on */
	private ServerSocket serverSocket = null;

	/** Is the server listening on the port? */
	private boolean listening = true;

	/** Server event handler to call for new client connections */
	private ServerEventHandler serverEventHandler = null;

	/** TCP port to listen on */
	private int portNumber = 0;

	/** Connection timeout in milliseconds */
	private int conTimeout = 0;

	/**
	 * Pre-condition the util.Env provide the properties
	 *
	 * @throws EPPConException
	 *            Exception getting the server port or connection timeout
	 */
	public EPPPlainServer() throws EPPConException {
		cat.debug("EPPPlainServer.EPPPlainServer(): entering Constructor");

		try {
			this.portNumber = EPPEnv.getServerPort();
			this.conTimeout = EPPEnv.getConTimeOut();
		}
		catch (EPPEnvException myException) {
			cat.error("Connection Failed Due to : " + myException.getMessage());
			throw new EPPConException("Connection Failed Due to : " + myException.getMessage());
		}

		cat.debug("EPPPlainServer.EPPPlainServer(): ServerPort = " + this.portNumber);

		cat.debug("EPPPlainServer.EPPPlainServer(): Exiting Constructor");
	}

	/**
	 * Runs the Plain TCP Server, which starts the server socket, and enters the
	 * client connection loop.
	 *
	 * @param aServerEventHandler
	 *           Server event handler to get called for new client connections
	 *
	 * @throws EPPConException
	 *            Exception for the server
	 */
	@Override
	public void RunServer(ServerEventHandler aServerEventHandler) throws EPPConException {
		cat.debug("EPPPlainServer.RunServer(): Entering Method");

		this.serverEventHandler = aServerEventHandler;

		try {
			this.serverSocket = new ServerSocket(this.portNumber);

			// myServerSocket.setSoTimeout(myConTimeout);
		}
		catch (IOException myException) {
			cat.error("IO Exception : " + myException.getMessage());
			throw new EPPConException("IO Exception : " + myException.getMessage());
		}

		loop();
		close();

		cat.debug("EPPPlainServer.RunServer(): Exiting Method");
	}

	/**
	 * Client connection loop for the EPP Plain TCP Server.
	 *
	 * @throws EPPConException
	 *            Exception in the EPP Plain TCP Server loop
	 */
	public void loop() throws EPPConException {
		cat.debug("EPPPlainServer.loop(): Entering Method");

		try {
			while (this.listening) {
				EPPServerThread p = new EPPServerThread(this.serverSocket.accept(), this.serverEventHandler);
				p.start();
			}
		}
		catch (IOException myException) {
			cat.error("I/O Error occured when wating for connection");
			throw new EPPConException("I/O Error occured when wating for connection");
		}
		catch (SecurityException myException) {
			cat.error("security Manger exists and its checkListen method doesn't allow accpet operation");
			throw new EPPConException("security Manger exists and its checkListen method doesn't allow accpet operation");
		}

		cat.debug("EPPPlainServer.loop(): Exiting Method");
	}

	/**
	 * Close the server by closing the server socket.
	 *
	 * @throws EPPConException
	 *            Exception closing the server
	 */
	public void close() throws EPPConException {
		cat.debug("EPPPlainServer.close(): Entering Method");

		try {
			this.serverSocket.close();
		}
		catch (Exception myException) {
			cat.error("Close on Server socket Failed");
			throw new EPPConException("Close on Server socket Failed");
		}

		cat.debug("EPPPlainServer.close(): Exiting Method");
	}
}
