/***********************************************************
Copyright (C) 2015 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.serverstub;

// Logging Imports
import java.util.Enumeration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.domain.EPPDomainCheckCmd;
import com.verisign.epp.codec.domain.EPPDomainCreateCmd;
import com.verisign.epp.codec.domain.EPPDomainDeleteCmd;
import com.verisign.epp.codec.domain.EPPDomainInfoCmd;
import com.verisign.epp.codec.domain.EPPDomainRenewCmd;
import com.verisign.epp.codec.domain.EPPDomainTransferCmd;
import com.verisign.epp.codec.domain.EPPDomainUpdateCmd;
import com.verisign.epp.codec.gen.EPPCommand;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPService;
import com.verisign.epp.framework.EPPEvent;
import com.verisign.epp.framework.EPPEventResponse;
import com.verisign.epp.serverstub.v1_0.FeeV1_0DomainHandler;

/**
 * The <code>FeeDomainHandler</code> class is a concrete implementation of the
 * abstract <code>com.verisign.epp.framework.EPPDomainHandler</code> class. It
 * handles accepting commands in a server that supports more than one version of
 * the fee extension. The version specific <code>FeeSubDomainHandler</code> is
 * called based on the fee extension version passed or the set of login services
 * provided by the client.
 *
 * @see com.verisign.epp.framework.EPPEvent
 * @see com.verisign.epp.framework.EPPEventResponse
 */
public class FeeDomainHandler extends DomainHandler {

  /** Used for logging */
  private static Logger cat = LoggerFactory.getLogger(FeeDomainHandler.class);

  // Sub-handler for handling version 1.0 extension messages.
  FeeV1_0DomainHandler v1_0SubHandler = new FeeV1_0DomainHandler();

  /**
   * Invoked when a Domain Check command is received.
   *
   * @param aEvent
   *           The <code>EPPEvent</code> that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           <code>EPPDomaindHandler</code>
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doDomainCheck(EPPEvent aEvent, Object aData) {
    EPPDomainCheckCmd theCommand = (EPPDomainCheckCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doDomainCheck(aEvent, aData);

    // v1_0 of fee check?
    if (theCommand.hasExtension(com.verisign.epp.codec.fee.v1_0.EPPFeeCheck.class)) {
      this.v1_0SubHandler.doDomainCheck(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);
    }
    else {
      cat.info("doDomainCheck: fee check = null");
      System.out.println("doDomainCheck: fee check = null");
    }

    return theEventResponse;
  }

  /**
   * Invoked when a Domain Info command is received.
   *
   * @param aEvent
   *           The <code>EPPEvent</code> that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           <code>EPPDomaindHandler</code>
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doDomainInfo(EPPEvent aEvent, Object aData) {
    EPPDomainInfoCmd theCommand = (EPPDomainInfoCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doDomainInfo(aEvent, aData);

    return theEventResponse;
  }

  /**
   * Invoked when a Domain Create command is received.
   *
   * @param aEvent
   *           The <code>EPPEvent</code> that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           <code>EPPDomaindHandler</code>
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doDomainCreate(EPPEvent aEvent, Object aData) {
    EPPDomainCreateCmd theCommand = (EPPDomainCreateCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doDomainCreate(aEvent, aData);

    // v1_0 of fee create?
    if (theCommand.hasExtension(com.verisign.epp.codec.fee.v1_0.EPPFeeCreate.class)) {
      this.v1_0SubHandler.doDomainCreate(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);
    }
    else {
      cat.info("doDomainCreate: fee create = null");
      System.out.println("doDomainCreate: fee create = null");
    }

    return theEventResponse;
  }

  /**
   * Invoked when a Domain Delete command is received.
   *
   * @param aEvent
   *           The <code>EPPEvent</code> that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           <code>EPPDomaindHandler</code>
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doDomainDelete(EPPEvent aEvent, Object aData) {
    EPPDomainDeleteCmd theCommand = (EPPDomainDeleteCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doDomainDelete(aEvent, aData);

    // Determine what version of the fee extension is supported
    SessionData theSessionData = (SessionData) aData;

    boolean hasV09ExtService = false;
    boolean hasV1_0ExtService = false;

    Enumeration extSvcEnum = theSessionData.getLoginCmd().getExtensionServices().elements();
    while (extSvcEnum.hasMoreElements()) {
      EPPService theExtService = (EPPService) extSvcEnum.nextElement();

      if (theExtService.getNamespaceURI().equals(com.verisign.epp.codec.fee.v1_0.EPPFeeExtFactory.NS)) {
        hasV1_0ExtService = true;
      }
    }

    /*
     * Pick all handlers that apply. This is not compliant with section 2
     * "Migrating to Newer Versions of This Extension", but is needed to
     * support isolated client tests that expect a specific extension version
     * in the response.
     */

    // v1_0 of fee delete?
    if (hasV1_0ExtService) {
      this.v1_0SubHandler.doDomainDelete(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);
    }

    if (!hasV1_0ExtService) {
      cat.info("doDomainDelete: fee extension not supported by client");
      System.out.println("doDomainDelete: fee extension not supported by client");
    }

    return theEventResponse;
  }

  /**
   * Invoked when a Domain Renew command is received.
   *
   * @param aEvent
   *           The <code>EPPEvent</code> that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           <code>EPPDomaindHandler</code>
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doDomainRenew(EPPEvent aEvent, Object aData) {
    EPPDomainRenewCmd theCommand = (EPPDomainRenewCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doDomainRenew(aEvent, aData);

    // v1_0 of fee renew?
    if (theCommand.hasExtension(com.verisign.epp.codec.fee.v1_0.EPPFeeRenew.class)) {
      this.v1_0SubHandler.doDomainRenew(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);
    }
    else {
      cat.info("doDomainRenew: fee renew = null");
      System.out.println("doDomainRenew: fee renew = null");
    }

    return theEventResponse;
  }

  /**
   * Invoked when a Domain Transfer command is received.
   *
   * @param aEvent
   *           The <code>EPPEvent</code> that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           <code>EPPDomaindHandler</code>
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doDomainTransfer(EPPEvent aEvent, Object aData) {
    EPPDomainTransferCmd theCommand = (EPPDomainTransferCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doDomainTransfer(aEvent, aData);

    // Determine what version of the fee extension is supported
    SessionData theSessionData = (SessionData) aData;

    boolean hasV09ExtService = false;
    boolean hasV1_0ExtService = false;

    Enumeration extSvcEnum = theSessionData.getLoginCmd().getExtensionServices().elements();
    while (extSvcEnum.hasMoreElements()) {
      EPPService theExtService = (EPPService) extSvcEnum.nextElement();

      if (theExtService.getNamespaceURI().equals(com.verisign.epp.codec.fee.v1_0.EPPFeeExtFactory.NS)) {
        hasV1_0ExtService = true;
      }
    }

    // Transfer Query?
    if (theCommand.getOp().equals(EPPCommand.OP_QUERY)) {

      /*
       * Pick all handlers that apply. This is not compliant with section 2
       * "Migrating to Newer Versions of This Extension", but is needed to
       * support isolated client tests that expect a specific extension
       * version in the response.
       */

      // v1_0 of transfer query?
      if (hasV1_0ExtService) {
        this.v1_0SubHandler.doDomainTransfer(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);
      }

      if (!!hasV1_0ExtService) {
        cat.info("doDomainTransfer: fee extension not supported by client");
        System.out.println("doDomainTransfer: fee extension not supported by client");
      }

    }
    else if (theCommand.getOp().equals(EPPCommand.OP_REQUEST)) {

      // v1_0 of fee transfer request?
      if (theCommand.hasExtension(com.verisign.epp.codec.fee.v1_0.EPPFeeTransfer.class)) {
        this.v1_0SubHandler.doDomainTransfer(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);
      }
      else {
        cat.info("doDomainTransfer: fee transfer request = null");
        System.out.println("doDomainTransfer: fee transfer request = null");
      }
    }

    return theEventResponse;
  }

  /**
   * Invoked when a Domain Update command is received.
   *
   * @param aEvent
   *           The <code>EPPEvent</code> that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           <code>EPPDomaindHandler</code>
   *
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected EPPEventResponse doDomainUpdate(EPPEvent aEvent, Object aData) {
    EPPDomainUpdateCmd theCommand = (EPPDomainUpdateCmd) aEvent.getMessage();

    EPPEventResponse theEventResponse = super.doDomainUpdate(aEvent, aData);

    // v1_0 of fee update?
    if (theCommand.hasExtension(com.verisign.epp.codec.fee.v1_0.EPPFeeUpdate.class)) {
      this.v1_0SubHandler.doDomainUpdate(theCommand, (EPPResponse) theEventResponse.getResponse(), aData);
    }
    else {
      cat.info("doDomainUpdate: fee update = null");
      System.out.println("doDomainUpdate: fee update = null");
    }

    return theEventResponse;
  }

}
