/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.domain;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Represents a domain Period. A domain name object MAY have a specified
 * validity period. If server policy supports domain object validity periods,
 * the validity period is defined when a domain object is created, and it MAY be
 * extended by the EPP &lt;renew&gt; or &lt;transfer&gt; commands. As a matter
 * of server policy, this specification does not define actions to be taken upon
 * expiration of a domain object's validity period. <br>
 * <br>
 * Validity periods are measured in years or months with the appropriate units
 * specified using the {@code unit} attribute. Valid values for the {@code unit}
 * attribute are {@code y} for years and {@code m} for months.
 */
public class EPPDomainPeriod implements com.verisign.epp.codec.gen.EPPCodecComponent {

  /**
   * XML local name for {@code EPPDomainPeriod}.
   */
  public static final String ELM_LOCALNAME = "period";

  /**
   * XML Element Name of {@code EPPDomainPeriod} root element.
   */
  public static final String ELM_NAME = EPPDomainMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * Period in Unit Month
   */
  public final static String PERIOD_UNIT_MONTH = "m";

  /**
   * Period in Unit Year
   */
  public final static String PERIOD_UNIT_YEAR = "y";

  /**
   * Unspecified Period
   */
  final static int UNSPEC_PERIOD = -1;

  /**
   * XML attribute name for the {@code period} element.
   */
  private final static String ATTR_PERIOD_UNIT = "unit";

  /**
   * Maximum number of periods.
   */
  private final static int MAX_PERIOD = 99;

  /**
   * Minimum number of periods.
   */
  private final static int MIN_PERIOD = 1;

  /**
   * Domain period.
   */
  private int period = 0;

  /**
   * Domain period unit.
   */
  private java.lang.String pUnit = "y";

  /**
   * {@code EPPDomainPeriod} default constructor. The period is initialized to
   * {@code unspecified}. The period must be set before invoking
   * {@code encode}.
   */
  public EPPDomainPeriod() {
    this.period = UNSPEC_PERIOD;
  }

  /**
   * {@code EPPDomainPeriod} constructor that takes the domain period (in unit
   * of year) as an argument
   *
   * @param aPeriod
   *           int
   */
  public EPPDomainPeriod(int aPeriod) {
    this.period = aPeriod;
    this.pUnit = "y";
  }

  /**
   * {@code EPPDomainPeriod} constructor that takes the domain period and
   * period unit as an arguments
   *
   * @param aPUnit
   *           String
   * @param aPeriod
   *           int
   */
  public EPPDomainPeriod(String aPUnit, int aPeriod) {
    this.pUnit = aPUnit;

    if (!this.pUnit.equals(PERIOD_UNIT_YEAR) && !this.pUnit.equals(PERIOD_UNIT_MONTH)) {
      this.pUnit = PERIOD_UNIT_YEAR;
    }

    this.period = aPeriod;
  }

  /**
   * Clone {@code EPPDomainPeriod}.
   *
   * @return clone of {@code EPPDomainPeriod}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPDomainPeriod clone = null;

    clone = (EPPDomainPeriod) super.clone();

    return clone;
  }

  /**
   * Decode the EPPDomainPeriod attributes from the aElement DOM Element tree.
   *
   * @param aElement
   *           - Root DOM Element to decode EPPDomainPeriod from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  public void decode(Element aElement) throws EPPDecodeException {
    // Period
    String tempVal = null;
    tempVal = aElement.getFirstChild().getNodeValue();
    this.pUnit = aElement.getAttribute(ATTR_PERIOD_UNIT);

    if (tempVal == null) {
      this.period = UNSPEC_PERIOD;
    }
    else {
      this.period = Integer.parseInt(tempVal);
    }
  }

  /**
   * Encode a DOM Element tree from the attributes of the EPPDomainPeriod
   * instance.
   *
   * @param aDocument
   *           - DOM Document that is being built. Used as an Element factory.
   *
   * @return Element - Root DOM Element representing the EPPDomainPeriod
   *         instance.
   *
   * @exception EPPEncodeException
   *               - Unable to encode EPPDomainPeriod instance.
   */
  @Override
  public Element encode(Document aDocument) throws EPPEncodeException {
    // Period with Attribute of Unit
    Element root = aDocument.createElementNS(EPPDomainMapFactory.NS, ELM_NAME);

    // Check the Period unit
    if (this.pUnit == null) {
      throw new EPPEncodeException("EPPDomainPeriod: Period Unit should not be null");
    }
    else {
      // if (!pUnit.equals(PERIOD_UNIT_YEAR) &&
      // !pUnit.equals(PERIOD_UNIT_MONTH))
      // throw new EPPEncodeException("EPPDomainPeriod: Period Unit has an
      // invalid value");
      // add attribute here
      root.setAttribute(ATTR_PERIOD_UNIT, this.pUnit);

      // add value
      Text currVal = aDocument.createTextNode(this.period + "");

      // append child
      root.appendChild(currVal);
    }

    return root;
  }

  /**
   * implements a deep {@code EPPDomainPeriod} compare.
   *
   * @param aObject
   *           {@code EPPDomainPeriod} instance to compare with
   *
   * @return {@code true} if equal; {@code false} otherwise.
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPDomainPeriod)) {
      return false;
    }

    EPPDomainPeriod theComp = (EPPDomainPeriod) aObject;

    // period
    if (this.period != theComp.period) {
      return false;
    }

    // pUnit
    if (!((this.pUnit == null) ? (theComp.pUnit == null) : this.pUnit.equals(theComp.pUnit))) {
      return false;
    }

    return true;
  }

  /**
   * Get domain period.
   *
   * @return int
   */
  public int getPeriod() {
    return this.period;
  }

  /**
   * Get domain period unit.
   *
   * @return String
   */
  public String getPUnit() {
    return this.pUnit;
  }

  /**
   * Test whether the period has been specfied: {@code true} is unspecified and
   * {@code false} is specified.
   *
   * @return boolean
   */
  public boolean isPeriodUnspec() {
    if (this.period == UNSPEC_PERIOD) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Set domain period.
   *
   * @param newPeriod
   *           int
   *
   * @throws EPPCodecException
   *            DOCUMENT ME!
   */
  public void setPeriod(int newPeriod) throws EPPCodecException {

    if (newPeriod != UNSPEC_PERIOD) {
      if ((newPeriod < MIN_PERIOD) && (newPeriod > MAX_PERIOD)) {
        throw new EPPCodecException(
              "period of " + newPeriod + " is out of range, must be between " + MIN_PERIOD + " and " + MAX_PERIOD);
      }
    }

    this.period = newPeriod;

  }

  /**
   * Set domain period of un. Creation date: (5/30/01 11:36:52 AM)
   *
   * @param newPUnit
   *           java.lang.String
   */
  public void setPUnit(java.lang.String newPUnit) {
    this.pUnit = newPUnit;

    if (!this.pUnit.equals(PERIOD_UNIT_YEAR) && !this.pUnit.equals(PERIOD_UNIT_MONTH)) {
      this.pUnit = PERIOD_UNIT_YEAR;
    }
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   *
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPDomainMapFactory.NS;
  }
}
