/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.defReg;

import java.util.Vector;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCheckCmd;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Represents an EPP DefReg {@code <check>} command, which is used to determine if
 * an object may be provisioned within a repository. It provides a hint that allows a client to
 * anticipate the success or failure of provisioning an object using the {@code <create>} command.
 * Object availability and provisioning conditions are a matter of server policy.<br><br>
 * 
 * In addition to the standard EPP command elements, the {@code <check>} command MUST contain
 * a {@code <defReg:check>} element that identifies the defReg namespace. The
 * {@code <defReg:check>} element 
 * contains the following child elements:
 * 
 * <ul>
 * <li>
 * One or more {@code <defReg:name>} elements
 * that contain the fully qualified names of defRegs to be queried.
 * </li>
 * </ul>
 * 
 * <code>EPPDefRegCheckResp</code> is the concrete <code>EPPReponse</code>
 * associated with <code>EPPDefRegCheckCmd</code>.
 *
 * @see com.verisign.epp.codec.defReg.EPPDefRegCheckResp
 */
public class EPPDefRegCheckCmd extends EPPCheckCmd {
  /** Maximum number of defRegs to check at once. */
  public static final int MAX_DEFREGS = 99;

  /** XML Element Name of <code>EPPDefRegCheckCmd</code> root element. */
  final static String ELM_NAME = "defReg:check";

  /** XML Element Name for the <code>names</code> attribute. */
  private final static String ELM_DEFREG_NAME = "defReg:name";

  /**
   * DefReg Names to check. This is a <code>Vector</code> of
   * <code>String</code> instances.
   */
  private Vector names;

  /**
   * <code>EPPDefRegCheckCmd</code> default constructor. It will set the names
   * attribute to an empty <code>Vector</code>.
   */
  public EPPDefRegCheckCmd() {
    names = new Vector();
  }

  // End EPPDefRegCheckCmd.EPPDefRegCheckCmd()

  /**
   * <code>EPPDefRegCheckCmd</code> constructor that will check an individual
   * defReg name.
   *
   * @param aTransId
   *           Transaction Id associated with command.
   * @param aName
   *           DefReg name to check
   */
  public EPPDefRegCheckCmd(String aTransId, EPPDefRegName aName) {
    super(aTransId);

    names = new Vector();
    names.addElement(aName);
  }

  // End EPPDefRegCheckCmd.EPPDefRegCheckCmd(String, String)

  /**
   * <code>EPPDefRegCheckCmd</code> constructor that will check a list of
   * defReg names.
   *
   * @param aTransId
   *           Transaction Id associated with command.
   * @param someNames
   *           <code>Vector</code> of defReg name <code>String</code>
   *           instances.
   */
  public EPPDefRegCheckCmd(String aTransId, Vector someNames) {
    super(aTransId);

    names = someNames;
  }

  // End EPPDefRegCheckCmd.EPPDefRegCheckCmd(String, Vector)

  /**
   * Gets the EPP command Namespace associated with
   * <code>EPPDefRegCheckCmd</code>.
   *
   * @return <code>EPPDefRegMapFactory.NS</code>
   */
  public String getNamespace() {
    return EPPDefRegMapFactory.NS;
  }

  /**
   * Gets the key for the defensive registration object, which is the defensive registration name.
   * 
   * @return The defensive registration name if set with a single value; {@code null} otherwise.
   */
  @Override
  public String getKey() {
	  if (this.names != null && this.names.size() == 1) {
		  return (String) this.names.elementAt(0);
	  }
	  else {
		  return null;
	  }
  }

  /**
   * Compare an instance of <code>EPPDefRegCheckCmd</code> with this instance.
   *
   * @param aObject
   *           Object to compare with.
   *
   * @return DOCUMENT ME!
   */
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPDefRegCheckCmd)) {
      return false;
    }

    if (!super.equals(aObject)) {
      return false;
    }

    EPPDefRegCheckCmd theMap = (EPPDefRegCheckCmd) aObject;

    // DefReg Names
    if (!EPPUtil.equalVectors(names, theMap.names)) {
      return false;
    }

    return true;
  }

  // End EPPDefRegCheckCmd.equals(Object)

  /**
   * Clone <code>EPPDefRegCheckCmd</code>.
   *
   * @return Deep copy clone of <code>EPPDefRegCheckCmd</code>
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  public Object clone() throws CloneNotSupportedException {
    EPPDefRegCheckCmd clone = (EPPDefRegCheckCmd) super.clone();

    clone.names = (Vector) names.clone();

    return clone;
  }

  // End EPPDefRegCheckCmd.clone()

  /**
   * Implementation of <code>Object.toString</code>, which will result in an
   * indented XML <code>String</code> representation of the concrete
   * <code>EPPCodecComponent</code>.
   *
   * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
   *         otherwise.
   */
  public String toString() {
    return EPPUtil.toString(this);
  }

  // End EPPDefRegCheckCmd.toString()

  /**
   * Sets defReg name to check.
   *
   * @param aName
   *           Name to check.
   */
  public void setName(EPPDefRegName aName) {
    names = new Vector();
    names.addElement(aName);
  }

  // End EPPDefRegCheckCmd.setNames(Vector)

  /**
   * Gets defReg names to check.
   *
   * @return Vector of defReg name <code>String</code>'s.
   */
  public Vector getNames() {
    return names;
  }

  // End EPPDefRegCheckCmd.getNames()

  /**
   * Sets defReg names to check.
   *
   * @param someNames
   *           Names to check.
   */
  public void setNames(Vector someNames) {
    names = someNames;
  }

  // End EPPDefRegCheckCmd.setNames(Vector)

  /**
   * Encode a DOM Element tree from the attributes of the
   * <code>EPPDefRegCheckCmd</code> instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         <code>EPPDefRegCheckCmd</code> instance.
   *
   * @exception EPPEncodeException
   *               Unable to encode <code>EPPDefRegCheckCmd</code> instance.
   */
  protected Element doEncode(Document aDocument) throws EPPEncodeException {
    if (names.size() == 0) {
      throw new EPPEncodeException("No defRegs names specified in EPPHostCheckCmd");
    }

    if (names.size() > MAX_DEFREGS) {
      throw new EPPEncodeException(names.size() + " defReg names is greater than the maximum of " + MAX_DEFREGS);
    }

    Element root = aDocument.createElementNS(EPPDefRegMapFactory.NS, ELM_NAME);

    root.setAttribute("xmlns:defReg", EPPDefRegMapFactory.NS);

    // Names
    EPPUtil.encodeCompVector(aDocument, root, names);

    return root;
  }

  // End EPPDefRegCheckCmd.doEncode(Document)

  /**
   * Decode the <code>EPPDefRegCheckCmd</code> attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode <code>EPPDefRegCheckCmd</code> from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  protected void doDecode(Element aElement) throws EPPDecodeException {
    // DefReg Names
    names = EPPUtil.decodeCompVector(aElement, EPPDefRegMapFactory.NS, ELM_DEFREG_NAME, EPPDefRegName.class);

    if (names == null) {
      names = new Vector();
    }
  }

  // End EPPDefRegCheckCmd.doDecode(Node)
}
