/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.validate.v02;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Provides the response to a {@link EPPValidateCheckCmd} that indicates whether
 * or not the contacts are valid for the referenced contact type and TLD.
 * 
 * @see com.verisign.epp.codec.validate.v02.EPPValidateCheckCmd
 * @see com.verisign.epp.codec.validate.v02.EPPValidateCheckResult
 */
public class EPPValidateCheckResp extends EPPResponse {

  /**
   * Constant for the name verification check response
   */
  public static final String ELM_LOCALNAME = "chkData";

  /**
   * XML Element Name of <code>EPPValidateCheckResp</code> root element.
   */
  final static String ELM_NAME = EPPValidateMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /** Category for logging */
  private static Logger cat = LoggerFactory.getLogger(EPPValidateCheckResp.class);

  /**
   * List of check results
   */
  private List<EPPValidateCheckResult> results = new ArrayList<EPPValidateCheckResult>();

  /**
   * <code>EPPValidateCheckResp</code> default constructor.
   */
  public EPPValidateCheckResp() {
  }

  /**
   * <code>EPPValidateCheckResp</code> constructor that sets the transaction
   * identifier.
   *
   * @param aTransId
   *           Transaction Id associated with response.
   */
  public EPPValidateCheckResp(EPPTransId aTransId) {
    super(aTransId);
  }

  /**
   * <code>EPPValidateCheckResp</code> constructor that will set the result of
   * an individual result.
   *
   * @param aTransId
   *           Transaction Id associated with response.
   * @param aResult
   *           Result of a single label.
   */
  public EPPValidateCheckResp(EPPTransId aTransId, EPPValidateCheckResult aResult) {
    super(aTransId);
    this.results.add(aResult);
  }

  /**
   * <code>EPPValidateCheckResp</code> constructor that will set the result of
   * multiple results.
   *
   * @param aTransId
   *           Transaction Id associated with response.
   * @param aResults
   *           List of <code>EPPValidateCheckResult</code> instances.
   */
  public EPPValidateCheckResp(EPPTransId aTransId, List<EPPValidateCheckResult> aResults) {
    super(aTransId);
    this.results = aResults;
  }

  /**
   * Get the EPP response type associated with
   * <code>EPPValidateCheckResp</code>.
   *
   * @return EPPValidateCheckResp.ELM_NAME
   */
  public String getType() {
    return ELM_NAME;
  }

  /**
   * Get the EPP command Namespace associated with
   * <code>EPPValidateCheckResp</code>.
   *
   * @return <code>EPPValidateMapFactory.NS</code>
   */
  public String getNamespace() {
    return EPPValidateMapFactory.NS;
  }

  /**
   * Compare an instance of <code>EPPValidateCheckResp</code> with this
   * instance.
   *
   * @param aObject
   *           Object to compare with.
   *
   * @return <code>true</code> if equal; <code>false</code> otherwise.
   */
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPValidateCheckResp)) {
      return false;
    }

    if (!super.equals(aObject)) {
      return false;
    }

    EPPValidateCheckResp other = (EPPValidateCheckResp) aObject;

    // results
    if (!EqualityUtil.equals(this.results, other.results)) {
      cat.error("EPPValidateCheckResp.equals(): results not equal");
      return false;
    }

    return true;
  }

  /**
   * Clone <code>EPPValidateCheckResp</code>.
   *
   * @return clone of <code>EPPValidateCheckResp</code>
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  public Object clone() throws CloneNotSupportedException {
    EPPValidateCheckResp clone = (EPPValidateCheckResp) super.clone();

    clone.results = new ArrayList<EPPValidateCheckResult>(this.results.size());
    for (EPPValidateCheckResult item : this.results)
      clone.results.add((EPPValidateCheckResult) item.clone());

    return clone;
  }

  /**
   * Implementation of <code>Object.toString</code>, which will result in an
   * indented XML <code>String</code> representation of the concrete
   * <code>EPPCodecComponent</code>.
   *
   * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
   *         otherwise.
   */
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Adds a result to the list of check results.
   * 
   * @param aResult
   *           Result to add.
   */
  public void addResult(EPPValidateCheckResult aResult) {
    this.results.add(aResult);
  }

  /**
   * Set the results of a <code>EPPValidateCheckResp</code> Response. There is
   * one <code>EPPValidateCheckResult</code> instance in <code>aResults</code>
   * for each contact requested in the <code>EPPValidateCheckCmd</code>
   * Command.
   *
   * @param aResults
   *           List of <code>EPPValidateCheckResult</code> instances.
   */
  public void setCheckResults(List<EPPValidateCheckResult> aResults) {
    if (aResults == null) {
      this.results = new ArrayList<EPPValidateCheckResult>();
    }
    else {
      this.results = aResults;
    }
  }

  /**
   * Get the results of a <code>EPPValidateCheckResp</code> Response.
   *
   * @return List of <code>EPPValidateCheckResult</code> instances.
   */
  public List<EPPValidateCheckResult> getCheckResults() {
    return this.results;
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * <code>EPPValidateCheckResp</code> instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         <code>EPPValidateCheckResp</code> instance.
   *
   * @exception EPPEncodeException
   *               Unable to encode <code>EPPValidateCheckResp</code> instance.
   */
  protected Element doEncode(Document aDocument) throws EPPEncodeException {
    Element root = aDocument.createElementNS(EPPValidateMapFactory.NS, ELM_NAME);

    // Results
    EPPUtil.encodeCompList(aDocument, root, this.results);

    return root;
  }

  /**
   * Decode the <code>EPPValidateCheckResp</code> attributes from the aElement
   * DOM Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode <code>EPPValidateCheckResp</code>
   *           from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  protected void doDecode(Element aElement) throws EPPDecodeException {
    // Results
    this.results = EPPUtil.decodeCompList(aElement, EPPValidateMapFactory.NS, EPPValidateCheckResult.ELM_NAME,
          EPPValidateCheckResult.class);
  }

}
