/***********************************************************
Copyright (C) 2023 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.serverstub.v1_0;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.host.EPPHostCreateCmd;
import com.verisign.epp.codec.host.EPPHostInfoCmd;
import com.verisign.epp.codec.host.EPPHostUpdateCmd;
import com.verisign.epp.codec.ttl.v1_0.EPPTtl;
import com.verisign.epp.codec.ttl.v1_0.EPPTtlCreate;
import com.verisign.epp.codec.ttl.v1_0.EPPTtlInfData;
import com.verisign.epp.codec.ttl.v1_0.EPPTtlInfo;
import com.verisign.epp.codec.ttl.v1_0.EPPTtlUpdate;
import com.verisign.epp.codec.ttl.v1_0.RRecType;
import com.verisign.epp.serverstub.TtlSubHostHandler;

/**
 * The <code>TtlHostHandler</code> class extends <code>HostHandler</code> to
 * include handling the TTL extension.
 */
public class TtlV1_0HostHandler implements TtlSubHostHandler {

  /**
   * Category for logging
   */
  private static Logger cat = LoggerFactory.getLogger(TtlV1_0HostHandler.class);

  /**
   * Handle an EPP Host Info Command.
   * 
   * @param aInfoCommand
   *           Command sent by the client
   * @param aResponse
   *           Response created previously up the stack
   * @param aData
   *           Server data. This is assumed to be an instance of
   *           <code>SessionData</code>.
   */
  public void doHostInfo(EPPHostInfoCmd aInfoCommand, EPPResponse aResponse, Object aData) {

    cat.info("doHostInfo: Add TTL extension to response");

    if (aInfoCommand.hasExtension(EPPTtlInfo.class)) {
      EPPTtlInfo ttlInfoExt = (EPPTtlInfo) aInfoCommand.getExtension(EPPTtlInfo.class);
            
      if (ttlInfoExt.isPolicy()) {
        cat.info("doDomainInfo: Add TTL extension with policy attributes to response");
        EPPTtlInfData theInfExt = new EPPTtlInfData();
        theInfExt.addTtl(new EPPTtl(RRecType.A, null, 3600, 172800, 86400, null));
        theInfExt.addTtl(new EPPTtl(RRecType.AAAA, null, 3600, 172800, 86400, 86400));
        
        aResponse.addExtension(theInfExt);              
      }
      else {
        cat.info("doDomainInfo: Add TTL extension to response");
        EPPTtlInfData theInfExt = new EPPTtlInfData();
        theInfExt.addTtl(new EPPTtl(RRecType.A, 3600));
        theInfExt.addTtl(new EPPTtl(RRecType.AAAA, 86400));
        aResponse.addExtension(theInfExt);                      
      }
    }
  }

  /**
   * Handle an EPP Host Create Command.
   * 
   * @param aCreateCommand
   *           Command sent by the client
   * @param aResponse
   *           Response created previously up the stack
   * @param aData
   *           Server data. This is assumed to be an instance of
   *           <code>SessionData</code>.
   */
  public void doHostCreate(EPPHostCreateCmd aCreateCommand, EPPResponse aResponse, Object aData) {
    cat.info(
          "doHostCreate: TTL create extension = " + (EPPTtlCreate) aCreateCommand.getExtension(EPPTtlCreate.class));
  }

  /**
   * Handle an EPP Host Update Command.
   * 
   * @param aUpdateCommand
   *           Command sent by the client
   * @param aResponse
   *           Response created previously up the stack
   * @param aData
   *           Server data. This is assumed to be an instance of
   *           <code>SessionData</code>.
   */
  public void doHostUpdate(EPPHostUpdateCmd aUpdateCommand, EPPResponse aResponse, Object aData) {
    cat.info(
          "doHostUpdate: TTL update extension = " + (EPPTtlUpdate) aUpdateCommand.getExtension(EPPTtlUpdate.class));
  }

}
