/***********************************************************
Copyright (C) 2023 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.interfaces.v1_0;

import java.util.Random;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.domain.EPPDomainCreateResp;
import com.verisign.epp.codec.domain.EPPDomainInfoResp;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.host.EPPHostInfoResp;
import com.verisign.epp.codec.secdnsext.v11.EPPSecDNSAlgorithm;
import com.verisign.epp.codec.secdnsext.v11.EPPSecDNSExtCreate;
import com.verisign.epp.codec.secdnsext.v11.EPPSecDNSExtDsData;
import com.verisign.epp.codec.ttl.v1_0.EPPTtl;
import com.verisign.epp.codec.ttl.v1_0.EPPTtlCreate;
import com.verisign.epp.codec.ttl.v1_0.EPPTtlInfData;
import com.verisign.epp.codec.ttl.v1_0.EPPTtlInfo;
import com.verisign.epp.codec.ttl.v1_0.EPPTtlUpdate;
import com.verisign.epp.codec.ttl.v1_0.RRecType;
import com.verisign.epp.interfaces.EPPApplicationSingle;
import com.verisign.epp.interfaces.EPPCommandException;
import com.verisign.epp.interfaces.EPPDomain;
import com.verisign.epp.interfaces.EPPHost;
import com.verisign.epp.interfaces.EPPSession;
import com.verisign.epp.util.Environment;
import com.verisign.epp.util.TestThread;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * {@code EPPTtlTst} is a unit test of the TTL extension interface. 
 */
public class EPPTtlTst {
  /**
   * Handle to the Singleton EPP Application instance (
   * {@code EPPApplicationSingle})
   */
  private static EPPApplicationSingle app = EPPApplicationSingle.getInstance();

  /** Name of configuration file to use for test (default = epp.config). */
  private static String configFileName = "epp.config";

  /** Logging category */
  private static Logger cat = LoggerFactory.getLogger(EPPTtlTst.class);

  /** EPP Domain associated with test */
  private EPPDomain domain = null;

  /** EPP Host associated with test */
  private EPPHost host = null;

  /** EPP Session associated with test */
  private EPPSession session = null;

  /** Current test iteration */
  private int iteration = 0;

  /**
   * Random instance for the generation of unique objects (hosts, IP addresses,
   * etc.).
   */
  private Random rd = new Random(System.currentTimeMillis());

  /**
   * JUNIT test method to implement the {@code EPPTtlDomainTst TestCase}.
   * Each sub-test will be invoked in order to satisfy testing the EPPDomain
   * interface.
   */
  @Test
  public void testTtl() {
    int numIterations = 1;

    String iterationsStr = System.getProperty("iterations");

    if (iterationsStr != null) {
      numIterations = Integer.parseInt(iterationsStr);
    }

    for (iteration = 0; (numIterations == 0) || (iteration < numIterations); iteration++) {
      printStart("Test Suite");

      domainInfo();

      hostInfo();

      domainCreate();

      hostCreate();

      domainUpdate();

      hostUpdate();

      printEnd("Test Suite");
    }
  }

  /**
   * Unit test of {@code EPPDomain.sendInfo} with TTL extension.
   */
  private void domainInfo() {
    printStart("domainInfo");

    EPPDomainInfoResp theResponse;

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("domainInfo: Info domain with TTL extension and policy default value of false");

      domain.setTransId("ABC-12345");

      domain.addDomainName("ttl-example.com");
      domain.addExtension(new EPPTtlInfo());

      theResponse = domain.sendInfo();

      System.out.println("\ndomainInfo: Response = [" + theResponse + "]");

      assert theResponse.hasExtension(EPPTtlInfData.class);

      System.out.println(
            "domainInfo: Response TTL data = " + (EPPTtlInfData) theResponse.getExtension(EPPTtlInfData.class));
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("domainInfo: Info domain with TTL extension and policy set to false");

      domain.setTransId("ABC-12345");

      domain.addDomainName("ttl-example.com");
      domain.addExtension(new EPPTtlInfo(false));

      theResponse = domain.sendInfo();

      System.out.println("\ndomainInfo: Response = [" + theResponse + "]");

      assert theResponse.hasExtension(EPPTtlInfData.class);

      System.out.println(
            "domainInfo: Response TTL data = " + (EPPTtlInfData) theResponse.getExtension(EPPTtlInfData.class));
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("domainInfo: Info domain with TTL extension and with policy set to true");

      domain.setTransId("ABC-12345");

      domain.addDomainName("ttl-example.com");
      domain.addExtension(new EPPTtlInfo(true));

      theResponse = domain.sendInfo();

      System.out.println("\ndomainInfo: Response = [" + theResponse + "]");

      assert theResponse.hasExtension(EPPTtlInfData.class);

      System.out.println(
            "domainInfo: Response TTL data = " + (EPPTtlInfData) theResponse.getExtension(EPPTtlInfData.class));
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("domainInfo");
  }

  /**
   * Unit test of {@code EPPHost.sendInfo} with TTL extension.
   */
  private void hostInfo() {
    printStart("hostInfo");

    EPPHostInfoResp theResponse;

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("hostInfo: Info host with TTL extension and policy default value of false");

      host.setTransId("ABC-12345");

      host.addHostName("ttl-ns1.example.com");
      host.addExtension(new EPPTtlInfo());

      theResponse = host.sendInfo();

      System.out.println("\nhostInfo: Response = [" + theResponse + "]");

      assert theResponse.hasExtension(EPPTtlInfData.class);

      System.out.println(
            "hostInfo: Response TTL data = " + (EPPTtlInfData) theResponse.getExtension(EPPTtlInfData.class));
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("hostInfo: Info host with TTL extension and with policy set to false");

      host.setTransId("ABC-12345");

      host.addHostName("ttl-ns1.example.com");
      host.addExtension(new EPPTtlInfo(false));

      theResponse = host.sendInfo();

      System.out.println("\nhostInfo: Response = [" + theResponse + "]");

      assert theResponse.hasExtension(EPPTtlInfData.class);

      System.out.println(
            "hostInfo: Response TTL data = " + (EPPTtlInfData) theResponse.getExtension(EPPTtlInfData.class));
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("hostInfo: Info host with TTL extension and with policy set to true");

      host.setTransId("ABC-12345");

      host.addHostName("ttl-ns1.example.com");
      host.addExtension(new EPPTtlInfo(true));

      theResponse = host.sendInfo();

      System.out.println("\nhostInfo: Response = [" + theResponse + "]");

      assert theResponse.hasExtension(EPPTtlInfData.class);

      System.out.println(
            "hostInfo: Response TTL data = " + (EPPTtlInfData) theResponse.getExtension(EPPTtlInfData.class));
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("hostInfo");
  }

  /**
   * Unit test of {@code EPPDomain.sendCreate} with TTL extension.
   */
  public void domainCreate() {
    printStart("domainCreate");

    EPPDomainCreateResp response;

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("domainCreate: Create domain ttl-example.com with TTL extension");

      domain.setTransId("ABC-12345");
      domain.addDomainName("ttl-example.com");
      domain.setAuthString("");
      domain.addHostName("ns1.example.com");
      domain.addHostName("ns1.example.net");

      // TTL Extension
      EPPTtlCreate theTtlExt = new EPPTtlCreate();
      theTtlExt.addTtl(new EPPTtl(RRecType.NS, 172800));
      theTtlExt.addTtl(new EPPTtl(RRecType.DS, 300));
      theTtlExt.addTtl(new EPPTtl(RRecType.CUSTOM, "DELEG", 3600));
      domain.addExtension(theTtlExt);

      // DNSSEC Extension
      EPPSecDNSExtCreate theDnssecExt = new EPPSecDNSExtCreate();
      theDnssecExt.appendDsData(new EPPSecDNSExtDsData(12345, EPPSecDNSAlgorithm.ECDSAP256SHA256,
            EPPSecDNSExtDsData.SHA256_DIGEST_TYPE, "49FD46E6C4B45C55D4AC"));
      domain.addExtension(theDnssecExt);

      response = domain.sendCreate();

      // -- Output all of the response attributes
      System.out.println("domainCreate: Response = [" + response + "]\n\n");

      assert response.isSuccess();
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("domainCreate");
  }

  /**
   * Unit test of {@code EPPHost.sendCreate} with TTL extension.
   */
  public void hostCreate() {
    printStart("hostCreate");

    EPPResponse response;

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("hostCreate: Create host ttl-ns1.example.com with TTL extension");

      host.setTransId("ABC-12345");
      host.addHostName("ttl-ns1.example.com");
      host.addIPV4Address("192.0.2.2");
      host.addIPV6Address("1080::8:800:200C:417A");

      // TTL Extension
      EPPTtlCreate theCreateExt = new EPPTtlCreate();
      theCreateExt.addTtl(new EPPTtl(RRecType.A));
      theCreateExt.addTtl(new EPPTtl(RRecType.AAAA, 86400));
      host.addExtension(theCreateExt);

      response = host.sendCreate();

      // -- Output all of the response attributes
      System.out.println("hostCreate: Response = [" + response + "]\n\n");

      assert response.isSuccess();
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("hostCreate");
  }

  /**
   * Unit test of {@code EPPDomain.sendUpdate} with TTL extension.
   */
  public void domainUpdate() {
    printStart("domainUpdate");

    EPPResponse response;

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("domainUpdate: Update domain ttl-example.com with TTL extension");

      domain.setTransId("ABC-12345");
      domain.addDomainName("ttl-example.com");

      // TTL Extension
      EPPTtlUpdate theUpdateExt = new EPPTtlUpdate();
      theUpdateExt.addTtl(new EPPTtl(RRecType.NS));
      domain.addExtension(theUpdateExt);

      response = domain.sendUpdate();

      // -- Output all of the response attributes
      System.out.println("domainUpdate: Response = [" + response + "]\n\n");

      assert response.isSuccess();
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("domainUpdate");
  }

  /**
   * Unit test of {@code EPPHost.sendUpdate} with TTL extension.
   */
  public void hostUpdate() {
    printStart("hostUpdate");

    EPPResponse response;

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("hostUpdate: Update host ttl-ns1.example.com with TTL extension");

      host.setTransId("ABC-12345");
      host.addHostName("ttl-ns1.example.com");

      // TTL Extension
      EPPTtlUpdate theUpdateExt = new EPPTtlUpdate();
      theUpdateExt.addTtl(new EPPTtl(RRecType.A, 86400));
      theUpdateExt.addTtl(new EPPTtl(RRecType.AAAA, 3600));
      host.addExtension(theUpdateExt);

      response = host.sendUpdate();

      // -- Output all of the response attributes
      System.out.println("hostUpdate: Response = [" + response + "]\n\n");

      assert response.isSuccess();
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("hostUpdate");
  }

  /**
   * Unit test of {@code EPPSession.initSession}. The session attribute is
   * initialized with the attributes defined in the EPP sample files.
   */
  private void initSession() {
    printStart("initSession");

    // Set attributes for initSession
    session.setClientID(Environment.getProperty("EPP.Test.clientId", "ClientX"));
    session.setPassword(Environment.getProperty("EPP.Test.password", "foo-BAR2"));
    session.setTransId("ABC-12345-XYZ");
    session.setVersion("1.0");
    session.setLang("en");

    // Initialize the session
    try {
      session.initSession();
    }

    catch (EPPCommandException e) {
      EPPResponse response = session.getResponse();

      // Is a server specified error?
      if ((response != null) && (!response.isSuccess())) {
        Assert.fail("Server Error : " + response);
      }
      else {
        e.printStackTrace();

        Assert.fail("initSession Error : " + e);
      }
    }

    printEnd("initSession");
  }

  /**
   * Unit test of {@code EPPSession.endSession}. The session with the EPP
   * Server will be terminated.
   */
  private void endSession() {
    printStart("endSession");

    session.setTransId("ABC-12345-XYZ");

    // End the session
    try {
      session.endSession();
    }

    catch (EPPCommandException e) {
      EPPResponse response = session.getResponse();

      // Is a server specified error?
      if ((response != null) && (!response.isSuccess())) {
        Assert.fail("Server Error : " + response);
      }

      else // Other error
      {
        e.printStackTrace();

        Assert.fail("initSession Error : " + e);
      }
    }

    printEnd("endSession");
  }

  /**
   * JUNIT {@code setUp} method, which sets the default client Id to
   * "theRegistrar".
   */
  @Before
  public void setUp() {
    try {
      String theSessionClassName = System.getProperty("EPP.SessionClass");

      if (theSessionClassName != null) {
        try {
          Class theSessionClass = Class.forName(theSessionClassName);

          if (!EPPSession.class.isAssignableFrom((theSessionClass))) {
            Assert.fail(theSessionClassName + " is not a subclass of EPPSession");
          }

          session = (EPPSession) theSessionClass.getDeclaredConstructor().newInstance();
        }
        catch (Exception ex) {
          Assert.fail("Exception instantiating EPP.SessionClass value " + theSessionClassName + ": " + ex);
        }
      }
      else {
        session = new EPPSession();
      }

    }

    catch (Exception e) {
      e.printStackTrace();

      Assert.fail("Error initializing the session: " + e);
    }

    initSession();

    domain = new EPPDomain(session);
    host = new EPPHost(session);
  }

  /**
   * JUNIT {@code tearDown}, which currently does nothing.
   */
  @After
  public void tearDown() {
    endSession();
  }

  /**
   * JUNIT {@code suite} static method, which returns the tests associated
   * with {@code EPPTtlDomainTst}.
   */
  @BeforeClass
  public static void beforeClass() {
    String theConfigFileName = System.getProperty("EPP.ConfigFile");
    if (theConfigFileName != null)
      configFileName = theConfigFileName;

    try {
      app.initialize(configFileName);
    }

    catch (EPPCommandException e) {
      e.printStackTrace();

      Assert.fail("Error initializing the EPP Application: " + e);
    }
  }

  /**
   * Handle an {@code EPPCommandException}, which can be either a server
   * generated error or a general exception. If the exception was caused by a
   * server error, "Server Error :&lt;Response XML&gt;" will be specified. If
   * the exception was caused by a general algorithm error, "General Error
   * :&lt;Exception Description&gt;" will be specified.
   *
   * @param aException
   *           Exception thrown during test
   */
  public void handleException(Exception aException) {
    EPPResponse response = session.getResponse();

    aException.printStackTrace();

    // Is a server specified error?
    if ((response != null) && (!response.isSuccess())) {
      Assert.fail("Server Error : " + response);
    }

    else {
      Assert.fail("General Error : " + aException);
    }
  }

  /**
   * This method tries to generate a unique String as Domain Name and Name
   * Server
   *
   * @return Unique domain name
   */
  public String makeDomainName() {
    long tm = System.currentTimeMillis();

    return new String(Thread.currentThread() + String.valueOf(tm + rd.nextInt(12)).substring(10) + ".com");
  }

  /**
   * Makes a unique IP address based off of the current time.
   *
   * @return Unique IP address {@code String}
   */
  public String makeIP() {
    long tm = System.currentTimeMillis();

    return new String(
          String.valueOf(tm + rd.nextInt(50)).substring(10) + "." + String.valueOf(tm + rd.nextInt(50)).substring(10)
                + "." + String.valueOf(tm + rd.nextInt(50)).substring(10) + "."
                + String.valueOf(tm + rd.nextInt(50)).substring(10));
  }

  /**
   * Makes a unique host name for a domain using the current time.
   *
   * @param aBaseDomainName
   *           Base domain name used for host name
   *
   * @return Unique host name {@code String}
   */
  public String makeHostName(String aBaseDomainName) {
    long tm = System.currentTimeMillis();

    return new String(String.valueOf(tm + rd.nextInt(10)).substring(10) + "." + aBaseDomainName);
  }

  /**
   * Print the start of a test with the {@code Thread} name if the current
   * thread is a {@code TestThread}.
   *
   * @param aTest
   *           name for the test
   */
  private void printStart(String aTest) {
    if (Thread.currentThread() instanceof TestThread) {
      System.out.print(Thread.currentThread().getName() + ", iteration " + iteration + ": ");

      cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aTest + " Start");
    }

    System.out.println("Start of " + aTest);

    System.out.println("****************************************************************\n");
  }

  /**
   * Print the end of a test with the {@code Thread} name if the current
   * thread is a {@code TestThread}.
   *
   * @param aTest
   *           name for the test
   */
  private void printEnd(String aTest) {
    System.out.println("****************************************************************");

    if (Thread.currentThread() instanceof TestThread) {
      System.out.print(Thread.currentThread().getName() + ", iteration " + iteration + ": ");

      cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aTest + " End");
    }

    System.out.println("End of " + aTest);

    System.out.println("\n");
  }

  /**
   * Print message
   *
   * @param aMsg
   *           message to print
   */
  private void printMsg(String aMsg) {
    if (Thread.currentThread() instanceof TestThread) {
      System.out.println(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);

      cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);
    }

    else {
      System.out.println(aMsg);

      cat.info(aMsg);
    }
  }

  /**
   * Print error message
   *
   * @param aMsg
   *           errpr message to print
   */
  private void printError(String aMsg) {
    if (Thread.currentThread() instanceof TestThread) {
      System.err.println(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);

      cat.error(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);
    }

    else {
      System.err.println(aMsg);

      cat.error(aMsg);
    }
  }
}
