/***********************************************************
Copyright (C) 2024 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.ttl.v1_0;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * TTL Info Extension that enables a client to indicate whether to include the
 * TTL policy attributes in the response.
 */
public class EPPTtlInfo implements EPPCodecComponent {

  private static final long serialVersionUID = 1L;

  /**
   * XML local name for {@code EPPTtlInfo}.
   */
  public static final String ELM_LOCALNAME = "info";

  /**
   * XML root tag for {@code EPPTtlInfo}.
   */
  public static final String ELM_NAME = EPPTtlExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * "policy" attribute that indicates the whether to include the policy
   * attributes in the response.
   */
  private final static String ATTR_POLICY = "policy";

  /**
   * Category for logging
   */
  private static Logger cat = LoggerFactory.getLogger(EPPTtlInfo.class);

  /**
   * Policy attribute with a default value of {@code false}.
   */
  private boolean policy = false;

  /**
   * Returns the local name &quot;info&quot; for the info extension.
   * 
   * @return Local name &quot;info&quot; for the info extension.
   */
  protected String getLocalName() {
    return ELM_LOCALNAME;
  }

  /**
   * Default constructor for {@code EPPTtlInfo}.
   */
  public EPPTtlInfo() {
  }

  /**
   * Constructor for {@code EPPTtlInfo} that takes a single TTL.
   *
   * @param aPolicy
   *           The policy
   */
  public EPPTtlInfo(boolean aPolicy) {
    this.policy = aPolicy;
  }

  /**
   * Is the policy attributes requested in the response?
   * 
   * @return {@code true} to include the policy attributes in the response;
   *         {@code false} otherwise.
   */
  public boolean isPolicy() {
    return policy;
  }

  /**
   * Sets whether the policy attributes are requested in the response.
   * 
   * @param aPolicy
   *           {@code true} to include the policy attributes in the response;
   *           {@code false} otherwise.
   */
  public void setPolicy(boolean aPolicy) {
    this.policy = aPolicy;
  }

  /**
   * Encode of a {@code EPPTtlInfo} instance that includes checks of the TTLs.
   * 
   * @param aDocument
   *           DOM Document, which acts is an Element factory
   * 
   * @return Element Root element associated with the object
   * 
   * @exception EPPEncodeException
   *               Error encoding {@code EPPTtlExtension}
   */
  public Element encode(Document aDocument) throws EPPEncodeException {

    Element root = aDocument.createElementNS(EPPTtlExtFactory.NS, ELM_NAME);

    EPPUtil.encodeBooleanAttr(root, ATTR_POLICY, this.policy);

    return root;
  }

  /**
   * Decode a DOM element tree to initialize the instance attributes. The
   * {@code aElement} argument represents the root DOM element and is used to
   * traverse the DOM nodes for instance attribute values.
   * 
   * @param aElement
   *           {@code Element} to decode
   * 
   * @exception EPPDecodeException
   *               Error decoding {@code Element}
   */
  public void decode(Element aElement) throws EPPDecodeException {
    if (aElement.getAttribute(ATTR_POLICY) == null) {
      this.policy = false;
    }
    else {
      this.policy = EPPUtil.decodeBooleanAttr(aElement, ATTR_POLICY);
    }

  }

  /**
   * clone an {@code EPPCodecComponent}.
   * 
   * @return clone of concrete {@code EPPTtlExtension}
   * 
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  public Object clone() throws CloneNotSupportedException {
    EPPTtlInfo clone = (EPPTtlInfo) super.clone();
    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   * 
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Compare an instance of {@code EPPTtlInfo} with this instance.
   * 
   * @param aObject
   *           Object to compare with.
   * 
   * @return {@code true} if equal; {@code false} otherwise.
   */
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPTtlInfo)) {
      cat.error("EPPTtlInfo.equals(): " + aObject.getClass().getName() + " not EPPTtlInfo instance");

      return false;
    }

    EPPTtlInfo other = (EPPTtlInfo) aObject;

    // Policy
    if (!EqualityUtil.equals(this.policy, other.policy)) {
      cat.error("EPPTtlInfo.equals(): policy not equal");
      return false;
    }

    return true;
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   * 
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPTtlExtFactory.NS;
  }

}
