/***********************************************************
Copyright (C) 2023 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.ttl.v1_0;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Abstract base class for the TTL extensions of the create and update commands
 * and the info response, which all accept a list of {@link EPPTtl} objects.
 */
public abstract class EPPTtlExtension implements EPPCodecComponent {

  private static final long serialVersionUID = 1L;

  /**
   * XML root tag for {@code EPPTtlExtension}.
   */
  private String ELM_NAME = EPPTtlExtFactory.NS_PREFIX + ":" + getLocalName();

  /**
   * Category for logging
   */
  private static Logger cat = LoggerFactory.getLogger(EPPTtlExtension.class);

  /**
   * List of TTLs
   */
  private List<EPPTtl> ttls = new ArrayList<EPPTtl>();

  /**
   * Abstract method that the sub-class must define to return the local name
   * for the root element.
   * 
   * @return Local name of the root element of the extensions.
   */
  protected abstract String getLocalName();

  /**
   * Default constructor for {@code EPPTtlExtension}.
   */
  public EPPTtlExtension() {
  }

  /**
   * Constructor for {@code EPPTtlExtension} that takes a single TTL.
   * 
   * @param aTtl
   *           A single TTL value
   */
  public EPPTtlExtension(EPPTtl aTtl) {
    this.addTtl(aTtl);
  }

  /**
   * Constructor for {@code EPPTtlExtension} that a list of TTLs.
   * 
   * @param aTtls
   *           A list of TTLs
   */
  public EPPTtlExtension(List<EPPTtl> aTtls) {
    this.setTtls(aTtls);
  }

  /**
   * Are the TTLs defined?
   * 
   * @return {@code true} if the TTLs are defined; {@code false} otherwise.
   */
  public boolean hasTtls() {
    return !this.ttls.isEmpty();
  }

  /**
   * Gets the list of TTLs if defined.
   * 
   * @return List of TTLs if defined; empty list otherwise.
   */
  public List<EPPTtl> getTtls() {
    return this.ttls;
  }

  /**
   * Adds a TTL to the list of TTLs.
   * 
   * @param aTtl
   *           The TTL to add.
   */
  public void addTtl(EPPTtl aTtl) {
    if (aTtl == null)
      return;

    this.ttls.add(aTtl);
  }

  /**
   * Sets the list of TTLs.
   * 
   * @param aTtls
   *           The TTLs to set.
   */
  public void setTtls(List<EPPTtl> aTtls) {
    if (aTtls == null)
      this.ttls = new ArrayList<EPPTtl>();
    else
      this.ttls = aTtls;
  }

  /**
   * Encode instance into a DOM element tree. A DOM Document is passed as an
   * argument and functions as a factory for DOM objects. The root element
   * associated with the instance is created and each instance attribute is
   * appended as a child node.
   * 
   * @param aDocument
   *           DOM Document, which acts is an Element factory
   * 
   * @return Element Root element associated with the object
   * 
   * @exception EPPEncodeException
   *               Error encoding {@code EPPTtlExtension}
   */
  public Element encode(Document aDocument) throws EPPEncodeException {

    if (aDocument == null) {
      throw new EPPEncodeException("aDocument is null" + " on in EPPTtlExtension.encode(Document)");
    }

    // Check for required attributes
    if (this.ttls.isEmpty()) {
      throw new EPPEncodeException("At least one TTL is required in EPPTtlExtension.encode(Document).");
    }

    // Create root element
    Element root = aDocument.createElementNS(EPPTtlExtFactory.NS, ELM_NAME);

    // TTLs
    EPPUtil.encodeCompList(aDocument, root, this.ttls);

    return root;
  }

  /**
   * Decode a DOM element tree to initialize the instance attributes. The
   * {@code aElement} argument represents the root DOM element and is used to
   * traverse the DOM nodes for instance attribute values.
   * 
   * @param aElement
   *           {@code Element} to decode
   * 
   * @exception EPPDecodeException
   *               Error decoding {@code Element}
   */
  public void decode(Element aElement) throws EPPDecodeException {

    // TTLS
    this.ttls = EPPUtil.decodeCompList(aElement, EPPTtlExtFactory.NS, EPPTtl.ELM_LOCALNAME, EPPTtl.class);
  }

  /**
   * clone an {@code EPPCodecComponent}.
   * 
   * @return clone of concrete {@code EPPTtlExtension}
   * 
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  public Object clone() throws CloneNotSupportedException {
    EPPTtlExtension clone = (EPPTtlExtension) super.clone();

    // Fees
    clone.ttls = new ArrayList<EPPTtl>();
    for (EPPTtl ttl : this.ttls) {
      clone.ttls.add((EPPTtl) ttl.clone());
    }

    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   * 
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Compare an instance of {@code EPPTtlExtension} with this instance.
   * 
   * @param aObject
   *           Object to compare with.
   * 
   * @return {@code true} if equal; {@code false} otherwise.
   */
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPTtlExtension)) {
      cat.error("EPPTtlExtension.equals(): " + aObject.getClass().getName() + " not EPPTtlExtension instance");

      return false;
    }

    EPPTtlExtension other = (EPPTtlExtension) aObject;

    // TTLs
    if (!EqualityUtil.equals(this.ttls, other.ttls)) {
      cat.error("EPPTtlExtension.equals(): ttls not equal");
      return false;
    }

    return true;
  }

}
