/***********************************************************
Copyright (C) 2013 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.relateddomainext;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * EPPCodecComponent that encodes and decodes a &lt;relDom:available&gt; tag
 * <p>
 * Title: EPP 1.0 Related Domain - available tag
 * </p>
 * <p>
 * Description: The EPPRelatedDomainExtAvailable object represents the
 * collection of domains that are available for registration in a family of
 * related domains. As such it is composed of a collection of
 * {@link EPPRelatedDomainExtName} objects. <br>
 * As XML, it is represented by a &lt;relDom:available&gt; element containing a
 * number of &lt;relDom:name&gt; elements.
 * </p>
 */
public class EPPRelatedDomainExtAvailable implements EPPCodecComponent {

  /**
   * Serial version id - increment this if the structure changes.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Category for logging
   */
  private static Logger cat = LoggerFactory.getLogger(EPPRelatedDomainExtAvailable.class);

  /**
   * Element tag name for available
   */
  public static final String ELM_NAME = EPPRelatedDomainExtFactory.NS_PREFIX + ":available";

  /**
   * List of domain names represented by the {@link EPPRelatedDomainExtName} to
   * be associated with the &lt;relDom:available&gt; element
   */
  private List<EPPRelatedDomainExtName> availableDomains = null;

  /**
   * Default constructor
   */
  public EPPRelatedDomainExtAvailable() {
  }

  /**
   * Constructor with a list of available domain names.
   * 
   * @param aAvailableDomains
   *           List of available domain names
   */
  public EPPRelatedDomainExtAvailable(final List<EPPRelatedDomainExtName> aAvailableDomains) {
    this.availableDomains = aAvailableDomains;

  }

  /**
   * A deep clone of the EPPRelatedDomainAvailable
   * 
   * @see java.lang.Object#clone()
   */
  public Object clone() throws CloneNotSupportedException {
    final EPPRelatedDomainExtAvailable theClone = new EPPRelatedDomainExtAvailable();

    if (this.availableDomains != null) {
      theClone.availableDomains = new ArrayList();

      for (EPPRelatedDomainExtName availableDomain : this.availableDomains) {

        if (availableDomain != null) {
          theClone.availableDomains.add((EPPRelatedDomainExtName) availableDomain.clone());
        }
        else {
          theClone.availableDomains.add(null);
        }

      }

    }
    else {
      theClone.availableDomains = null;
    }

    return theClone;
  }

  /**
   * Populate the data of this instance with the data stored in the given
   * Element of the DOM tree
   * 
   * @param aElement
   *           The root element of the report fragment of XML
   * @throws EPPDecodeException
   *            Thrown if any errors occur during decoding.
   */
  public void decode(final Element aElement) throws EPPDecodeException {

    this.availableDomains = EPPUtil.decodeCompList(aElement, EPPRelatedDomainExtFactory.NS,
          EPPRelatedDomainExtName.ELM_DOMAIN_NAME, EPPRelatedDomainExtName.class);
  }

  /**
   * Append all data from the list of the list of available related domain
   * names represented by {@link EPPRelatedDomainExtName} to given DOM Document
   * 
   * @param aDocument
   *           The DOM Document to append data to
   * @return Encoded DOM {@code Element}
   * @throws EPPEncodeException
   *            Thrown when errors occur during the encode attempt or if the
   *            instance is invalid.
   */
  public Element encode(final Document aDocument) throws EPPEncodeException {

    if (aDocument == null) {
      throw new EPPEncodeException("aDocument is null" + " in EPPRelatedDomainExtAvailable.encode(Document)");
    }

    try {
      // Validate States
      validateState();
    }
    catch (final EPPCodecException e) {
      cat.error("EPPRelatedDomainExtAvailable.encode(): Invalid state on encode: " + e);
      throw new EPPEncodeException("EPPRelatedDomainExtAvailable invalid state: " + e);
    }

    final Element root = aDocument.createElementNS(EPPRelatedDomainExtFactory.NS, ELM_NAME);

    // dsData
    EPPUtil.encodeCompList(aDocument, root, this.availableDomains);
    return root;
  }

  /**
   * A deep comparison of this with another EPPRelatedDomainExtAvailable.
   * 
   * @see java.lang.Object#equals(java.lang.Object)
   */
  public boolean equals(final Object aObj) {

    if (!(aObj instanceof EPPRelatedDomainExtAvailable)) {
      return false;
    }

    final EPPRelatedDomainExtAvailable theComp = (EPPRelatedDomainExtAvailable) aObj;

    if (!EPPUtil.equalLists(this.availableDomains, theComp.availableDomains)) {
      cat.error("EPPRelatedDomainExtAvailable.equals(): availableDomains not equal");
      return false;
    }

    return true;
  }

  /**
   * Adds a available domain represented by {@link EPPRelatedDomainExtName} to
   * the list of available domains.
   * 
   * @param aAvailableDomain
   *           Available domain to add to the list.
   */
  public void addAvailableDomain(EPPRelatedDomainExtName aAvailableDomain) {
    if (this.availableDomains == null) {
      this.availableDomains = new ArrayList<EPPRelatedDomainExtName>();
    }

    this.availableDomains.add(aAvailableDomain);
  }

  /**
   * Are there any available domains included in the list of available domains?
   * 
   * @return {@code true} if the available domain list is not {@code null} and
   *         there is at least one available domain in the list; {@code false}
   *         otherwise.
   */
  public boolean hasAvailableDomains() {
    if (this.availableDomains != null && this.availableDomains.size() > 0) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Returns the list of available related domain names represented by
   * {@link EPPRelatedDomainExtName}
   * 
   * @return the relatedDomains
   */
  public List<EPPRelatedDomainExtName> getAvailableDomains() {
    return this.availableDomains;
  }

  /**
   * Sets the list of available related domain names represented by
   * {@link EPPRelatedDomainExtName}
   * 
   * @param relatedDomains
   *           the relatedDomains to set
   */
  public void setAvailableDomains(final List<EPPRelatedDomainExtName> relatedDomains) {
    this.availableDomains = relatedDomains;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   * 
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   * 
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPRelatedDomainExtFactory.NS;
  }

  /**
   * Validate the state of the {@code EPPRelatedDomainExtAvailable} instance. A
   * valid state means that all of the required attributes have been set. If
   * validateState returns without an exception, the state is valid. If the
   * state is not valid, the {@code EPPCodecException} will contain a
   * description of the error. throws EPPCodecException State error. This will
   * contain the name of the attribute that is not valid.
   * 
   * @throws EPPCodecException
   *            Thrown if the instance is in an invalid state
   */
  private void validateState() throws EPPCodecException {
    if ((this.availableDomains == null) || (this.availableDomains.size() == 0)) {
      throw new EPPCodecException("EPPRelatedDomainExtAvailable contains no elements.");
    }

  }

}
