/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.framework.registry.v02;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPFactory;
import com.verisign.epp.codec.gen.EPPMessage;
import com.verisign.epp.codec.registry.v02.EPPRegistryCheckCmd;
import com.verisign.epp.codec.registry.v02.EPPRegistryCreateCmd;
import com.verisign.epp.codec.registry.v02.EPPRegistryDeleteCmd;
import com.verisign.epp.codec.registry.v02.EPPRegistryInfoCmd;
import com.verisign.epp.codec.registry.v02.EPPRegistryMapFactory;
import com.verisign.epp.codec.registry.v02.EPPRegistryUpdateCmd;
import com.verisign.epp.framework.EPPEvent;
import com.verisign.epp.framework.EPPEventException;
import com.verisign.epp.framework.EPPEventHandler;
import com.verisign.epp.framework.EPPEventResponse;
import com.verisign.epp.framework.EPPHandleEventException;


/**
 * The {@code EPPRegistryHandler} class provides an interface for handling EPP
 * Registry Commands in a Server implementation. EPPEvents are handled by the
 * handleEvent() method here and routed to the appropriate abstract member
 * function. Subclasses should override the abstract methods to define the
 * desired behavior of a particular command when it is received. A subclassed
 * instance of {@code EPPRegistryHandler} should be registered with the
 * {@code EPPDispatcher} so that EEPEvents related to the Registry Mapping will
 * be handled there. <br>
 * <br>
 * 
 * 
 * @see EPPEventHandler
 * @see EPPEvent
 */
public abstract class EPPRegistryHandler implements EPPEventHandler {
  /** Category for logging */
  private static Logger cat = LoggerFactory.getLogger(EPPRegistryHandler.class);
        

  /**
   * Construct an instance of {@code EPPRegistryHandler} that automatically
   * loads the {@link EPPRegistryMapFactory} into the {@link EPPFactory}.
   */
  public EPPRegistryHandler() {
    try {
      EPPFactory.getInstance().addMapFactory(EPPRegistryMapFactory.class.getName());
    }
    catch (EPPCodecException e) {
      cat.error("Couldn't load the Map Factory associated with the Registry Mapping", e);
      System.exit(1);
    }
  }

  /**
   * This method receives an {@code EPPEvent} and routes it to the appropriate
   * abstract method.
   * 
   * @param aEvent
   *           An {@link EPPEvent} that contains the
   *           {@link com.verisign.epp.codec.gen.EPPCommand}
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPRegistryHandler} instance
   * 
   * @return Result of handling the event
   * 
   * @exception EPPEventException
   *               Thrown if an unrecognized {@link EPPEvent} is received
   */
  public EPPEventResponse handleEvent(EPPEvent aEvent, Object aData) throws EPPEventException {
    try {
      this.preHandleEvent(aEvent, aData);
    }
    catch (EPPHandleEventException e) {
      return new EPPEventResponse(e.getResponse());
    }

    EPPMessage message = aEvent.getMessage();
    EPPEventResponse response = null;

    if (message instanceof EPPRegistryInfoCmd) {
      response = doRegistryInfo(aEvent, aData);
    }
    if (message instanceof EPPRegistryCheckCmd) {
      response = doRegistryCheck(aEvent, aData);
    }
    if (message instanceof EPPRegistryCreateCmd) {
      response = doRegistryCreate(aEvent, aData);
    }
    if (message instanceof EPPRegistryUpdateCmd) {
      response = doRegistryUpdate(aEvent, aData);
    }
    if (message instanceof EPPRegistryDeleteCmd) {
      response = doRegistryDelete(aEvent, aData);
    }

    try {
      this.postHandleEvent(aEvent, aData);
    }
    catch (EPPHandleEventException e) {
      return new EPPEventResponse(e.getResponse());
    }

    return response;
  }

  /**
   * Invoked when a Registry Info command is received. Subclasses should define
   * the behavior when a Registry Info command is received.
   * 
   * @param aEvent
   *           The {@link EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPRegistryHandler}
   * 
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected abstract EPPEventResponse doRegistryInfo(EPPEvent aEvent, Object aData);

  /**
   * Invoked when a Registry Check command is received. Subclasses should
   * define the behavior when a Registry Check command is received.
   * 
   * @param aEvent
   *           The {@link EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPRegistryHandler}
   * 
   * @return EPPEventResponse The response that should be sent back to the
   *         client.
   */
  protected abstract EPPEventResponse doRegistryCheck(EPPEvent aEvent, Object aData);

  /**
   * Invoked when a Registry Create command is received. Subclasses should
   * define the behavior when a Registry Create command is received.
   * 
   * @param aEvent
   *           The {@link EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPRegistryHandler}.
   * 
   * @return The {@link EPPEventResponse} that should be sent back to the
   *         client.
   */
  protected abstract EPPEventResponse doRegistryCreate(EPPEvent aEvent, Object aData);

  /**
   * Invoked when a Registry Update command is received. Subclasses should
   * define the behavior when a Registry Update command is received.
   * 
   * @param aEvent
   *           The {@link EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPRegistryHandler}
   * 
   * @return The {@link EPPEventResponse} that should be sent back to the
   *         client.
   */
  protected abstract EPPEventResponse doRegistryUpdate(EPPEvent aEvent, Object aData);

  /**
   * Invoked when a Registry Delete command is received. Subclasses should
   * define the behavior when a Registry Delete command is received.
   * 
   * @param aEvent
   *           The {@link EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPRegistryHandler}
   * 
   * @return The {@link EPPEventResponse} that should be sent back to the
   *         client.
   */
  protected abstract EPPEventResponse doRegistryDelete(EPPEvent aEvent, Object aData);

  /**
   * Gets the XML namespace associated with this handler.
   * 
   * @return {@link EPPRegistryMapFactory#NS}
   */
  public String getNamespace() {
    return EPPRegistryMapFactory.NS;
  }

  /**
   * Handles any common behavior that all registry commands need to execute
   * before they execute their command specific behavior. The default
   * implementation does nothing.
   * 
   * @param aEvent
   *           The {@link EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@code EPPRegistryHandler}
   * 
   * @exception EPPHandleEventException
   *               Thrown if an error condition occurs. It must contain an
   *               {@link EPPEventResponse}
   */
  protected void preHandleEvent(EPPEvent aEvent, Object aData) throws EPPHandleEventException {
  }

  /**
   * Handles any common behavior that all registry commands need to execute
   * after they execute their command specific behavior. The default
   * implementation does nothing
   * 
   * @param aEvent
   *           The {@link EPPEvent} that is being handled
   * @param aData
   *           Any data that a Server needs to send to this
   *           {@link EPPRegistryHandler}
   * 
   * @exception EPPHandleEventException
   *               Thrown if an error condition occurs. It must contain an
   *               {@link EPPEventResponse}
   */
  protected void postHandleEvent(EPPEvent aEvent, Object aData) throws EPPHandleEventException {
  }
}
