/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPRegistryZoneData} contains the {@link EPPRegistryZone} for use in
 * the {@link EPPRegistryInfoResp} to include the {@code accessible} boolean
 * attribute. The {@code accessible} boolean attribute is only applicable in the
 * info response.
 */
public class EPPRegistryZoneData implements EPPCodecComponent {

  /**
   * Logger
   */
      private static Logger cat = LoggerFactory.getLogger(EPPRegistryZoneData.class);
        

  /**
   * XML root tag for {@code EPPRegistryZoneData}.
   */
  public static final String ELM_NAME = EPPRegistryZone.ELM_NAME;

  /**
   * XML attribute name for the {@code accessible} attribute.
   */
  public static final String ATTR_ACCESSIBLE = "accessible";

  /**
   * Zone information that is wrapped by {@code EPPRegistryZoneData}
   */
  EPPRegistryZone zone = null;

  /**
   * Indicates whether the zone is accessible to the client, with a default
   * value of {@code true}.
   */
  boolean accessible = true;

  /**
   * Default constructor for {@code EPPRegistryZoneData}. The {@code zone} must
   * be set prior to calling {{@link #encode(Document)}.
   */
  public EPPRegistryZoneData() {
  }

  /**
   * Default constructor for {@code EPPRegistryZoneData}. The {@code zone} must
   * be set prior to calling {{@link #encode(Document)}.
   * 
   * @param aZone
   *           Contained zone information
   * @param aAccessible
   *           Indicates whether the zone is accessible with {@code true} or
   *           available with {@code false}.
   */
  public EPPRegistryZoneData(EPPRegistryZone aZone, boolean aAccessible) {
    this.zone = aZone;
    this.accessible = aAccessible;
  }

  /**
   * Gets the contained zone information
   *
   * @return The zone information if defined; {@code null} otherwise.
   */
  public EPPRegistryZone getZone() {
    return this.zone;
  }

  /**
   * Sets the contained zone information.
   *
   * @param aZone
   *           Zone information
   */
  public void setZone(EPPRegistryZone aZone) {
    this.zone = aZone;
  }

  /**
   * Is the zone accessible to the client?
   *
   * @return {@code true} if the zone is accessible; {@code false} otherwise.
   */
  public boolean isAccessible() {
    return this.accessible;
  }

  /**
   * Sets if the zone is access to the client.
   *
   * @param aAccessible
   *           {@code true} if the zone is accessible; {@code false} otherwise.
   */
  public void setAccessible(boolean aAccessible) {
    this.accessible = aAccessible;
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPRegistryZoneData} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         {@code EPPRegistryZoneData} instance.
   *
   * @exception EPPEncodeException
   *               - Unable to encode {@code EPPRegistryZoneData} instance.
   */
  @Override
  public Element encode(Document aDocument) throws EPPEncodeException {
    if (this.zone == null) {
      throw new EPPEncodeException("The zone is null");
    }

    Element theRootElm = this.zone.encode(aDocument);

    // Accessible
    EPPUtil.encodeBooleanAttr(theRootElm, ATTR_ACCESSIBLE, this.accessible);

    return theRootElm;
  }

  /**
   * Decode the {@code EPPRegistryZoneData} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPRegistryZoneData} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  public void decode(Element aElement) throws EPPDecodeException {
    this.zone = new EPPRegistryZone();
    this.zone.decode(aElement);

    // Accessible
    if (aElement.getAttribute(ATTR_ACCESSIBLE) != null) {
      this.accessible = EPPUtil.decodeBooleanAttr(aElement, ATTR_ACCESSIBLE);
    }
    else {
      // Set default
      this.accessible = true;
    }
  }

  /**
   * implements a deep {@code EPPRegistryZoneData} compare.
   *
   * @param aObject
   *           {@code EPPRegistryZoneData} instance to compare with
   *
   * @return {@code true} if this object is the same as the aObject argument;
   *         {@code false} otherwise
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPRegistryZoneData)) {
      return false;
    }

    EPPRegistryZoneData theComp = (EPPRegistryZoneData) aObject;

    // Accessible
    if (!EqualityUtil.equals(this.accessible, theComp.accessible)) {
      cat.error("EPPRegistryZoneData.equals(): accessible not equal");
      return false;
    }

    return true;
  }

  /**
   * Clone {@code EPPRegistryZoneData}.
   *
   * @return clone of {@code EPPRegistryZoneData}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPRegistryZoneData clone = (EPPRegistryZoneData) super.clone();

    // Zone
    if (this.zone != null) {
      clone.zone = (EPPRegistryZone) this.zone.clone();
    }

    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   *
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPRegistryMapFactory.NS;
  }

}
