/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Represents the detailed information of a registry system object. Upon
 * receiving an &lt;info&gt; command, with a &lt;registry:system&gt; element in
 * it, the server puts a &lt;registry:system&gt; element in the response. <br>
 * <br>
 *
 * Each element in the list contains the following info:
 *
 * <ul>
 * <li>&lt;registry:maxConnections&gt; - The OPTIONAL attribute that contains
 * the maximum number of connections that the client can establish with the
 * registry system.</li>
 * <li>&lt;registry:idleTimeout&gt; - The OPTIONAL attribute that contains the
 * idle timeout for a connection in milliseconds. If a connection does not
 * receive a command within {@code idleTimeout} milliseconds, the server will
 * close the connection.
 * <li>&lt;registry:absoluteTimeout&gt; - The OPTIONAL attribute that contains
 * the absolute timeout for a connection in milliseconds. The absolute timeout
 * represents the maximum duration in milliseconds that a connection can be
 * established. The server will close a connection that has been established for
 * more than {@code milliseconds} milliseconds.</li>
 * <li>&lt;registry:commandTimeout&gt; - The OPTIONAL attribute that contains
 * the command timeout for a connection in milliseconds. The server will close a
 * connection that has an active command that exceeds {@code commandTimeout}
 * milliseconds.</li>
 * <li>&lt;registry:transLimit&gt; - The OPTIONAL attribute that contains the
 * maximum number of transactions that can be submitted on the connection per
 * the "perMs" attribute milliseconds. It is up to server policy what to do with
 * the connection when the client exceeds the {@code transLimit}.</li>
 * </ul>
 */
public class EPPRegistrySystemInfo implements EPPCodecComponent {

  /**
   * XML local name for {@code EPPRegistrySystemInfo}.
   */
  public static final String ELM_LOCALNAME = "system";

  /**
   * XML root tag for {@code EPPRegistrySystemInfo}.
   */
  public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * XML tag name for the {@code maxConnections} attribute.
   */
  private final static String ELM_MAX_CONNECTIONS = "maxConnections";

  /**
   * XML tag name for the {@code idleTimeout} attribute.
   */
  private final static String ELM_IDLE_TIMEOUT = "idleTimeout";

  /**
   * XML tag name for the {@code absoluteTimeout} attribute.
   */
  private final static String ELM_ABSOLUTE_TIMEOUT = "absoluteTimeout";

  /**
   * XML tag name for the {@code commandTimeout} attribute.
   */
  private final static String ELM_COMMAND_TIMEOUT = "commandTimeout";

  /**
   * XML tag name for the {@code transLimit} attribute.
   */
  private final static String ELM_TRANS_LIMIT = "transLimit";

  /**
   * {@code perMs} attribute.
   */
  private final static String ATTR_PER_MS = "perMs";

  /**
   * Category for logging
   */
      private static Logger cat = LoggerFactory.getLogger(EPPRegistrySystemInfo.class);
        

  /**
   * The OPTIONAL attribute that contains the maximum number of connections
   * that the client can establish with the registry system.
   */
  private Integer maxConnections = null;

  /**
   * The OPTIONAL attribute that contains the idle timeout for a connection in
   * milliseconds. If a connection does not receive a command within
   * {@code idleTimeout} milliseconds, the server will close the connection.
   */
  private Integer idleTimeout = null;

  /**
   * The OPTIONAL attribute that contains the absolute timeout for a connection
   * in milliseconds. The absolute timeout represents the maximum duration in
   * milliseconds that a connection can be established. The server will close a
   * connection that has been established for more than {@code milliseconds}
   * milliseconds.
   */
  private Integer absoluteTimeout = null;

  /**
   * The OPTIONAL attribute that contains the command timeout for a connection
   * in milliseconds. The server will close a connection that has an active
   * command that exceeds {@code commandTimeout} milliseconds.
   */
  private Integer commandTimeout = null;

  /**
   * The OPTIONAL attribute that contains the maximum number of transactions
   * that can be submitted on the connection per the "perMs" attribute
   * milliseconds. It is up to server policy what to do with the connection
   * when the client exceeds the {@code transLimit}.
   */
  private Integer transLimit = null;

  /**
   * Required attribute with the {@code perMs} attribute that defines the
   * number of milliseconds associated with the {@code transLimit} value.
   */
  private Integer perMs = null;

  /**
   * Default constructor.
   */
  public EPPRegistrySystemInfo() {
    super();
  }

  /**
   * Constructor that takes all registry system attributes.
   *
   * @param aMaxConnections
   *           Maximum number of connections that the client can establish with
   *           the registry system. Set to {@code null} if undefined.
   * @param aIdleTimeout
   *           Idle timeout for a connection in milliseconds. Set to
   *           {@code null} if undefined.
   * @param aAbsoluteTimeout
   *           Absolute timeout for a connection in milliseconds. Set to
   *           {@code null} if undefined.
   * @param aCommandTimeout
   *           Command timeout for a connection in milliseconds. Set to
   *           {@code null} if undefined.
   * @param aTransLimit
   *           Maximum number of transactions that can be submitted on the
   *           connection per the "perMs" attribute milliseconds. Set to
   *           {@code null} if undefined.
   * @param aPerMs
   *           Number of milliseconds associated with the {@code transLimit}
   *           attribute. Set to {@code null} if undefined.
   */
  public EPPRegistrySystemInfo(Integer aMaxConnections, Integer aIdleTimeout, Integer aAbsoluteTimeout,
        Integer aCommandTimeout, Integer aTransLimit, Integer aPerMs) {
    this.maxConnections = aMaxConnections;
    this.idleTimeout = aIdleTimeout;
    this.absoluteTimeout = aAbsoluteTimeout;
    this.commandTimeout = aCommandTimeout;
    this.transLimit = aTransLimit;
    this.perMs = aPerMs;
  }

  /**
   * Is the maximum connections set?
   *
   * @return {@code true} if the maximum connections is defined; {@code false}
   *         otherwise.
   */
  public boolean hasMaxConnections() {
    if (this.maxConnections != null) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Gets the maximum number of connections.
   * 
   * @return the maximum connections if defined; {@code null} otherwise.
   */
  public Integer getMaxConnections() {
    return this.maxConnections;
  }

  /**
   * Sets the maximum number of connections.
   * 
   * @param aMaxConnections
   *           the maximum number of connections to set. Set to {@code null} if
   *           undefined.
   */
  public void setMaxConnections(Integer aMaxConnections) {
    this.maxConnections = aMaxConnections;
  }

  /**
   * Is the idle timeout set?
   *
   * @return {@code true} if the idle timeout is defined; {@code false}
   *         otherwise.
   */
  public boolean hasIdleTimeout() {
    if (this.idleTimeout != null) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Gets the idle timeout in milliseconds.
   * 
   * @return the idle timeout in milliseconds if defined; {@code null} 
   *         otherwise.
   */
  public Integer getIdleTimeout() {
    return this.idleTimeout;
  }

  /**
   * Sets the idle timeout in milliseconds.
   * 
   * @param aIdleTimeout
   *           the idle timeout in milliseconds. Set to {@code null} if
   *           undefined.
   */
  public void setIdleTimeout(Integer aIdleTimeout) {
    this.idleTimeout = aIdleTimeout;
  }

  /**
   * Is the absolute timeout set?
   *
   * @return {@code true} if the absolute timeout is defined; {@code false}
   *         otherwise.
   */
  public boolean hasAbsoluteTimeout() {
    if (this.absoluteTimeout != null) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Gets the absolute timeout in milliseconds.
   * 
   * @return the absolute timeout in milliseconds if defined; {@code null}
   *         otherwise.
   */
  public Integer getAbsoluteTimeout() {
    return this.absoluteTimeout;
  }

  /**
   * Sets the absolute timeout in milliseconds.
   * 
   * @param aAbsoluteTimeout
   *           the absolute timeout in milliseconds to set. Set to {@code null}
   *           if undefined.
   */
  public void setAbsoluteTimeout(Integer aAbsoluteTimeout) {
    this.absoluteTimeout = aAbsoluteTimeout;
  }

  /**
   * Is the command timeout set?
   *
   * @return {@code true} if the command timeout is defined; {@code false}
   *         otherwise.
   */
  public boolean hasCommandTimeout() {
    if (this.commandTimeout != null) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Gets the command timeout in milliseconds.
   * 
   * @return the command timeout in milliseconds if defined; {@code null}
   *         otherwise.
   */
  public Integer getCommandTimeout() {
    return this.commandTimeout;
  }

  /**
   * Sets the command timeout in milliseconds.
   * 
   * @param aCommandTimeout
   *           the command timeout to set. Set to {@code null} if undefined.
   */
  public void setCommandTimeout(Integer aCommandTimeout) {
    this.commandTimeout = aCommandTimeout;
  }

  /**
   * Is the transaction limit is set?
   *
   * @return {@code true} if the transaction limit is defined; {@code false}
   *         otherwise.
   */
  public boolean hasTransLimit() {
    if (this.transLimit != null) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Gets the transaction limit. If the transaction limit is set, the
   * {@code perMs} attribute must be set.
   * 
   * @return the transaction limit if defined; {@code null} otherwise.
   */
  public Integer getTransLimit() {
    return this.transLimit;
  }

  /**
   * Sets the transaction limit. If the transaction limit is set, the
   * {@code perMs} attribute must be set.
   * 
   * @param aTransLimit
   *           the transaction limit to set. Set to {@code null} if undefined.
   */
  public void setTransLimit(Integer aTransLimit) {
    this.transLimit = aTransLimit;
  }

  /**
   * Sets the transaction limit with the {@code perMs} milliseconds.
   * 
   * @param aTransLimit
   *           the transaction limit to set. Set to {@code null} if undefined.
   * @param aPerMs
   *           Milliseconds that the transaction limit is associated with. Set
   *           to {@code null} if undefined.
   */
  public void setTransLimit(Integer aTransLimit, Integer aPerMs) {
    this.transLimit = aTransLimit;
    this.perMs = aPerMs;
  }

  /**
   * Is the per milliseconds attribute set?
   *
   * @return {@code true} if the per milliseconds attribute is defined;
   *         {@code false} otherwise.
   */
  public boolean hasPerMs() {
    if (this.perMs != null) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Gets the milliseconds that the transaction limit is associated with.
   * 
   * @return the per milliseconds if defined; {@code null} otherwise.
   */
  public Integer getPerMs() {
    return this.perMs;
  }

  /**
   * Sets the milliseconds that the transaction limit is associated with.
   * 
   * @param aPerMs
   *           per milliseconds to set. Set to {@code null} if undefined.
   */
  public void setPerMs(Integer aPerMs) {
    this.perMs = aPerMs;
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPRegistrySystemInfo} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         {@code EPPRegistrySystemInfo} instance.
   *
   * @exception EPPEncodeException
   *               - Unable to encode {@code EPPRegistrySystemInfo} instance.
   */
  @Override
  public Element encode(Document aDocument) throws EPPEncodeException {

    if (aDocument == null) {
      throw new EPPEncodeException("aDocument is null" + " on in EPPRegistrySystemInfo.encode(Document)");
    }

    if (this.hasTransLimit() && !this.hasPerMs()) {
      throw new EPPEncodeException(
            "Transaction limit is set without the per milliseconds attribute in EPPRegistrySystemInfo.encode(Document)");
    }

    Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

    // Maximum Connections
    EPPUtil.encodeInteger(aDocument, root, this.maxConnections, EPPRegistryMapFactory.NS,
          EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_MAX_CONNECTIONS);

    // Idle Timeout
    EPPUtil.encodeInteger(aDocument, root, this.idleTimeout, EPPRegistryMapFactory.NS,
          EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_IDLE_TIMEOUT);

    // Absolute Timeout
    EPPUtil.encodeInteger(aDocument, root, this.absoluteTimeout, EPPRegistryMapFactory.NS,
          EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_ABSOLUTE_TIMEOUT);

    // Command Timeout
    EPPUtil.encodeInteger(aDocument, root, this.commandTimeout, EPPRegistryMapFactory.NS,
          EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_COMMAND_TIMEOUT);

    // Transaction Limit
    if (this.hasTransLimit()) {
      Element theElm = aDocument.createElementNS(EPPRegistryMapFactory.NS,
            EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_TRANS_LIMIT);

      theElm.setAttribute(ATTR_PER_MS, this.perMs.toString());

      Text transLimitText = aDocument.createTextNode(this.transLimit.toString());
      theElm.appendChild(transLimitText);

      root.appendChild(theElm);
    }

    return root;
  }

  /**
   * Decode the {@code EPPRegistrySystemInfo} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPRegistrySystemInfo} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  public void decode(Element aElement) throws EPPDecodeException {

    // Maximum Connections
    this.maxConnections = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, ELM_MAX_CONNECTIONS);

    // Idle Timeout
    this.idleTimeout = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, ELM_IDLE_TIMEOUT);

    // Absolute Timeout
    this.absoluteTimeout = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, ELM_ABSOLUTE_TIMEOUT);

    // Command Timeout
    this.commandTimeout = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, ELM_COMMAND_TIMEOUT);

    // Transaction Limit
    Element theTransLimitElm = EPPUtil.getElementByTagNameNS(aElement, EPPRegistryMapFactory.NS, ELM_TRANS_LIMIT);
    if (theTransLimitElm != null) {
      Node theTextNode = theTransLimitElm.getFirstChild();
      if (theTextNode != null) {
        this.transLimit = Integer.valueOf(theTextNode.getNodeValue());

        // Per Milliseconds
        String thePerMs = theTransLimitElm.getAttribute(ATTR_PER_MS);
        if (thePerMs != null && !thePerMs.isEmpty()) {
          this.perMs = Integer.valueOf(thePerMs);
        }
        else {
          this.perMs = null;
        }
      }

    }
    else {
      this.transLimit = null;
      this.perMs = null;
    }
  }

  /**
   * implements a deep {@code EPPRegistrySystemInfo} compare.
   *
   * @param aObject
   *           {@code EPPRegistrySystemInfo} instance to compare with
   *
   * @return {@code true} if this object is the same as the aObject argument;
   *         {@code false} otherwise
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPRegistrySystemInfo)) {
      return false;
    }

    EPPRegistrySystemInfo theComp = (EPPRegistrySystemInfo) aObject;

    // Maximum Connections
    if (!EqualityUtil.equals(this.maxConnections, theComp.maxConnections)) {
      cat.error("EPPRegistrySystemInfo.equals(): maxConnections not equal");
      return false;
    }

    // Idle Timeout
    if (!EqualityUtil.equals(this.idleTimeout, theComp.idleTimeout)) {
      cat.error("EPPRegistrySystemInfo.equals(): idleTimeout not equal");
      return false;
    }

    // Absolute Timeout
    if (!EqualityUtil.equals(this.absoluteTimeout, this.absoluteTimeout)) {
      cat.error("EPPRegistrySystemInfo.equals(): absoluteTimeout not equal");
      return false;
    }

    // Command Timeout
    if (!EqualityUtil.equals(this.commandTimeout, theComp.commandTimeout)) {
      cat.error("EPPRegistrySystemInfo.equals(): commandTimeout not equal");
      return false;
    }

    // Transaction Limit
    if (!EqualityUtil.equals(this.transLimit, theComp.transLimit)) {
      cat.error("EPPRegistrySystemInfo.equals(): transLimit not equal");
      return false;
    }

    // Per Milliseconds
    if (!EqualityUtil.equals(this.perMs, theComp.perMs)) {
      cat.error("EPPRegistrySystemInfo.equals(): perMs not equal");
      return false;
    }

    return true;
  }

  /**
   * Clone {@code EPPRegistrySystemInfo}.
   *
   * @return clone of {@code EPPRegistrySystemInfo}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPRegistrySystemInfo clone = (EPPRegistrySystemInfo) super.clone();
    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   *
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPRegistryMapFactory.NS;
  }

}
