/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.registry.v02;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Represents the postal-address information policy information. The
 * &lt;registry:postalInfo&gt; element contains the following child elements:
 * <br>
 * <ul>
 * <li>&lt;registry:locCharRegex&gt; - The OPTIONAL regular expression that
 * represents the character set that can be used for the
 * &lt;contact:postalInfo&gt; localized form (type="loc") element content. The
 * regular expression MUST be applicable to all &lt;contact:postalInfo&gt;
 * element content.</li>
 * <li>&lt;registry:name&gt; - The minimum and maximum length of
 * &lt;contact:name&gt; element defined RFC 5733 using the
 * &lt;registry:minLength&gt; and &lt;registry:maxLength&gt; child elements,
 * respectively.</li>
 * <li>&lt;registry:org&gt; - The minimum and maximum length of the
 * &lt;contact:org&gt; element defined in RFC 5733 using the
 * &lt;registry:minLength&gt; and &lt;registry:maxLength&gt; child elements,
 * respectively.</li>
 * <li>&lt;registry:address&gt; - The address information policy information.
 * </li>
 * <li>&lt;registry:voiceRequired&gt; - An OPTIONAL boolean flag indicating
 * whether the server requires the &lt;contact:voice&gt; element to be defined,
 * with a default value of "false".</li>
 * <li>&lt;registry:voiceExt&gt; - The OPTIONAL minimum and maximum length of
 * the &lt;contact:voice&gt; extension "x" attribute defined in RFC 5733 using
 * the &lt;registry:minLength&gt; and &lt;registry:maxLength&gt; child elements,
 * respectively.</li>
 * <li>&lt;registry:faxExt&gt; - The OPTIONAL minimum and maximum length of the
 * &lt;contact:fax&gt; extension "x" attribute defined in RFC 5733 [RFC5733]
 * using the &lt;registry:minLength&gt; and &lt;registry:maxLength&gt; child
 * elements, respectively.</li>
 * <li>&lt;registry:emailRegex&gt; - An OPTIONAL &lt;registry:emailRegex&gt;
 * element that defines the regular expression used to validate the
 * &lt;contact:email&gt; element defined in [RFC5733]</li>
 * </ul>
 *
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryContact
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryContactName
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryContactOrg
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryContactAddress
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryRegex
 */
public class EPPRegistryPostal implements EPPCodecComponent {

  /**
   * Category for logging
   */
  private static Logger cat =  LoggerFactory.getLogger(EPPRegistryPostal.class);

  /**
   * Constant for the local name
   */
  public static final String ELM_LOCALNAME = "postalInfo";

  /**
   * Constant for the prefix and local name
   */
  public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * XML local name for {@code locCharRegex} attribute.
   */
  public static final String ELM_LOC_CHAR_REGEX = "locCharRegex";

  /**
   * XML local name for {@code voiceRequired} attribute.
   */
  public static final String ELM_VOICE_REQUIRED = "voiceRequired";

  /**
   * XML local name for {@code emailRegex} attribute.
   */
  public static final String ELM_EMAIL_REGEX = "emailRegex";

  /**
   * XML local name for {@code voiceExt} attribute.
   */
  public static final String ELM_VOICE_EXT = "voiceExt";

  /**
   * XML local name for {@code faxExt} attribute.
   */
  public static final String ELM_FAX_EXT = "faxExt";

  /**
   * Localized "loc" postal information field regular expression
   */
  private EPPRegistryRegex locCharRegex = null;

  /**
   * Contact name
   */
  private EPPRegistryContactName name = null;

  /**
   * Contact organization
   */
  private EPPRegistryContactOrg org = null;

  /**
   * Contact address
   */
  private EPPRegistryContactAddress address = null;

  /**
   * Is the contact voice required?
   */
  private Boolean voiceRequired = Boolean.FALSE;

  /**
   * The OPTIONAL minimum and maximum length of the &lt;contact:voice&gt;
   * extension "x" attribute.
   */
  private EPPRegistryAbstractMinMax voiceExt = null;

  /**
   * The OPTIONAL minimum and maximum length of the &lt;contact:fax&gt;
   * extension "x" attribute.
   */
  private EPPRegistryAbstractMinMax faxExt = null;

  /**
   * Email regular expression
   */
  private EPPRegistryRegex emailRegex = null;

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPRegistryPostal} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         {@code EPPRegistryPostal} instance.
   *
   * @exception EPPEncodeException
   *               - Unable to encode {@code EPPRegistryPostal} instance.
   */
  @Override
  public Element encode(Document aDocument) throws EPPEncodeException {
    try {
      validateState();
    }
    catch (EPPCodecException e) {
      throw new EPPEncodeException("Invalid state on EPPRegistryPostal.encode: " + e);
    }
    Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

    // locCharRegex
    if (this.locCharRegex != null) {
      EPPUtil.encodeComp(aDocument, root, this.locCharRegex);
    }

    // name
    EPPUtil.encodeComp(aDocument, root, this.name);

    // org
    EPPUtil.encodeComp(aDocument, root, this.org);

    // address
    EPPUtil.encodeComp(aDocument, root, this.address);
    if (this.voiceRequired == null) {
      this.voiceRequired = Boolean.FALSE;
    }

    // voiceRequired
    EPPUtil.encodeString(aDocument, root, this.voiceRequired.toString(), EPPRegistryMapFactory.NS,
          EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_VOICE_REQUIRED);

    // voiceExt
    EPPUtil.encodeComp(aDocument, root, this.voiceExt);

    // faxExt
    EPPUtil.encodeComp(aDocument, root, this.faxExt);

    // emailRegex
    if (this.emailRegex != null) {
      EPPUtil.encodeComp(aDocument, root, this.emailRegex);
    }

    return root;
  }

  /**
   * Decode the {@code EPPRegistryPostal} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPRegistryPostal} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  public void decode(Element aElement) throws EPPDecodeException {

    // locCharRegex
    this.setLocCharRegex((EPPRegistryRegex) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS, ELM_LOC_CHAR_REGEX,
          EPPRegistryRegex.class));

    // name
    this.setName((EPPRegistryContactName) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
          EPPRegistryContactName.ELM_NAME, EPPRegistryContactName.class));

    // org
    this.setOrg((EPPRegistryContactOrg) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
          EPPRegistryContactOrg.ELM_NAME, EPPRegistryContactOrg.class));

    // address
    this.address = (EPPRegistryContactAddress) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
          EPPRegistryContactAddress.ELM_NAME, EPPRegistryContactAddress.class);

    // voiceRequired
    this.voiceRequired = EPPUtil.decodeBoolean(aElement, EPPRegistryMapFactory.NS, ELM_VOICE_REQUIRED);
    if (this.voiceRequired == null) {
      this.voiceRequired = Boolean.FALSE;
    }

    // voiceExt
    this.setVoiceExt((EPPRegistryMinMaxLength) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS, ELM_VOICE_EXT,
          EPPRegistryMinMaxLength.class));

    // faxExt
    this.setFaxExt((EPPRegistryMinMaxLength) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS, ELM_FAX_EXT,
          EPPRegistryMinMaxLength.class));

    // emailRegex
    this.setEmailRegex((EPPRegistryRegex) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS, ELM_EMAIL_REGEX,
          EPPRegistryRegex.class));
  }

  /**
   * Validate the state of the <code>EPPRegistryPostal</code> instance. A valid
   * state means that all of the required attributes have been set. If
   * validateState returns without an exception, the state is valid. If the
   * state is not valid, the EPPCodecException will contain a description of
   * the error. throws EPPCodecException State error. This will contain the
   * name of the attribute that is not valid.
   *
   * @throws EPPCodecException
   */
  void validateState() throws EPPCodecException {
    if (this.name == null) {
      throw new EPPCodecException("name required element is not set");
    }
    if (this.org == null) {
      throw new EPPCodecException("org required element is not set");
    }
    if (this.address == null) {
      throw new EPPCodecException("address required element is not set");
    }
  }

  /**
   * Clone <code>EPPRegistryPostal</code>.
   *
   * @return clone of <code>EPPRegistryPostal</code>
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPRegistryPostal clone = (EPPRegistryPostal) super.clone();

    // locCharRegex
    if (this.hasLocCharRegex()) {
      clone.locCharRegex = (EPPRegistryRegex) this.locCharRegex.clone();
    }

    // name
    if (this.name != null) {
      clone.name = (EPPRegistryContactName) this.name.clone();
    }

    // org
    if (this.hasOrg()) {
      clone.org = (EPPRegistryContactOrg) this.org.clone();
    }

    // address
    if (this.address != null) {
      clone.address = (EPPRegistryContactAddress) this.address.clone();
    }

    // voiceExt
    if (this.hasVoiceExt()) {
      clone.voiceExt = (EPPRegistryAbstractMinMax) this.voiceExt.clone();
    }

    // faxExt
    if (this.hasFaxExt()) {
      clone.faxExt = (EPPRegistryAbstractMinMax) this.faxExt.clone();
    }

    // emailRegex
    if (this.hasEmailRegex()) {
      clone.emailRegex = (EPPRegistryRegex) this.emailRegex.clone();
    }

    return clone;
  }

  /**
   * implements a deep <code>EPPRegistryPostal</code> compare.
   *
   * @param aObject
   *           <code>EPPRegistryPostal</code> instance to compare with
   *
   * @return {@code true} if this object is the same as the aObject argument;
   *         {@code false} otherwise
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPRegistryPostal)) {
      return false;
    }

    EPPRegistryPostal theComp = (EPPRegistryPostal) aObject;

    // locCharRegex
    if (!EqualityUtil.equals(this.locCharRegex, theComp.locCharRegex)) {
      cat.error("EPPRegistryPostal.equals(): locCharRegex not equal");
      return false;
    }

    // name
    if (!EqualityUtil.equals(this.name, theComp.name)) {
      cat.error("EPPRegistryPostal.equals(): name not equal");
      return false;
    }

    // org
    if (!EqualityUtil.equals(this.org, theComp.org)) {
      cat.error("EPPRegistryPostal.equals(): org not equal");
      return false;
    }

    // address
    if (!EqualityUtil.equals(this.address, theComp.address)) {
      cat.error("EPPRegistryPostal.equals(): address not equal");
      return false;
    }

    // voiceRequired
    if (!EqualityUtil.equals(this.voiceRequired, theComp.voiceRequired)) {
      cat.error("EPPRegistryPostal.equals(): voiceRequired not equal");
      return false;
    }

    // voiceExt
    if (!EqualityUtil.equals(this.voiceExt, theComp.voiceExt)) {
      cat.error("EPPRegistryPostal.equals(): voiceExt not equal");
      return false;
    }

    // faxExt
    if (!EqualityUtil.equals(this.faxExt, theComp.faxExt)) {
      cat.error("EPPRegistryPostal.equals(): faxExt not equal");
      return false;
    }

    // emailRegex
    if (!EqualityUtil.equals(this.emailRegex, theComp.emailRegex)) {
      cat.error("EPPRegistryPostal.equals(): emailRegex not equal");
      return false;
    }

    return true;
  }

  /**
   * Implementation of <code>Object.toString</code>, which will result in an
   * indented XML <code>String</code> representation of the concrete
   * <code>EPPCodecComponent</code>.
   *
   * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Is the locality "loc" character field regular expression defined?
   *
   * @return {@code true} if the locality character field regular expression is
   *         defined; {@code false} otherwise.
   */
  public boolean hasLocCharRegex() {
    return (this.locCharRegex != null ? true : false);
  }

  /**
   * Gets info about locality "loc" character field regular expression.
   *
   * @return instance of {@link EPPRegistryRegex} that specifies regular
   *         expression used for the locality "loc" fields if defined;
   *         {@code null} otherwise.
   */
  public EPPRegistryRegex getLocCharRegex() {
    return this.locCharRegex;
  }

  /**
   * Sets info about locality "loc" character field regular expression.
   *
   * @param aLocCharRegex
   *           instance of {@link EPPRegistryRegex} that specifies regular
   *           expression used for the locality "loc" fields.
   */
  public void setLocCharRegex(EPPRegistryRegex aLocCharRegex) {
    if (aLocCharRegex != null) {
      aLocCharRegex.setRootName(ELM_LOC_CHAR_REGEX);
    }
    this.locCharRegex = aLocCharRegex;
  }

  /**
   * Gets the minimum and maximum length of name.
   *
   * @return {@code EPPRegistryContactName} instance that contains min/max
   *         length of contact name
   */
  public EPPRegistryContactName getName() {
    return this.name;
  }

  /**
   * Sets the minimum and maximum length of name.
   *
   * @param name
   *           {@code EPPRegistryContactName} instance that contains min/max
   *           length of contact name
   */
  public void setName(EPPRegistryContactName name) {
    this.name = name;
  }

  /**
   * Is the organization "Org" defined?
   *
   * @return {@code true} if the organization "Org" is defined; {@code false}
   *         otherwise.
   */
  public boolean hasOrg() {
    return (this.org != null ? true : false);
  }

  /**
   * Gets the minimum and maximum length of organization.
   *
   * @return {@code EPPRegistryContactOrg} instance that contains min/max
   *         length of contact organization
   */
  public EPPRegistryContactOrg getOrg() {
    return this.org;
  }

  /**
   * Sets the minimum and maximum length of organization.
   *
   * @param org
   *           {@code EPPRegistryContactOrg} instance that contains min/max
   *           length of contact organization
   */
  public void setOrg(EPPRegistryContactOrg org) {
    this.org = org;
  }

  /**
   * Gets address.
   *
   * @return {@code EPPRegistryContactAddress} instance that contains contact
   *         address attributes
   */
  public EPPRegistryContactAddress getAddress() {
    return this.address;
  }

  /**
   * Sets address.
   *
   * @param address
   *           {@code EPPRegistryContactAddress} instance that contains contact
   *           address attributes
   */
  public void setAddress(EPPRegistryContactAddress address) {
    this.address = address;
  }

  /**
   * Is the voice required flag defined?
   *
   * @return {@code true} if the voice required flag is defined; {@code false}
   *         otherwise.
   */
  public boolean hasVoiceRequired() {
    return (this.voiceRequired != null ? true : false);
  }

  /**
   * Gets voice required flag.
   *
   * @return {@code true} if voice is required. {@code false} otherwise.
   */
  public Boolean getVoiceRequired() {
    return this.voiceRequired;
  }

  /**
   * Sets voice required flag.
   *
   * @param voiceRequired
   *           {@code true} if voice is required. {@code false} otherwise.
   */
  public void setVoiceRequired(Boolean voiceRequired) {
    this.voiceRequired = voiceRequired;
  }

  /**
   * Is the voice extension defined?
   *
   * @return {@code true} if the voice extension is defined; {@code false}
   *         otherwise.
   */
  public boolean hasVoiceExt() {
    return (this.voiceExt != null ? true : false);
  }

  /**
   * Gets the optional voice extension minimum and maximum length.
   *
   * @return <code>EPPRegistryMinMaxLength</code> instance containing the
   *         minimum and maximum length if defined; <code>null</code> otherise.
   */
  public EPPRegistryAbstractMinMax getVoiceExt() {
    return this.voiceExt;
  }

  /**
   * Sets the optional voice extension minimum and maximum length.
   *
   * @param aVoiceExt
   *           <code>EPPRegistryMinMaxLength</code> instance containing the
   *           minimum and maximum length.
   */
  public void setVoiceExt(EPPRegistryMinMaxLength aVoiceExt) {
    if (aVoiceExt != null) {
      aVoiceExt.setRootName(ELM_VOICE_EXT);
    }
    this.voiceExt = aVoiceExt;
  }

  /**
   * Is the fax extension defined?
   *
   * @return {@code true} if the fax extension is defined; {@code false}
   *         otherwise.
   */
  public boolean hasFaxExt() {
    return (this.faxExt != null ? true : false);
  }

  /**
   * Gets the optional fax extension minimum and maximum length.
   *
   * @return <code>EPPRegistryMinMaxLength</code> instance containing the
   *         minimum and maximum length if defined; <code>null</code> otherise.
   */
  public EPPRegistryAbstractMinMax getFaxExt() {
    return this.faxExt;
  }

  /**
   * Sets the optional fax extension minimum and maximum length.
   *
   * @param aFaxExt
   *           <code>EPPRegistryMinMaxLength</code> instance containing the
   *           minimum and maximum length.
   */
  public void setFaxExt(EPPRegistryMinMaxLength aFaxExt) {
    if (aFaxExt != null) {
      aFaxExt.setRootName(ELM_FAX_EXT);
    }
    this.faxExt = aFaxExt;
  }

  /**
   * Is the email regular expression defined?
   *
   * @return {@code true} if the email regular expression is defined;
   *         {@code false} otherwise.
   */
  public boolean hasEmailRegex() {
    return (this.emailRegex != null ? true : false);
  }

  /**
   * Get info about regular expression used to validate the contact email.
   *
   * @return instance of {@link EPPRegistryRegex} that specifies regular
   *         expression used to validate the email if defined; {@code null}
   *         otherwise.
   */
  public EPPRegistryRegex getEmailRegex() {
    return this.emailRegex;
  }

  /**
   * Set info about regular expression used to validate the contact email.
   *
   * @param aEmailRegex
   *           instance of {@link EPPRegistryRegex} that specifies regular
   *           expression used to validate the contact email.
   */
  public void setEmailRegex(EPPRegistryRegex aEmailRegex) {
    if (aEmailRegex != null) {
      aEmailRegex.setRootName(ELM_EMAIL_REGEX);
    }
    this.emailRegex = aEmailRegex;
  }

  /**
   * Returns the XML namespace associated with the
   * <code>EPPCodecComponent</code>.
   *
   * @return XML namespace for the <code>EPPCodecComponent</code>.
   */
  @Override
  public String getNamespace() {
    return EPPRegistryMapFactory.NS;
  }

}
