/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import com.verisign.epp.codec.gen.EPPCodecException;

/**
 * Defines the length of time that the domain object will remain in the
 * pendingDelete status prior to be purged.
 *
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryRGP
 */
public class EPPRegistryPendingDeletePeriodType extends EPPRegistryPeriodType {

  /**
   * XML local name for {@code EPPRegistryPendingDeletePeriodType}.
   */
  public static final String ELM_LOCALNAME = "pendingDelete";

  /**
   * Default constructor. Must call {@link #setNumber(Integer)} and
   * {@link #setUnit(EPPRegistryPeriodType.Unit)} before calling
   * {@link #encode(org.w3c.dom.Document)} method.
   */
  public EPPRegistryPendingDeletePeriodType() {
  }

  /**
   * Constructs an instance of {@code EPPRegistryPendingDeletePeriodType} with
   * {@code number} and {@code unit}.
   *
   * @param aNumber
   *           number must be &gt; 0
   * @param aUnit
   *           unit must be one of {@code Unit.y}, {@code Unit.m},
   *           {@code Unit.d}, or {@code Unit.h}.
   */
  public EPPRegistryPendingDeletePeriodType(Integer aNumber, Unit aUnit) {
    super(aNumber, aUnit);
  }

  /**
   * Constructs an instance of {@code EPPRegistryPendingDeletePeriodType} with
   * {@code number} and {@code unit}.
   *
   * @param aNumber
   *           number must be  &gt; 0
   * @param aUnit
   *           unit must be one of {@code Unit.y}, {@code Unit.m},
   *           {@code Unit.d}, or {@code Unit.h}.
   */
  public EPPRegistryPendingDeletePeriodType(int aNumber, Unit aUnit) {
    super(aNumber, aUnit);
  }

  /**
   * Extra validation rules on top of
   * {@link EPPRegistryPendingDeletePeriodType}: {@code number} must be greater
   * than "0"; {@code unit} must be one of {@code Unit.y}, {@code Unit.m},
   * {@code Unit.d}, or {@code Unit.h}.
   *
   * @throws EPPCodecException
   *            Validation error
   */
  @Override
  void extraValidate() throws EPPCodecException {
    // Number
    int n = this.number.intValue();
    if (n <= 0) {
      throw new EPPCodecException(getRootName() + ": number should be greater than 0");
    }

    // Number
    if (super.unit != Unit.y && super.unit != Unit.m && super.unit != Unit.d && super.unit != Unit.h) {
      throw new EPPCodecException(getRootName() + ": invalid unit. Valid values:  y/m/d/h");
    }
  }

  /**
   * Returns the XML namespace associated with the
   * <code>EPPCodecComponent</code>.
   *
   * @return XML namespace for the <code>EPPCodecComponent</code>.
   */
  @Override
  public String getNamespace() {
    return EPPRegistryMapFactory.NS;
  }

  /**
   * Gets the root element local name.
   *
   * @return Root element local name.
   */
  @Override
  protected String getRootName() {
    return ELM_LOCALNAME;
  }

}
