/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPRegistryLanguage} defines the supported language codes and
 * character code point policy.
 */
public class EPPRegistryLanguage implements EPPCodecComponent {
  /**
   * Logger
   */
      private static Logger cat = LoggerFactory.getLogger(EPPRegistryLanguage.class);
        

  /**
   * XML local name for {@code EPPRegistryLanguage}.
   */
  public static final String ELM_LOCALNAME = "language";

  /**
   * XML root tag for {@code EPPRegistryLanguage}.
   */
  public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * XML local name for the {@code table} attribute.
   */
  final static String ELM_TABLE_NAME = "table";

  /**
   * XML local name for the {@code variantStrategy} attribute.
   */
  final static String ELM_VARIANT_STRATEGY_NAME = "variantStrategy";

  /**
   * Attribute name for the {@code code} attribute.
   */
  final static String ATTR_CODE = "code";

  /**
   * Possible values for the {@code variantStrategy} attribute.
   */
  public static enum VariantStrategy {

    /**
     * Variant registrations are blocked for all clients
     */
    blocked,

    /**
     * Variant registrations are allowed for client of the original IDN
     * registration.
     */
    restricted,

    /**
     * Variant registrations are open to all clients.
     */
    open
  }

  /**
   * The OPTIONAL language table URI that contains the set of code points for
   * the language.
   */
  private String table = null;

  /**
   * An OPTIONAL strategy for the handling of variants for the language. If no
   * &lt;registry:variantStrategy&gt; element is specified then variants are
   * not supported by the language.
   */
  private VariantStrategy variantStrategy = null;

  /**
   * The required "code" attribute defines the language code for the supported
   * language. The language code SHOULD be an ISO 639 (ISO 639-1 or ISO 639-2)
   * value.
   */
  private String code = null;

  /**
   * Default constructor for {@code EPPRegistryLanguage}.
   */
  public EPPRegistryLanguage() {
  }

  /**
   * {@code EPPRegistryLanguage} constructor that takes all attributes.
   *
   * @param aCode
   *           The required language code that should be a ISO 639 (ISO 639-1
   *           or ISO 639-2) value.
   * @param aTable
   *           The OPTIONAL language table URI. Set to {@code null} if
   *           undefined.
   * @param aVariantStrategy
   *           The OPTIONAL variant strategy. Set to {@code null} if undefined.
   */
  public EPPRegistryLanguage(String aCode, String aTable, VariantStrategy aVariantStrategy) {
    this.code = aCode;
    this.table = aTable;
    this.variantStrategy = aVariantStrategy;
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPRegistryLanguage} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         {@code EPPRegistryLanguage} instance.
   *
   * @exception EPPEncodeException
   *               - Unable to encode {@code EPPRegistryLanguage} instance.
   */
  @Override
  public Element encode(Document aDocument) throws EPPEncodeException {
    // Validate state
    if (this.code == null || this.code.trim().length() == 0) {
      throw new EPPEncodeException(
            "Invalide state on EPPRegistryLanguage.encode: " + "attribute code is required.");
    }

    Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

    // Code
    root.setAttribute(ATTR_CODE, this.code);

    // Table
    if (this.table != null) {
      EPPUtil.encodeString(aDocument, root, this.table, EPPRegistryMapFactory.NS,
            EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_TABLE_NAME);
    }

    // Variant Strategy
    if (this.variantStrategy != null) {
      EPPUtil.encodeString(aDocument, root, this.variantStrategy.toString(), EPPRegistryMapFactory.NS,
            EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_VARIANT_STRATEGY_NAME);
    }

    return root;
  }

  /**
   * Decode the {@code EPPRegistryLanguage} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPRegistryLanguage} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  public void decode(Element aElement) throws EPPDecodeException {
    // Code
    this.code = aElement.getAttribute(ATTR_CODE);

    // Table
    this.table = EPPUtil.decodeString(aElement, EPPRegistryMapFactory.NS, ELM_TABLE_NAME);

    // Variant Strategy
    String theStrategyStr = EPPUtil.decodeString(aElement, EPPRegistryMapFactory.NS, ELM_VARIANT_STRATEGY_NAME);
    if (theStrategyStr == null) {
      this.variantStrategy = null;
    }
    else {
      this.variantStrategy = VariantStrategy.valueOf(theStrategyStr);
    }
  }

  /**
   * Clone {@code EPPRegistryLanguage}.
   *
   * @return clone of {@code EPPRegistryLanguage}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    return super.clone();
  }

  /**
   * implements a deep {@code EPPRegistryLanguage} compare.
   *
   * @param aObject
   *           {@code EPPRegistryLanguage} instance to compare with
   *
   * @return {@code true} if this object is the same as the aObject argument;
   *         {@code false} otherwise
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPRegistryLanguage)) {
      return false;
    }

    EPPRegistryLanguage theComp = (EPPRegistryLanguage) aObject;

    // Code
    if (!EqualityUtil.equals(this.code, theComp.code)) {
      cat.error("EPPRegistryLanguage.equals(): code not equal");
      return false;
    }

    // Table
    if (!EqualityUtil.equals(this.table, theComp.table)) {
      cat.error("EPPRegistryLanguage.equals(): table not equal");
      return false;
    }

    // Variant Strategy
    if (!EqualityUtil.equals(this.variantStrategy, theComp.variantStrategy)) {
      cat.error("EPPRegistryLanguage.equals(): variantStrategy not equal");
      return false;
    }

    return true;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Gets the OPTIONAL language table URI that contains the set of code points
   * for the language.
   *
   * @return The OPTIONAL langauge table URI if defined; {@code null}
   *         otherwise.
   */
  public String getTable() {
    return this.table;
  }

  /**
   * Sets the OPTIONAL language table URI that contains the set of code points
   * for the language.
   *
   * @param aTable
   *           The OPTIONAL langauge table URI
   */
  public void setTable(String aTable) {
    this.table = aTable;
  }

  /**
   * Gets the OPTIONAL strategy for the handling of variants for the language.
   * If no &lt;registry:variantStrategy&gt; element is specified then variants
   * are not supported by the language.
   *
   * @return The OPTIONAL variant strategy if defined; {@code null} otherwise.
   */
  public VariantStrategy getVariantStrategy() {
    return this.variantStrategy;
  }

  /**
   * Sets the OPTIONAL strategy for the handling of variants for the language.
   * If no &lt;registry:variantStrategy&gt; element is specified then variants
   * are not supported by the language.
   *
   * @param aVariantStrategy
   *           The OPTIONAL variant strategy
   */
  public void setVariantStrategy(VariantStrategy aVariantStrategy) {
    this.variantStrategy = aVariantStrategy;
  }

  /**
   * Gets the required "code" attribute defines the language code for the
   * supported language. The language code SHOULD be an ISO 639 (ISO 639-1 or
   * ISO 639-2) value.
   *
   * @return The language code if defined; {@code null} otherwise.
   */
  public String getCode() {
    return this.code;
  }

  /**
   * Sets the required "code" attribute defines the language code for the
   * supported language. The language code SHOULD be an ISO 639 (ISO 639-1 or
   * ISO 639-2) value.
   *
   * @param aCode
   *           The language code
   */
  public void setCode(String aCode) {
    this.code = aCode;
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   *
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPRegistryMapFactory.NS;
  }

}
