/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;


/**
 * Represents an EPP Registry &lt;registry:infData&gt; response to an
 * {@code EPPRegistryInfoCmd}. When an &lt;info&gt; command has been processed
 * successfully, the EPP &lt;resData&gt; element MUST contain a child
 * &lt;registry:infData&gt; element that identifies the registry namespace and
 * the location of the registry schema. <br>
 * <br>
 *
 * If the corresponding registry create command contains &lt;registry:all&gt;
 * element, The &lt;registry:infData&gt; element must contain the following
 * child elements: <br>
 * <br>
 *
 * <ul>
 * <li>A &lt;registry:zoneList&gt; element that contains the list of supported
 * zones by the server with a set of summary attributes per zone. Each set of
 * summary attributes are enclosed in the &lt;registry:zone&gt; element.</li>
 * </ul>
 * <br>
 * If the corresponding registry create command contains &lt;registry:name&gt;
 * element, The &lt;registry:infData&gt; element must contain the following
 * child elements: <br>
 * <br>
 *
 * <ul>
 * <li>A &lt;registry:zone&gt; element that contains the detail info of a zone
 * object.</li>
 * </ul>
 *
 * <br>
 * <br>
 *
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryInfoCmd
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryZoneList
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryZoneSummary
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryZoneData
 */
public class EPPRegistryInfoResp extends EPPResponse {

  /**
   * XML local name for {@code EPPRegistryInfoResp}.
   */
  public static final String ELM_LOCALNAME = "infData";

  /**
   * XML root tag for {@code EPPRegistryInfoResp}.
   */
  public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * {@code List} of zone summary attributes.
   */
  private EPPRegistryZoneList zoneList = null;

  /**
   * {@code zoneInfo} instance that contains detailed zone info.
   */
  private EPPRegistryZoneData zoneInfo = null;

  /**
   * {@code systemInfo} instance that contains detailed system info.
   */
  private EPPRegistrySystemInfo systemInfo = null;

  /**
   * Category for logging
   */
      private static Logger cat = LoggerFactory.getLogger(EPPRegistryInfoResp.class);
        

  /**
   * {@code EPPRegistryInfoResp} default constructor. Must call either
   * {@code setZoneList}, {@code setZoneInfo}, or {@code setSystemInfo} setter
   * methods before invoking {@code encode}.
   */
  public EPPRegistryInfoResp() {
  }

  /**
   * Create a {@code EPPRegistryInfoResp} instance with a set of zone summary
   * attributes.
   *
   * @param aTransId
   *           transaction Id associated with response
   * @param aZoneList
   *           object containing a list of zone summary attributes
   */
  public EPPRegistryInfoResp(EPPTransId aTransId, EPPRegistryZoneList aZoneList) {
    super(aTransId);
    this.zoneList = aZoneList;
    this.zoneInfo = null;
    this.systemInfo = null;
  }

  /**
   * Create a {@code EPPRegistryInfoResp} instance with zone detailed
   * information.
   *
   * @param aTransId
   *           transaction Id associated with response
   * @param aZoneInfo
   *           object containing detailed zone info
   */
  public EPPRegistryInfoResp(EPPTransId aTransId, EPPRegistryZoneData aZoneInfo) {
    super(aTransId);
    this.zoneInfo = aZoneInfo;
    this.zoneList = null;
    this.systemInfo = null;
  }

  /**
   * Create a {@code EPPRegistryInfoResp} instance with system detailed
   * information.
   *
   * @param aTransId
   *           transaction Id associated with response
   * @param aSystemInfo
   *           object containing detailed system info
   */
  public EPPRegistryInfoResp(EPPTransId aTransId, EPPRegistrySystemInfo aSystemInfo) {
    super(aTransId);
    this.systemInfo = aSystemInfo;
    this.zoneInfo = null;
    this.zoneList = null;
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPRegistryInfoResp} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         {@code EPPRegistryInfoResp} instance.
   *
   * @exception EPPEncodeException
   *               Unable to encode {@code EPPRegistryInfoResp} instance.
   */
  @Override
  protected Element doEncode(Document aDocument) throws EPPEncodeException {
    try {
      validateState();
    }
    catch (EPPCodecException e) {
      cat.error("EPPRegistryInfoResp.doEncode(): Invalid state on encode: " + e);
      throw new EPPEncodeException("Invalid state on EPPRegistryInfoResp.encode: " + e);
    }

    Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

    // Zone List
    if (this.zoneList != null) {
      EPPUtil.encodeComp(aDocument, root, this.zoneList);
    }

    // Zone Info
    if (this.zoneInfo != null) {
      EPPUtil.encodeComp(aDocument, root, this.zoneInfo);
    }

    // System Info
    if (this.systemInfo != null) {
      EPPUtil.encodeComp(aDocument, root, this.systemInfo);
    }

    return root;
  }

  /**
   * Decode the {@code EPPRegistryInfoResp} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPRegistryInfoResp} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  protected void doDecode(Element aElement) throws EPPDecodeException {
    // Zone List
    this.zoneList = (EPPRegistryZoneList) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
          EPPRegistryZoneList.ELM_NAME, EPPRegistryZoneList.class);

    // Zone Info
    this.zoneInfo = (EPPRegistryZoneData) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
          EPPRegistryZoneData.ELM_NAME, EPPRegistryZoneData.class);

    // System Info
    this.systemInfo = (EPPRegistrySystemInfo) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
          EPPRegistrySystemInfo.ELM_NAME, EPPRegistrySystemInfo.class);
  }

  /**
   * Validate the state of the {@code EPPRegistryInfoResp} instance. A valid
   * state means that all of the required attributes have been set. If
   * validateState returns without an exception, the state is valid. If the
   * state is not valid, the EPPCodecException will contain a description of
   * the error. throws EPPCodecException State error. This will contain the
   * name of the attribute that is not valid.
   *
   * @throws EPPCodecException
   */
  void validateState() throws EPPCodecException {
    if ((this.zoneList == null && this.zoneInfo == null && this.systemInfo == null)
          || (this.zoneList != null && this.zoneInfo != null) || (this.zoneList != null && this.systemInfo != null)
          || (this.zoneInfo != null && this.systemInfo != null)) {
      throw new EPPCodecException("One and only one zoneList, zoneInfo, and systemInfo can exist");
    }
  }

  /**
   * Clone {@code EPPRegistryInfoResp}.
   *
   * @return clone of {@code EPPRegistryInfoResp}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPRegistryInfoResp clone = (EPPRegistryInfoResp) super.clone();

    // Zone List
    if (this.zoneList != null) {
      clone.zoneList = (EPPRegistryZoneList) this.zoneList.clone();
    }

    // Zone Info
    if (this.zoneInfo != null) {
      clone.zoneInfo = (EPPRegistryZoneData) this.zoneInfo.clone();
    }

    // System Info
    if (this.systemInfo != null) {
      clone.systemInfo = (EPPRegistrySystemInfo) this.systemInfo.clone();
    }

    return clone;
  }

  /**
   * Gets the EPP response type associated with {@code EPPRegistryInfoResp}.
   *
   * @return {@code EPPRegistryInfoResp.ELM_NAME}
   */
  @Override
  public String getType() {
    return ELM_NAME;
  }

  /**
   * Gets the EPP command namespace associated with {@code EPPRegistryInfoResp}
   * .
   *
   * @return {@code EPPRegistryMapFactory.NS}
   */
  @Override
  public String getNamespace() {
    return EPPRegistryMapFactory.NS;
  }

  /**
   * Compare an instance of {@code EPPRegistryInfoResp} with this instance.
   *
   * @param aObject
   *           Object to compare with.
   *
   * @return {@code true} if this object is the same as the aObject argument;
   *         {@code false} otherwise
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPRegistryInfoResp)) {
      return false;
    }

    EPPRegistryInfoResp theComp = (EPPRegistryInfoResp) aObject;

    // Zone List
    if (!((this.zoneList == null) ? (theComp.zoneList == null) : this.zoneList.equals(theComp.zoneList))) {
      cat.error("EPPRegistryInfoResp.equals(): zoneList not equal");
      return false;
    }

    // Zone Info
    if (!((this.zoneInfo == null) ? (theComp.zoneInfo == null) : this.zoneInfo.equals(theComp.zoneInfo))) {
      cat.error("EPPRegistryInfoResp.equals(): zoneInfo not equal");
      return false;
    }

    // System Info
    if (!((this.systemInfo == null) ? (theComp.systemInfo == null) : this.systemInfo.equals(theComp.systemInfo))) {
      cat.error("EPPRegistryInfoResp.equals(): systemInfo not equal");
      return false;
    }

    return true;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Is the zone list set?
   *
   * @return {@code true} if the zone list is defined; {@code false} otherwise.
   */
  public boolean hasZoneList() {
    if (this.zoneList != null) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Gets the zone list.
   * 
   * @return The zone list if defined: {@code null} otherwise.
   */
  public EPPRegistryZoneList getZoneList() {
    return this.zoneList;
  }

  /**
   * Sets the zone list.
   * 
   * @param aZoneList
   *           Zone list. Set to {@code null} if undefined.
   */
  public void setZoneList(EPPRegistryZoneList aZoneList) {
    this.zoneList = aZoneList;
  }

  /**
   * Is the zone info set?
   *
   * @return {@code true} if the zone info is defined; {@code false} otherwise.
   */
  public boolean hasZoneInfo() {
    if (this.zoneInfo != null) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Gets the zone information.
   * 
   * @return The detailed zone information if defined; {@code null} otherwise.
   */
  public EPPRegistryZoneData getZoneInfo() {
    return this.zoneInfo;
  }

  /**
   * Sets the zone information.
   * 
   * @param aZoneInfo
   *           The detailed system information. Set to {@code null} if
   *           undefined.
   */
  public void setZoneInfo(EPPRegistryZoneData aZoneInfo) {
    this.zoneInfo = aZoneInfo;
  }

  /**
   * Is the system info set?
   *
   * @return {@code true} if the system info is defined; {@code false}
   *         otherwise.
   */
  public boolean hasSystemInfo() {
    if (this.systemInfo != null) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Gets the system information.
   * 
   * @return The detailed system information if defined; {@code null}
   *         otherwise.
   */
  public EPPRegistrySystemInfo getSystemInfo() {
    return this.systemInfo;
  }

  /**
   * Sets the system information.
   * 
   * @param aSystemInfo
   *           The detailed system information. Set to {@code null} if
   *           undefined.
   */
  public void setSystemInfo(EPPRegistrySystemInfo aSystemInfo) {
    this.systemInfo = aSystemInfo;
  }
}
