/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.registry.v02;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCheckCmd;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;


/**
 * Represents an EPP Registry &lt;check&gt; command, which is used to determine
 * if a zone name is known to the server. The &lt;registry:check&gt; element
 * MUST contain the following child elements:<br>
 * <br>
 *
 * <ul>
 * <li>One or more &lt;registry:name&gt; elements that contain the fully
 * qualified name of the queried zones. Use {@code getNames} and
 * {@code setNames} to get and set the elements. Use {@code addName} to add a
 * name to existing list or use {@code setName} to set an individual name.</li>
 * </ul>
 *
 * <br>
 * {@code EPPRegistryCheckResp} is the concrete {@code EPPReponse} associated
 * with {@code EPPRegistryCheckCmd}. <br>
 * <br>
 * 
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryCheckResp
 */
public class EPPRegistryCheckCmd extends EPPCheckCmd {

  /**
   * Logger
   */
      private static Logger cat = LoggerFactory.getLogger(EPPRegistryCheckCmd.class);
        

  /**
   * XML local name for {@code EPPRegistryCheckCmd}.
   */
  public static final String ELM_LOCALNAME = "check";

  /**
   * XML root tag for {@code EPPRegistryCheckCmd}.
   */
  public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * Zone Names to check. This is a {@code List} of {@code EPPRegistryZoneName}
   * instances.
   */
  private List<EPPRegistryZoneName> names = new ArrayList<EPPRegistryZoneName>();

  /**
   * {@code EPPRegistryCheckCmd} default constructor. It will set the names
   * attribute to an empty {@code List}.
   */
  public EPPRegistryCheckCmd() {
  }

  /**
   * {@code EPPRegistryCheckCmd} constructor that will check an individual zone
   * name.
   *
   * @param aTransId
   *           Transaction Id associated with command.
   * @param aName
   *           aLabel zone name to check
   */
  public EPPRegistryCheckCmd(String aTransId, String aName) {
    super(aTransId);

    this.addName(aName);
  }

  /**
   * {@code EPPRegistryCheckCmd} constructor that will check an individual zone
   * name.
   *
   * @param aTransId
   *           Transaction Id associated with command.
   * @param aName
   *           zone name to check
   */
  public EPPRegistryCheckCmd(String aTransId, EPPRegistryZoneName aName) {
    super(aTransId);

    this.addName(aName);
  }

  /**
   * {@code EPPRegistryCheckCmd} constructor that will check a list of zone
   * names.
   *
   * @param aTransId
   *           Transaction Id associated with command.
   * @param aNames
   *           {@code List} of zone name {@code String} instances.
   */
  public EPPRegistryCheckCmd(String aTransId, List<EPPRegistryZoneName> aNames) {
    super(aTransId);

    this.setNames(aNames);
  }

  /**
   * Gets the EPP command Namespace associated with {@code EPPRegistryCheckCmd}
   * .
   *
   * @return {@code EPPRegistryMapFactory.NS}
   */
  @Override
  public String getNamespace() {
    return EPPRegistryMapFactory.NS;
  }

  /**
   * Gets the key for the registry object, which is the zone name.
   * 
   * @return The zone name if set with a single value; {@code null} otherwise.
   */
  @Override
  public String getKey() {
	  if (this.names != null && this.names.size() == 1) {
		  return this.names.get(0).getName();
	  }
	  else {
		  return null;
	  }
  }
  
  /**
   * Compare an instance of {@code EPPRegistryCheckCmd} with this instance.
   *
   * @param aObject
   *           Object to compare with.
   *
   * @return {@code true} if this object is the same as the aObject argument;
   *         {@code false} otherwise.
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPRegistryCheckCmd)) {
      return false;
    }

    if (!super.equals(aObject)) {
      return false;
    }

    EPPRegistryCheckCmd theMap = (EPPRegistryCheckCmd) aObject;

    // Zone Names
    if (!EPPUtil.equalLists(this.names, theMap.names)) {
      cat.error("EPPRegistryCheckCmd.equals(): names not equal");
      return false;
    }

    return true;
  }

  /**
   * Clone {@code EPPRegistryCheckCmd}.
   *
   * @return Deep copy clone of {@code EPPRegistryCheckCmd}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPRegistryCheckCmd clone = (EPPRegistryCheckCmd) super.clone();

    clone.names = (List) ((ArrayList) this.names).clone();

    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPRegistryCheckCmd} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         {@code EPPRegistryCheckCmd} instance.
   *
   * @exception EPPEncodeException
   *               Unable to encode {@code EPPRegistryCheckCmd} instance.
   */
  @Override
  protected Element doEncode(Document aDocument) throws EPPEncodeException {
    if ((this.names == null) || (this.names.size() == 0)) {
      throw new EPPEncodeException("No zone names specified in EPPRegistryCheckCmd");
    }

    Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

    EPPUtil.encodeCompList(aDocument, root, this.names);

    return root;
  }

  /**
   * Decode the {@code EPPRegistryCheckCmd} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPRegistryCheckCmd} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  protected void doDecode(Element aElement) throws EPPDecodeException {
    this.names = EPPUtil.decodeCompList(aElement, EPPRegistryMapFactory.NS, EPPRegistryZoneName.ELM_ZONE_NAME,
          EPPRegistryZoneName.class);

    if (this.names == null) {
      this.names = new ArrayList<EPPRegistryZoneName>();
    }
  }

  /**
   * Get zone names to check.
   *
   * @return {@code List} of zone name {@code EPPRegistryZoneName}'s
   */
  public List<EPPRegistryZoneName> getNames() {
    return this.names;
  }

  /**
   * Set zone names to check.
   *
   * @param aNames
   *           {@code List} of zone name {@code EPPRegistryZoneName}'s
   */
  public void setNames(List<EPPRegistryZoneName> aNames) {
    if (aNames == null) {
      this.names = new ArrayList<EPPRegistryZoneName>();
    }
    else {
      this.names = aNames;
    }
  }

  /**
   * Set an individual aLabel zone name to check. This method clears existing
   * zone name {@code List}.
   *
   * @param aName
   *           aLabel zone name to check
   */
  public void setName(String aName) {
    this.names = new ArrayList<EPPRegistryZoneName>();
    this.addName(aName);
  }

  /**
   * Set an individual zone name to check. This method clears existing zone
   * name {@code List}.
   *
   * @param aName
   *           zone name to check
   */
  public void setName(EPPRegistryZoneName aName) {
    this.names = new ArrayList<EPPRegistryZoneName>();
    this.addName(aName);
  }

  /**
   * Append an aLabel zone name to the name {@code List} to check. This method
   * does NOT clear existing zone name {@code List}.
   *
   * @param aName
   *           aLabel zone name to append
   */
  public void addName(String aName) {
    this.names.add(new EPPRegistryZoneName(aName, EPPRegistryZoneName.Form.aLabel));
  }

  /**
   * Append an zone name to the name {@code List} to check. This method does
   * NOT clear existing zone name {@code List}.
   *
   * @param aName
   *           Zone name to append
   */
  public void addName(EPPRegistryZoneName aName) {
    this.names.add(aName);
  }

}
