/***********************************************************
Copyright (C) 2021 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.interfaces.v1_0;

import java.util.Random;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPResult;
import com.verisign.epp.codec.maintenance.v1_0.EPPMaintenanceDescription;
import com.verisign.epp.codec.maintenance.v1_0.EPPMaintenanceInfoCmd;
import com.verisign.epp.codec.maintenance.v1_0.EPPMaintenanceInfoResp;
import com.verisign.epp.interfaces.EPPApplicationSingle;
import com.verisign.epp.interfaces.EPPCommandException;
import com.verisign.epp.interfaces.EPPSession;
import com.verisign.epp.util.Environment;
import com.verisign.epp.util.TestThread;

import junit.framework.Assert;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

/**
 * Is a unit test of the {@code EPPMaintenance} class. The unit test will
 * initialize a session with an EPP Server, will invoke {@code EPPMaintenance}
 * operations, and will end a session with an EPP Server. The configuration file
 * used by the unit test defaults to epp.config, but can be changed by passing
 * the file path as the first command line argument. The unit test can be run in
 * multiple threads by setting the "threads" system property. For example, the
 * unit test can be run in 2 threads with the configuration file
 * ../../epp.config with the following command: <br>
 * <br>
 * java com.verisign.epp.interfaces.EPPMaintenanceTst -Dthreads=2
 * ../../epp.config <br>
 * <br>
 * The unit test is dependent on the use of <a
 * href=http://www.mcwestcorp.com/Junit.html>JUNIT 3.5</a><br>
*/
public class EPPMaintenanceTst extends TestCase {
  /**
   * Handle to the Singleton EPP Application instance (
   * {@code EPPApplicationSingle})
   */
  private static EPPApplicationSingle app = EPPApplicationSingle.getInstance();

  /** Name of configuration file to use for test (default = epp.config). */
  private static String configFileName = "epp.config";

  /** Logging category */
  private static Logger cat = LoggerFactory.getLogger(EPPMaintenanceTst.class);
        

  /** EPP Maintenance associated with test */
  private EPPMaintenance maintenance = null;

  /** EPP Session associated with test */
  private EPPSession session = null;

  /** Current test iteration */
  private int iteration = 0;

  /**
   * Random instance for the generation of unique objects (hosts, IP addresses,
   * etc.).
   */
  private Random rd = new Random(System.currentTimeMillis());

  /**
   * Allocates an {@code EPPMaintenanceTst} with a logical name. The
   * constructor will initialize the base class {@code TestCase} with the
   * logical name.
   *
   * @param name
   *           Logical name of the test
   */
  public EPPMaintenanceTst(String name) {
    super(name);
  }

  /**
   * JUNIT test method to implement the {@code EPPMaintenanceTst TestCase}.
   * Each sub-test will be invoked in order to satisfy testing the
   * EPPMaintenance interface.
   */
  public void testMaintenance() {
    int numIterations = 1;

    String iterationsStr = System.getProperty("iterations");

    if (iterationsStr != null) {
      numIterations = Integer.parseInt(iterationsStr);
    }

    for (iteration = 0; (numIterations == 0) || (iteration < numIterations); iteration++) {
      printStart("Test Suite");

      maintenanceInfo();

      printEnd("Test Suite");
    }
  }

  /**
   * Unit test of {@code EPPMaintenance.sendInfo()} to get the information for
   * a specific org identifier.
   */
  public void maintenanceInfo() {
    printStart("maintenanceInfo");

    EPPMaintenanceInfoResp response;

    // Ensure the poll queue is cleared before starting.
    this.clearPollQueue();

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("maintenanceInfo: List Info.");

      this.maintenance.setTransId("ABC-12345");

      this.maintenance.setInfoType(EPPMaintenanceInfoCmd.InfoType.list);

      response = this.maintenance.sendInfo();

      // -- Output all of the response attributes
      System.out.println("maintenanceInfo: Maintenance List Info Response = [" + response + "]\n\n");

      System.out.println("\n----------------------------------------------------------------");

      System.out.println("maintenanceInfo: Id.");

      this.maintenance.setTransId("ABC-12345");

      this.maintenance.setInfoType(EPPMaintenanceInfoCmd.InfoType.id);

      this.maintenance.setMaintenanceId("2e6df9b0-4092-4491-bcc8-9fb2166dcee6");

      response = this.maintenance.sendInfo();

      // -- Output all of the response attributes
      System.out.println("maintenanceInfo: Maintenance List Id Response = [" + response + "]\n\n");
      
      if (response.getMaintenance().hasDescriptions()) {
        for (EPPMaintenanceDescription description : response.getMaintenance().getDescriptions()) {
          System.out.println("maintenanceInfo: description =\n" + description.getDescription());        
        }
      }

    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("maintenanceInfo");
  }

  /**
   * Unit test of {@code EPPMaintenance} poll messaging.
   */
  public void testMaintenancePoll() {
    printStart("testMaintenancePoll");

    // Ensure the poll queue is cleared before starting.
    this.clearPollQueue();

    try {
      System.out.println("\n----------------------------------------------------------------");

      System.out.println("testMaintenancePoll: insert-maintenance-poll-msg Id.");

      this.maintenance.setTransId("ABC-12345");

      this.maintenance.setInfoType(EPPMaintenanceInfoCmd.InfoType.id);

      this.maintenance.setMaintenanceId("insert-maintenance-poll-msg");

      this.maintenance.sendInfo();

      System.out.println("\n----------------------------------------------------------------");

      System.out.println("testMaintenancePoll: Get poll message.");

      session.setTransId("ABC-12345");

      session.setPollOp(EPPSession.OP_REQ);

      EPPResponse reqResponse = session.sendPoll();

      System.out.println("testMaintenancePoll: Poll Request Response = [" + reqResponse + "]\n\n");

      Assert.assertEquals(EPPResult.SUCCESS_POLL_MSG, reqResponse.getResult().getCode());
      Assert.assertTrue(reqResponse instanceof EPPMaintenanceInfoResp);

      // Poll Ack
      session.setPollOp(EPPSession.OP_ACK);

      session.setMsgID(reqResponse.getMsgQueue().getId());

      EPPResponse ackResponse = session.sendPoll();

      System.out.println("testMaintenancePoll: Poll Ack Response = [" + ackResponse + "]\n\n");
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("testMaintenancePoll");
  }

  /**
   * Unit test of {@code EPPSession.initSession}. The session attribute is
   * initialized with the attributes defined in the EPP sample files.
   */
  private void initSession() {
    printStart("initSession");

    // Set attributes for initSession
    session.setClientID(Environment.getProperty("EPP.Test.clientId", "ClientX"));
    session.setPassword(Environment.getProperty("EPP.Test.password", "foo-BAR2"));
    session.setTransId("ABC-12345-XYZ");
    session.setVersion("1.0");
    session.setLang("en");

    // Initialize the session
    try {
      session.initSession();
    }

    catch (EPPCommandException e) {
      EPPResponse response = session.getResponse();

      // Is a server specified error?
      if ((response != null) && (!response.isSuccess())) {
        Assert.fail("Server Error : " + response);
      }
      else {
        e.printStackTrace();

        Assert.fail("initSession Error : " + e);
      }
    }

    printEnd("initSession");
  }

  /**
   * Unit test of {@code EPPSession.endSession}. The session with the EPP
   * Server will be terminated.
   */
  private void endSession() {
    printStart("endSession");

    session.setTransId("ABC-12345-XYZ");

    // End the session
    try {
      session.endSession();
    }

    catch (EPPCommandException e) {
      EPPResponse response = session.getResponse();

      // Is a server specified error?
      if ((response != null) && (!response.isSuccess())) {
        Assert.fail("Server Error : " + response);
      }

      else // Other error
      {
        e.printStackTrace();

        Assert.fail("initSession Error : " + e);
      }
    }

    printEnd("endSession");
  }

  /**
   * JUNIT {@code setUp} method, which sets the default client Id to
   * "theRegistrar".
   */
  protected void setUp() {
    try {
      String theSessionClassName = System.getProperty("EPP.SessionClass");

      if (theSessionClassName != null) {
        try {
          Class theSessionClass = Class.forName(theSessionClassName);

          if (!EPPSession.class.isAssignableFrom((theSessionClass))) {
            Assert.fail(theSessionClassName + " is not a subclass of EPPSession");
          }

          session = (EPPSession) theSessionClass.getDeclaredConstructor().newInstance();
        }
        catch (Exception ex) {
          Assert.fail("Exception instantiating EPP.SessionClass value " + theSessionClassName + ": " + ex);
        }
      }
      else {
        session = new EPPSession();
      }

    }

    catch (Exception e) {
      e.printStackTrace();

      Assert.fail("Error initializing the session: " + e);
    }

    initSession();

    this.maintenance = new EPPMaintenance(session);
  }

  /**
   * JUNIT {@code tearDown}, which currently does nothing.
   */
  protected void tearDown() {
    endSession();
  }

  /**
   * JUNIT {@code suite} static method, which returns the tests associated with
   * {@code EPPMaintenanceTst}.
   *
   * @return Test suite
   */
  public static Test suite() {
    TestSuite suite = new TestSuite(EPPMaintenanceTst.class);

    String theConfigFileName = System.getProperty("EPP.ConfigFile");
    if (theConfigFileName != null)
      configFileName = theConfigFileName;

    try {
      app.initialize(configFileName);
    }

    catch (EPPCommandException e) {
      e.printStackTrace();

      Assert.fail("Error initializing the EPP Application: " + e);
    }

    return suite;
  }

  /**
   * Handle an {@code EPPCommandException}, which can be either a server
   * generated error or a general exception. If the exception was caused by a
   * server error, "Server Error :&lt;Response XML&gt;" will be specified. If
   * the exception was caused by a general algorithm error, "General Error
   * :&lt;Exception Description&gt;" will be specified.
   *
   * @param aException
   *           Exception thrown during test
   */
  public void handleException(Exception aException) {
    EPPResponse response = session.getResponse();

    aException.printStackTrace();

    // Is a server specified error?
    if ((response != null) && (!response.isSuccess())) {
      Assert.fail("Server Error : " + response);
    }

    else {
      Assert.fail("General Error : " + aException);
    }
  }

  /**
   * Unit test main, which accepts the following system property options: <br>
   *
   * <ul>
   * <li>iterations Number of unit test iterations to run</li>
   * <li>validate Turn XML validation on ({@code true}) or off (
   * {@code false}). If validate is not specified, validation will be off.</li>
   * </ul>
   *
   *
   * @param args
   *           Command line arguments
   */
  public static void main(String[] args) {
    // Override the default configuration file name?
    if (args.length > 0) {
      configFileName = args[0];
    }

    // Number of Threads
    int numThreads = 1;

    String threadsStr = System.getProperty("threads");

    if (threadsStr != null) {
      numThreads = Integer.parseInt(threadsStr);

      // Run test suite in multiple threads?
    }

    if (numThreads > 1) {
      // Spawn each thread passing in the Test Suite
      for (int i = 0; i < numThreads; i++) {
        TestThread thread = new TestThread("EPPSessionTst Thread " + i, EPPMaintenanceTst.suite());

        thread.start();
      }
    }

    else { // Single threaded mode.
      junit.textui.TestRunner.run(EPPMaintenanceTst.suite());
    }

    try {
      app.endApplication();
    }

    catch (EPPCommandException e) {
      e.printStackTrace();

      Assert.fail("Error ending the EPP Application: " + e);
    }
  }

  /**
   * Print the start of a test with the {@code Thread} name if the current
   * thread is a {@code TestThread}.
   *
   * @param aTest
   *           name for the test
   */
  private void printStart(String aTest) {
    if (Thread.currentThread() instanceof TestThread) {
      System.out.print(Thread.currentThread().getName() + ", iteration " + iteration + ": ");

      cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aTest + " Start");
    }

    System.out.println("Start of " + aTest);

    System.out.println("****************************************************************\n");
  }

  /**
   * Print the end of a test with the {@code Thread} name if the current thread
   * is a {@code TestThread}.
   *
   * @param aTest
   *           name for the test
   */
  private void printEnd(String aTest) {
    System.out.println("****************************************************************");

    if (Thread.currentThread() instanceof TestThread) {
      System.out.print(Thread.currentThread().getName() + ", iteration " + iteration + ": ");

      cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aTest + " End");
    }

    System.out.println("End of " + aTest);

    System.out.println("\n");
  }

  /**
   * Print message
   *
   * @param aMsg
   *           message to print
   */
  private void printMsg(String aMsg) {
    if (Thread.currentThread() instanceof TestThread) {
      System.out.println(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);

      cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);
    }

    else {
      System.out.println(aMsg);

      cat.info(aMsg);
    }
  }

  /**
   * Print error message
   *
   * @param aMsg
   *           errpr message to print
   */
  private void printError(String aMsg) {
    if (Thread.currentThread() instanceof TestThread) {
      System.err.println(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);

      cat.error(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);
    }

    else {
      System.err.println(aMsg);

      cat.error(aMsg);
    }
  }

  /**
   * Utility method to clear any messages currently left in the poll queue.
   */
  public void clearPollQueue() {

    try {
      EPPResponse reqResponse = null;

      do {
        session.setTransId("ABC-12345");

        session.setPollOp(EPPSession.OP_REQ);

        reqResponse = session.sendPoll();

        System.out.println("clearPollQueue: Poll Request Response = [" + reqResponse + "]\n\n");

        if (reqResponse.getResult().getCode() == EPPResult.SUCCESS_POLL_MSG) {

          // Poll Ack
          session.setPollOp(EPPSession.OP_ACK);

          session.setMsgID(reqResponse.getMsgQueue().getId());

          EPPResponse ackResponse = session.sendPoll();

          System.out.println("clearPollQueue: Poll Ack Response = [" + ackResponse + "]\n\n");
        }
      }
      while (reqResponse != null && reqResponse.getResult().getCode() == EPPResult.SUCCESS_POLL_MSG);
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

  }

}
