/***********************************************************
Copyright (C) 2021 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.interfaces.v1_0;

import java.util.Vector;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.maintenance.v1_0.EPPMaintenanceInfoCmd;
import com.verisign.epp.codec.maintenance.v1_0.EPPMaintenanceInfoResp;
import com.verisign.epp.interfaces.EPPCommandException;
import com.verisign.epp.interfaces.EPPSession;

/**
 * {@code EPPMaintenance} is the primary client interface class used for the Maintenance
 * EPP mapping. An instance of {@code EPPMaintenance} is created with an
 * initialized {@code EPPSession}, and can be used for more than one
 * request within a single thread. A set of setter methods are provided to set
 * the attributes before a call to one of the send action methods. The responses
 * returned from the send action methods are either instances of
 * {@code EPPResponse} or instances of response classes in the
 * {@code com.verisign.epp.codec.maintenance} package.
 *
 * @see com.verisign.epp.codec.maintenance.v1_0.EPPMaintenanceInfoResp
 */
public class EPPMaintenance {

  /** An instance of a session. */
  private EPPSession session = null;

  /** Transaction Id provided by client */
  private String transId = null;


  /**
   * Extension objects associated with the command. This is a
   * {@code Vector} of {@code EPPCodecComponent} objects.
   */
  private Vector extensions = null;


  /**
   * Type of info command to send.
   */
  EPPMaintenanceInfoCmd.InfoType infoType = EPPMaintenanceInfoCmd.InfoType.list;

  /**
   * Maintenance identifier for {@link EPPMaintenanceInfoCmd.InfoType.id}.
   */
  private String maintenanceId;


  /**
   * Constructs an {@code EPPMaintenance} given an initialized EPP session.
   *
   * @param aSession
   *           Server session to use.
   */
  public EPPMaintenance(EPPSession aSession) {
    this.session = aSession;
  }

  /**
   * Adds a command extension object.
   *
   * @param aExtension
   *           command extension object associated with the command
   */
  public void addExtension(EPPCodecComponent aExtension) {
    if (this.extensions == null) {
      this.extensions = new Vector();
    }

    this.extensions.addElement(aExtension);
  }

  /**
   * Sets the command extension objects.
   *
   * @param aExtensions
   *           command extension objects associated with the command
   */
  public void setExtensions(Vector aExtensions) {
    this.extensions = aExtensions;
  }

  /**
   * Gets the command extensions.
   *
   * @return {@code Vector} of concrete {@code EPPCodecComponent}
   *         associated with the command if exists; {@code null}
   *         otherwise.
   */
  public Vector getExtensions() {
    return this.extensions;
  }

  /**
   * Sets the client transaction identifier.
   *
   * @param aTransId
   *           Client transaction identifier
   */
  public void setTransId(String aTransId) {
    this.transId = aTransId;
  }


  /**
   * Gets the i{@code infoType}for the info command.
   *
   * @return the {@code infoType} for the info command.
   */
  public EPPMaintenanceInfoCmd.InfoType getInfoType() {
    return this.infoType;
  }

  /**
   * Sets the {@code infoType} for the info command. The maintenance
   * notification identifier is used only when the {@code infoType} is set to
   * {@code InfoType.id}.
   *
   * @param aInfoType
   *           The {@code infoType} for the info command.
   */
  public void setInfoType(EPPMaintenanceInfoCmd.InfoType aInfoType) {
    this.infoType = aInfoType;
  }


  /**
   * Gets the maintenance identifier.
   *
   * @return The maintenance identifier if defined; {@code null} otherwise.
   */
  public String getMaintenanceId() {
    return this.maintenanceId;
  }

  /**
   * Sets the maintenance identifier to send info for.
   *
   * @param aMaintenanceId Maintenance identifier.
   */
  public void setMaintenanceId(String aMaintenanceId) {
    this.maintenanceId = aMaintenanceId;
    if (this.maintenanceId != null) {
      this.infoType = EPPMaintenanceInfoCmd.InfoType.id;
    }
    else {
      this.infoType = EPPMaintenanceInfoCmd.InfoType.list;
    }
  }
  /**


  /**
   * Sends a Maintenance Info Command to the server.<br>
   * <br>
   * The optional attributes can be set with the following:<br>
   * <br>
   * <ul>
   * <li>{@link #setTransId(String)} - Sets the client transaction identifier.
   * <li>{@link #setMaintenanceId(String)} - Sets the maintenance identifier.
   * </li>
   * </ul>
   *
   * @return {@link EPPMaintenanceInfoResp} that contains the maintenance information.
   *
   * @throws EPPCommandException
   *            On error
   */
  public EPPMaintenanceInfoResp sendInfo() throws EPPCommandException {
    // Create the command
    EPPMaintenanceInfoCmd theCommand = new EPPMaintenanceInfoCmd(this.transId);
    if (this.maintenanceId != null) {
      theCommand.setMaintenanceId(this.maintenanceId);
    }

    // Set command extension
    theCommand.setExtensions(this.extensions);

    // Reset Maintenance attributes
    this.resetMaintenance();

    // process the command and response
    return (EPPMaintenanceInfoResp) this.session.processDocument(theCommand, EPPMaintenanceInfoResp.class);
  }


  /**
   * Resets the Maintenance instance to its initial state.
   */
  protected void resetMaintenance() {
    this.transId = null;
    this.extensions = null;
    this.infoType = EPPMaintenanceInfoCmd.InfoType.list;
    this.maintenanceId = null;
  }

  /**
   * Gets the response associated with the last command. This method can be
   * used to retrieve the server error response in the catch block of
   * EPPCommandException.
   *
   * @return Response associated with the last command
   */
  public EPPResponse getResponse() {
    return this.session.getResponse();
  }

}
