/***********************************************************
Copyright (C) 2021 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.maintenance.v1_0;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Type of the maintenance that has the possible set of values defined by server
 * policy, such as "Routine Maintenance", "Software Update", "Software Upgrade",
 * or "Extended Outage".
 */
public class EPPMaintenanceType implements EPPCodecComponent {

  /**
   * Default language
   */
  public static final String DEFAULT_LANG = "en";

  /**
   * Category for logging
   */
      private static Logger cat = LoggerFactory.getLogger(EPPMaintenanceType.class);
        

  /**
   * XML local name for {@code EPPMaintenanceType}.
   */
  public static final String ELM_LOCALNAME = "type";

  /**
   * XML root tag for {@code EPPMaintenance}.
   */
  public static final String ELM_NAME = EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * XML attribute for the type lang attribute.
   */
  private static final String ATTR_LANG = "lang";

  /**
   * OPTIONAL language of the type.
   */
  private String lang = DEFAULT_LANG;

  /**
   * Type value
   */
  private String type;

  /**
   * Default constructor for {@code EPPMaintenanceType}. The required
   * attributes need to be set. The {@code lang} attribute defaults to
   * {@link DEFAULT_LANG}.
   */
  public EPPMaintenanceType() {
  }

  /**
   * Constructor for {@code EPPMaintenanceType} with the required attribute as
   * a parameter.
   *
   * @param aType
   *           Type value
   */
  public EPPMaintenanceType(String aType) {
    this.type = aType;
  }

  /**
   * Constructor for {@code EPPMaintenanceType} with all of the attributes as
   * parameters.
   *
   * @param aType
   *           Type value
   * @param aLang
   *           Language of the type
   */
  public EPPMaintenanceType(String aType, String aLang) {
    this.type = aType;
    this.lang = aLang;
  }

  /**
   * Gets the maintenance type
   *
   * @return The type of the maintenance if defined; {@code null} otherwise.
   */
  public String getType() {
    return this.type;
  }

  /**
   * Sets the maintenance type
   *
   * @param aType
   *           Type of the maintenance
   */
  public void setType(String aType) {
    this.type = aType;
  }

  /**
   * Is the type language defined with a non-default value?
   *
   * @return <code>true</code> if the type language is defined;
   *         <code>false</code> otherwise.
   */
  public boolean hasLang() {
    return (!this.lang.equals(DEFAULT_LANG) ? true : false);
  }

  /**
   * Gets the type language value.
   *
   * @return Type language if defined; <code>DEFAULT_LANG</code> otherwise.
   */
  public String getLang() {
    return this.lang;
  }

  /**
   * Sets the type language value.
   *
   * @param aLang
   *           Type language for value.
   */
  public void setLang(String aLang) {
    if (aLang == null) {
      this.lang = DEFAULT_LANG;
    }
    else {
      this.lang = aLang;
    }
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPMaintenanceType} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Root DOM Element representing the {@code EPPMaintenanceType}
   *         instance.
   *
   * @exception EPPEncodeException
   *               Unable to encode {@code EPPMaintenanceType} instance.
   */
  @Override
  public Element encode(Document aDocument) throws EPPEncodeException {

    // Validate state
    if ((this.type == null) || (this.lang == null)) {
      throw new EPPEncodeException("required attribute is not set");
    }

    // Create root element
    Element root = aDocument.createElementNS(EPPMaintenanceMapFactory.NS, ELM_NAME);

    // Type value
    Text currVal = aDocument.createTextNode(this.type);
    root.appendChild(currVal);

    // Lang
    root.setAttribute(ATTR_LANG, this.lang);

    return root;
  }

  /**
   * Decode the {@code EPPMaintenanceType} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPMaintenanceType} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement.
   */
  @Override
  public void decode(Element aElement) throws EPPDecodeException {

    // Type value
    this.type = EPPUtil.decodeStringValue(aElement);

    // Lang
    this.setLang(EPPUtil.decodeStringAttr(aElement, ATTR_LANG));
  }

  /**
   * implements a deep {@code EPPMaintenanceType} compare.
   *
   * @param aObject
   *           {@code EPPMaintenanceType} instance to compare with
   *
   * @return {@code true} of {@code aObject} is equal to instance;
   *         {@code false} otherwise.
   */
  @Override
  public boolean equals(Object aObject) {

    if (!(aObject instanceof EPPMaintenanceType)) {
      return false;
    }

    EPPMaintenanceType other = (EPPMaintenanceType) aObject;

    // Type
    if (!EqualityUtil.equals(this.type, other.type)) {
      cat.error("EPPMaintenanceType.equals(): type not equal");
      return false;
    }

    // Lang
    if (!EqualityUtil.equals(this.lang, other.lang)) {
      cat.error("EPPMaintenanceType.equals(): lang not equal");
      return false;
    }

    return true;
  }

  /**
   * Clone {@code EPPMaintenanceType}.
   *
   * @return clone of {@code EPPMaintenanceType}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPMaintenanceType clone = null;

    clone = (EPPMaintenanceType) super.clone();

    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   *
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPMaintenanceMapFactory.NS;
  }

}
