/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.launchpolicy.v01;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPLaunchPolicyPollPolicy} defines the poll messaging policy for the
 * phase. The poll policy contains the following required boolean attributes:
 * <br>
 * <ol>
 * <li>{@code intermediateStatus} - A boolean value indicating whether the
 * server will insert poll messages, per [RFC5730], for the applicable
 * intermediate statuses, including the "pendingValidation", "validated",
 * "pendingAllocation", and "invalid" statuses, using the &lt;domain:infData&gt;
 * element with the &lt;launch:infData&gt; extension.</li>
 * <li>{@code nonMandatoryInfo} - A boolean value indicating whether the server
 * will include non-mandatory information in the &lt;domain:infData&gt; element
 * of the poll message.</li>
 * <li>{@code extensionInfo} - A boolean value indicating whether the server
 * will include further extensions that would normally be included in the
 * response to the &lt;domain:info&gt; command, per [RFC5731], in the poll
 * message</li>
 * </ol>
 */
public class EPPLaunchPolicyPollPolicy implements EPPCodecComponent {

  /**
   * Category for logging
   */
      private static Logger cat = LoggerFactory.getLogger(EPPLaunchPolicyPollPolicy.class);
        

  /**
   * XML local name for {@code EPPLaunchPolicyPollPolicy}.
   */
  public static final String ELM_LOCALNAME = "pollPolicy";

  /**
   * XML root tag for {@code EPPLaunchPolicyPollPolicy}.
   */
  public static final String ELM_NAME = EPPLaunchPolicyExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * XML Element Name of {@code intermediateStatus} attribute.
   */
  private final static String ELM_INTERMEDIATE_STATUS = "intermediateStatus";

  /**
   * XML Element Name of {@code nonMandatoryInfo} attribute.
   */
  private final static String ELM_NON_MANDATORY_INFO = "nonMandatoryInfo";

  /**
   * XML Element Name of {@code extensionInfo} attribute.
   */
  private final static String ELM_EXTENSION_INFO = "extensionInfo";

  /**
   * A boolean value indicating whether the server will insert poll messages,
   * per [RFC5730], for the applicable intermediate statuses, including the
   * "pendingValidation", "validated", "pendingAllocation", and "invalid"
   * statuses, using the &lt;domain:infData&gt; element with the
   * &lt;launch:infData&gt; extension.
   */
  private Boolean intermediateStatus;

  /**
   * A boolean value indicating whether the server will include non-mandatory
   * information in the &lt;domain:infData&gt; element of the poll message.
   */
  private Boolean nonMandatoryInfo;

  /**
   * A boolean value indicating whether the server will include further
   * extensions that would normally be included in the response to the
   * &lt;domain:info&gt; command, per [RFC5731], in the poll message.
   */
  private Boolean extensionInfo;

  /**
   * Default constructor for {@code EPPLaunchPolicyPollPolicy}. All of the
   * boolean attributes must be set prior to calling {@link #encode(Document)}.
   */
  public EPPLaunchPolicyPollPolicy() {
  }

  /**
   * 
   * @param aIntermediateStatus
   *           Boolean indication of whether the intermediate status poll
   *           messages are included if defined
   * @param aNonMandatoryInfo
   *           Boolean value indicating whether the server will include
   *           non-mandatory information in the &lt;domain:infData&gt; element
   *           of the poll message.
   * @param aExtensionInfo
   *           Boolean value indicating whether the server will include further
   *           extensions that would normally be included in the response to
   *           the &lt;domain:info&gt; command, per [RFC5731], in the poll
   *           message.
   */
  public EPPLaunchPolicyPollPolicy(Boolean aIntermediateStatus, Boolean aNonMandatoryInfo, Boolean aExtensionInfo) {
    this.intermediateStatus = aIntermediateStatus;
    this.nonMandatoryInfo = aNonMandatoryInfo;
    this.extensionInfo = aExtensionInfo;
  }

  /**
   * encode instance into a DOM element tree. A DOM Document is passed as an
   * argument and functions as a factory for DOM objects. The root element
   * associated with the instance is created and each instance attribute is
   * appended as a child node.
   *
   * @param aDocument
   *           DOM Document, which acts is an Element factory
   *
   * @return Element Root element associated with the object
   *
   * @exception EPPEncodeException
   *               Error encoding {@code EPPLaunchPolicyPollPolicy}
   */
  @Override
  public Element encode(Document aDocument) throws EPPEncodeException {

    // Check required attributes
    if (this.intermediateStatus == null) {
      throw new EPPEncodeException("Undefined intermediateStatus in EPPLaunchPolicyPollPolicy");
    }
    if (this.nonMandatoryInfo == null) {
      throw new EPPEncodeException("Undefined nonMandatoryInfo in EPPLaunchPolicyPollPolicy");
    }
    if (this.extensionInfo == null) {
      throw new EPPEncodeException("Undefined extensionInfo in EPPLaunchPolicyPollPolicy");
    }

    Element root = aDocument.createElementNS(EPPLaunchPolicyExtFactory.NS, ELM_NAME);

    // Intermediate Status
    EPPUtil.encodeBoolean(aDocument, root, this.intermediateStatus, EPPLaunchPolicyExtFactory.NS,
          EPPLaunchPolicyExtFactory.NS_PREFIX + ":" + ELM_INTERMEDIATE_STATUS);

    // Non Mandatory Info
    EPPUtil.encodeBoolean(aDocument, root, this.nonMandatoryInfo, EPPLaunchPolicyExtFactory.NS,
          EPPLaunchPolicyExtFactory.NS_PREFIX + ":" + ELM_NON_MANDATORY_INFO);

    // Extension Info
    EPPUtil.encodeBoolean(aDocument, root, this.extensionInfo, EPPLaunchPolicyExtFactory.NS,
          EPPLaunchPolicyExtFactory.NS_PREFIX + ":" + ELM_EXTENSION_INFO);

    return root;
  }

  /**
   * decode a DOM element tree to initialize the instance attributes. The
   * {@code aElement} argument represents the root DOM element and is used to
   * traverse the DOM nodes for instance attribute values.
   *
   * @param aElement
   *           {@code Element} to decode
   *
   * @exception EPPDecodeException
   *               Error decoding {@code Element}
   */
  @Override
  public void decode(Element aElement) throws EPPDecodeException {
    // Intermediate Status
    this.intermediateStatus = EPPUtil.decodeBoolean(aElement, EPPLaunchPolicyExtFactory.NS, ELM_INTERMEDIATE_STATUS);

    // Non Mandatory Info
    this.nonMandatoryInfo = EPPUtil.decodeBoolean(aElement, EPPLaunchPolicyExtFactory.NS, ELM_NON_MANDATORY_INFO);

    // Extension Info
    this.extensionInfo = EPPUtil.decodeBoolean(aElement, EPPLaunchPolicyExtFactory.NS, ELM_EXTENSION_INFO);
  }

  /**
   * clone an {@code EPPCodecComponent}.
   *
   * @return clone of concrete {@code EPPOrgExt}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPLaunchPolicyPollPolicy clone = (EPPLaunchPolicyPollPolicy) super.clone();

    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Compare an instance of {@code EPPOrgExt} with this instance.
   *
   * @param aObject
   *           Object to compare with.
   *
   * @return {@code true} if equal; {@code false} otherwise.
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPLaunchPolicyPollPolicy)) {
      return false;
    }

    EPPLaunchPolicyPollPolicy other = (EPPLaunchPolicyPollPolicy) aObject;

    // Intermediate Status
    if (!EqualityUtil.equals(this.intermediateStatus, other.intermediateStatus)) {
      cat.error("EPPLaunchPolicyPollPolicy.equals(): intermediateStatus not equal");
      return false;
    }

    // Non Mandatory Info
    if (!EqualityUtil.equals(this.nonMandatoryInfo, other.nonMandatoryInfo)) {
      cat.error("EPPLaunchPolicyPollPolicy.equals(): nonMandatoryInfo not equal");
      return false;
    }

    // Extension Info
    if (!EqualityUtil.equals(this.extensionInfo, other.extensionInfo)) {
      cat.error("EPPLaunchPolicyPollPolicy.equals(): extensionInfo not equal");
      return false;
    }

    return true;
  }

  /**
   * Gets the boolean value indicating whether the server will insert poll
   * messages, per [RFC5730], for the applicable intermediate statuses,
   * including the "pendingValidation", "validated", "pendingAllocation", and
   * "invalid" statuses, using the &lt;domain:infData&gt; element with the
   * &lt;launch:infData&gt; extension.
   * 
   * @return Boolean indication of whether the intermediate status poll
   *         messages are included if defined; {@code null} otherwise.
   */
  public Boolean getIntermediateStatus() {
    return this.intermediateStatus;
  }

  /**
   * Sets the boolean value indicating whether the server will insert poll
   * messages, per [RFC5730], for the applicable intermediate statuses,
   * including the "pendingValidation", "validated", "pendingAllocation", and
   * "invalid" statuses, using the &lt;domain:infData&gt; element with the
   * &lt;launch:infData&gt; extension.
   * 
   * @param aIntermediateStatus
   *           The intermediate status boolean value
   */
  public void setIntermediateStatus(Boolean aIntermediateStatus) {
    this.intermediateStatus = aIntermediateStatus;
  }

  /**
   * Gets the boolean value indicating whether the server will include
   * non-mandatory information in the &lt;domain:infData&gt; element of the
   * poll message.
   * 
   * @return The intermediate status boolean value if defined; {@code null}
   *         otherwise.
   */
  public Boolean getNonMandatoryInfo() {
    return this.nonMandatoryInfo;
  }

  /**
   * Sets the boolean value indicating whether the server will include
   * non-mandatory information in the &lt;domain:infData&gt; element of the
   * poll message.
   * 
   * @param aNonMandatoryInfo
   *           The non-mandatory information boolean value
   */
  public void setNonMandatoryInfo(Boolean aNonMandatoryInfo) {
    this.nonMandatoryInfo = aNonMandatoryInfo;
  }

  /**
   * Gets the boolean value indicating whether the server will include further
   * extensions that would normally be included in the response to the
   * &lt;domain:info&gt; command, per [RFC5731], in the poll message.
   * 
   * @return The extension info boolean value if defined; {@code null}
   *         otherwise.
   */
  public Boolean getExtensionInfo() {
    return this.extensionInfo;
  }

  /**
   * Gets the boolean value indicating whether the server will include further
   * extensions that would normally be included in the response to the
   * &lt;domain:info&gt; command, per [RFC5731], in the poll message.
   * 
   * @param aExtensionInfo
   *           The extension info boolean value
   */
  public void setExtensionInfo(Boolean aExtensionInfo) {
    this.extensionInfo = aExtensionInfo;
  }

  /**
   * Returns the XML namespace associated with the {@code EPPCodecComponent}.
   *
   * @return XML namespace for the {@code EPPCodecComponent}.
   */
  @Override
  public String getNamespace() {
    return EPPLaunchPolicyExtFactory.NS;
  }

}
