/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.gen;

import java.util.List;

import com.verisign.epp.exception.EPPException;

/**
 * Interface used to filter poll message {@link EPPResponse} instances against
 * the list of client-specified object services and extension services. Any
 * non-supported services (object or extension) will be removed from the poll
 * message and reflected in the {@code EPPResponse} by the implementing
 * {@code EPPPollMessageFilter} class. The interface does support a no-operation
 * option to simply identify and log non-supported services without removing
 * them from the poll message.
 */
public interface EPPPollMessageFilter {

  /**
   * Filter any poll messages that are not supported by the client based on the
   * passed in login services (object extensions) and extension services
   * (command response extensions) from the poll message and encode into the
   * response the information associated with the unsupported extensions
   * (object or command / response).
   *
   * @param aResponse
   *           Source poll message
   * @param aServices
   *           {@code List} of {@link EPPService} login services (object
   *           extensions) supported by the client.
   * @param aExtServices
   *           {@code List} of {@link EPPService} login extension services
   *           (command response extensions) supported by the client.
   * @param aData
   *           Optional data object that will be passed through to the monitor
   *           when there is at least one unhandled namespace identified. This
   *           is useful to set transaction-level attributes based on the use
   *           of unhandled namespaces. Set to {@code null} if a data object is
   *           not needed.
   * @return Filtered poll message {@link EPPResponse} that contains extensions
   *         that the client supports.
   * @throws EPPException
   *            Exception filtering the poll message
   */
  EPPResponse filter(final EPPResponse aResponse, List<EPPService> aServices, List<EPPService> aExtServices,
        Object aData) throws EPPException;

  /**
   * Filter any poll messages that are not supported by the client based on the
   * passed in login services (object extensions) and extension services
   * (command response extensions) from the poll message and encode into the
   * response the information associated with the unsupported extensions
   * (object or command / response).
   *
   * @param aResponse
   *           Source poll message
   * @param aServices
   *           {@code List} of {@link EPPService} login services (object
   *           extensions) supported by the client.
   * @param aExtServices
   *           {@code List} of {@link EPPService} login extension services
   *           (command response extensions) supported by the client.
   * @param aNoOp
   *           Set to {@code true} to only identify unsupported services by
   *           logging them and not removing them or encoding them in another
   *           way in the response.
   * @param aData
   *           Optional data object that will be passed through to the monitor
   *           when there is at least one unhandled namespace identified. This
   *           is useful to set transaction-level attributes based on the use
   *           of unhandled namespaces. Set to {@code null} if a data object is
   *           not needed.
   * @return Filtered poll message {@link EPPResponse} that contains extensions
   *         that the client supports.
   * @throws EPPException
   *            Exception filtering the poll message
   */
  public EPPResponse filter(final EPPResponse aResponse, List<EPPService> aServices, List<EPPService> aExtServices,
        boolean aNoOp, Object aData) throws EPPException;
}
