/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.domain;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPAuthInfo;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPTransferCmd;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Represents an EPP Domain &lt;transfer&gt; command. The EPP &lt;transfer&gt;
 * command provides a query operation that allows a client to determine
 * real-time status of pending and completed transfer requests. In addition to
 * the standard EPP command elements, the &lt;transfer&gt; command MUST contain
 * an {@code op} attribute with value {@code query}, and a
 * &lt;domain:transfer&gt; element that identifies the domain namespace and the
 * location of the domain schema. The &lt;domain:transfer&gt; element SHALL
 * contain the following child elements: <br>
 * <br>
 *
 * <ul>
 * <li>A &lt;domain:name&gt; element that contains the fully qualified domain
 * name of the object for which a transfer request is to be created, approved,
 * rejected, or cancelled. Use {@code getName} and {@code setName} to get and
 * set the element.</li>
 * <li>An OPTIONAL &lt;domain:period&gt; element that contains the initial
 * registration period of the domain object. Use {@code getPeriod} and
 * {@code setPeriod} to get and set the element. If return {@code null}, period
 * has not been specified yet.</li>
 * <li>An "op" attribute that identifies the transfer operation to be performed.
 * Valid values, definitions, and authorizations for all attribute values are
 * defined in [EPP]. Use {@code getOp} and {@code setOp} to get and set the
 * element. One of the {@code EPPCommand.OP_} constants need to be specified.
 * </li>
 * <li>A &lt;domain:authInfo&gt; element that contains authorization information
 * associated with the domain object or authorization information associated
 * with the domain object's registrant or associated contacts. This element is
 * REQUIRED only when a transfer is requested, and it SHALL be ignored if used
 * otherwise. Use {@code getAuthInfo} and {@code setAuthInfo} to get and set the
 * element.</li>
 * </ul>
 *
 * <br>
 * <br>
 * Transfer of a domain object MUST implicitly transfer all host objects that
 * are subordinate to the domain object. For example, if domain object
 * "example.com" is transferred and host object "ns1.example.com" exists, the
 * host object MUST be transferred as part of the "example.com" transfer
 * process. Host objects that are subject to transfer when transferring a domain
 * object are listed in the response to an EPP &lt;info&gt; command performed on
 * the domain object. <br>
 * <br>
 * {@code EPPDomainTransferResp} is the concrete {@code EPPReponse} associated
 * with {@code EPPDomainTransferCmd}. <br>
 * 
 * @see com.verisign.epp.codec.domain.EPPDomainTransferResp
 */
public class EPPDomainTransferCmd extends EPPTransferCmd {
  /**
   * XML local name for {@code EPPDomainTransferCmd}.
   */
  public static final String ELM_LOCALNAME = "transfer";

  /**
   * XML Element Name of {@code EPPDomainTransferCmd} root element.
   */
  public static final String ELM_NAME = EPPDomainMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * XML Element Name for the {@code name} attribute.
   */
  private final static String ELM_DOMAIN_NAME = "name";

  /**
   * Domain Name of domain to query.
   */
  private String name = null;

  /**
   * Registration Period
   */
  private EPPDomainPeriod period = null;

  /**
   * Authorization information.
   */
  private EPPAuthInfo authInfo = null;

  /**
   * Allocates a new {@code EPPDomainTransferCmd} with default attribute
   * values. the defaults include the following: <br>
   * <br>
   *
   * <ul>
   * <li>name is set to {@code null}</li>
   * <li>period is set to {@code null}</li>
   * <li>authInfo is set to to {@code null}</li>
   * </ul>
   *
   * <br>
   * The transaction ID, operation, name, and authInfo must be set before
   * invoking {@code encode}.
   */
  public EPPDomainTransferCmd() {
    this.name = null;
    this.period = null;
    this.authInfo = null;
  }

  /**
   * {@code EPPDomainTransferCmd} constructor that takes the required
   * attributes as arguments. The period attribute is set to
   * {@code UNSPEC_PERIOD} and will not be included when {@code encode} is
   * invoked.
   *
   * @param aTransId
   *           Transaction Id associated with the command.
   * @param aOp
   *           One of the {@code EPPCommand.OP_} constants associated with the
   *           transfer command.
   * @param aName
   *           Domain name to create.
   */
  public EPPDomainTransferCmd(String aTransId, String aOp, String aName) {
    super(aTransId, aOp);

    this.name = aName;
  }

  /**
   * {@code EPPDomainTransferCmd} constructor that takes the required
   * attributes plus the optional attibute {@code aPeriod}.
   *
   * @param aTransId
   *           Transaction Id associated with the command.
   * @param aOp
   *           One of the {@code EPPCommand.OP_} constants associated with the
   *           transfer command.
   * @param aName
   *           Domain name to create.
   * @param aAuthInfo
   *           Authorization Information for operating with the domain.
   * @param aPeriod
   *           Registration period to be added to the domain upon transfer.
   */
  public EPPDomainTransferCmd(String aTransId, String aOp, String aName, EPPAuthInfo aAuthInfo,
        EPPDomainPeriod aPeriod) {
    super(aTransId, aOp);

    this.name = aName;
    this.period = aPeriod;
    setAuthInfo(aAuthInfo);
  }

  /**
   * Gets the EPP command Namespace associated with
   * {@code EPPDomainTransferCmd}.
   *
   * @return {@code EPPDomainMapFactory.NS}
   */
  @Override
  public String getNamespace() {
    return EPPDomainMapFactory.NS;
  }

	/**
	 * Gets the key for the domain name object, which is the domain name.
	 * 
	 * @return The domain name if set; {@code null} otherwise.
	 */
	@Override
	public String getKey() {
		return this.getName();
	}
  
  /**
   * Validate the state of the {@code EPPDomainTransferCmd} instance. A valid
   * state means that all of the required attributes have been set. If
   * validateState returns without an exception, the state is valid. If the
   * state is not valid, the EPPCodecException will contain a description of
   * the error. throws EPPCodecException State error. This will contain the
   * name of the attribute that is not valid.
   *
   * @throws EPPCodecException
   *            When validation failed
   */
  void validateState() throws EPPCodecException {
    if (super.getOp() == null) {
      throw new EPPCodecException("op required attribute is not set");
    }

    // Domain Name
    if (this.name == null) {
      throw new EPPCodecException("name required attribute is not set");
    }
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPDomainTransferCmd} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Root DOM Element representing the {@code EPPDomainTransferCmd}
   *         instance.
   *
   * @exception EPPEncodeException
   *               Unable to encode {@code EPPDomainTransferCmd} instance.
   */
  @Override
  protected Element doEncode(Document aDocument) throws EPPEncodeException {
    try {
      validateState();
    }
    catch (EPPCodecException e) {
      throw new EPPEncodeException("EPPDomainTransferCmd invalid state: " + e);
    }

    Element root = aDocument.createElementNS(EPPDomainMapFactory.NS, ELM_NAME);

    // Name
    EPPUtil.encodeString(aDocument, root, this.name, EPPDomainMapFactory.NS,
          EPPDomainMapFactory.NS_PREFIX + ":" + ELM_DOMAIN_NAME);

    // Period with Attribute of Unit
    if ((this.period != null) && (!this.period.isPeriodUnspec())) {
      EPPUtil.encodeComp(aDocument, root, this.period);
    }

    // authInfo
    EPPUtil.encodeComp(aDocument, root, this.authInfo);

    return root;
  }

  /**
   * Decode the {@code EPPDomainTransferCmd} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPDomainTransferCmd} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  protected void doDecode(Element aElement) throws EPPDecodeException {
    // Domain Name
    this.name = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_DOMAIN_NAME);

    // Period
    this.period = (EPPDomainPeriod) EPPUtil.decodeComp(aElement, EPPDomainMapFactory.NS, EPPDomainPeriod.ELM_NAME,
          EPPDomainPeriod.class);

    // authInfo
    this.authInfo = (EPPAuthInfo) EPPUtil.decodeComp(aElement, EPPDomainMapFactory.NS,
          EPPDomainMapFactory.ELM_DOMAIN_AUTHINFO, EPPAuthInfo.class);
  }

  /**
   * Gets the domain name to query.
   *
   * @return Domain Name if defined; {@code null} otherwise.
   */
  public String getName() {
    return this.name;
  }

  /**
   * Sets the domain name to query.
   *
   * @param aName
   *           Domain Name
   */
  public void setName(String aName) {
    this.name = aName;
  }

  /**
   * Compare an instance of {@code EPPDomainTransferCmd} with this instance.
   *
   * @param aObject
   *           Object to compare with.
   *
   * @return {@code true} when equal; {@code false} otherwise.
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPDomainTransferCmd)) {
      return false;
    }

    // EPPTransferCmd
    if (!super.equals(aObject)) {
      return false;
    }

    EPPDomainTransferCmd theMap = (EPPDomainTransferCmd) aObject;

    // Name
    if (!((this.name == null) ? (theMap.name == null) : this.name.equals(theMap.name))) {
      return false;
    }

    // Period
    if (!((this.period == null) ? (theMap.period == null) : this.period.equals(theMap.period))) {
      return false;
    }

    // AuthInfo
    if (!((this.authInfo == null) ? (theMap.authInfo == null) : this.authInfo.equals(theMap.authInfo))) {
      return false;
    }

    return true;
  }

  /**
   * Clone {@code EPPDomainTransferCmd}.
   *
   * @return clone of {@code EPPDomainTransferCmd}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPDomainTransferCmd clone = (EPPDomainTransferCmd) super.clone();

    if (this.authInfo != null) {
      clone.authInfo = (EPPAuthInfo) this.authInfo.clone();
    }

    if (this.period != null) {
      clone.period = (EPPDomainPeriod) this.period.clone();
    }

    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Get authorization Information.
   *
   * @return com.verisign.epp.codec.domain.EPPAuthInfo
   */
  public EPPAuthInfo getAuthInfo() {
    return this.authInfo;
  }

  /**
   * Gets the registration period of the transfer command in years.
   *
   * @return Registration Period in years if defined; {@code null} otherwise.
   */
  public EPPDomainPeriod getPeriod() {
    return this.period;
  }

  /**
   * Set authorization information.
   *
   * @param newAuthInfo
   *           com.verisign.epp.codec.domain.EPPAuthInfo
   */
  public void setAuthInfo(EPPAuthInfo newAuthInfo) {
    if (newAuthInfo != null) {
      this.authInfo = newAuthInfo;
      this.authInfo.setRootName(EPPDomainMapFactory.NS, EPPDomainMapFactory.ELM_DOMAIN_AUTHINFO);
    }
  }

  /**
   * Sets the registration period of the transfer command.
   *
   * @param aPeriod
   *           Registration Period.
   */
  public void setPeriod(EPPDomainPeriod aPeriod) {
    this.period = aPeriod;
  }

}
