/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.domain;

import java.util.Date;
import java.util.Iterator;
import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPAuthInfo;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPFactory;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;


/**
 * Represents an EPP Domain &lt;domain:infData&gt; response to an
 * {@code EPPDomainInfoCmd}. When an &lt;info&gt; command has been processed
 * successfully, the EPP &lt;resData&gt; element MUST contain a child
 * &lt;domain:infData&gt; element that identifies the domain namespace and the
 * location of the domain schema. The &lt;domain:infData&gt; element contains
 * the following child elements: <br>
 * <br>
 *
 * <ul>
 * <li>A &lt;domain:name&gt; element that contains the fully qualified name of
 * the domain. Use {@code getName} and {@code setName} to get and set the
 * element.</li>
 * <li>A &lt;domain:roid&gt; element that contains the Repository Object
 * IDentifier assigned to the domain object when the object was created. Use
 * {@code getRoid} and {@code setRoid} to get and set the element.</li>
 * <li>One or more &lt;domain:status&gt; elements that contain the current
 * status descriptors associated with the domain. See the
 * {@code EPPDomainStatus} description for a list of valid status values. Use
 * {@code getStatus} and {@code setStatus} to get and set the elements.</li>
 * <li>If supported by the server, one &lt;domain:registrant&gt; element and one
 * or more &lt;domain:contact&gt; elements that contain identifiers for the
 * human or organizational social information objects associated with the domain
 * object. Use {@code getContacts} and {@code setContacts} to get and set the
 * elements. Contacts should only be specified if the Contact Mapping is
 * supported.</li>
 * <li>Zero or more &lt;domain:ns&gt; elements that contain the fully qualified
 * names of the name server objects associated with the domain object. Use
 * {@code getNs} and {@code setNs} to get and set the elements.</li>
 * <li>Zero or more &lt;domain:host&gt; elements that contain the fully
 * qualified names of the host objects created under this superordinate domain
 * object. Use {@code getHost} and {@code setHost} to get and set the
 * elements.</li>
 * <li>A &lt;domain:clID&gt; element that contains the identifier of the
 * sponsoring client. Use {@code getClientId} and {@code setClientId} to get and
 * set the element.</li>
 * <li>A &lt;domain:crID&gt; element that contains the identifier of the client
 * that created the domain name. Use {@code getCreatedBy} and
 * {@code setCreatedBy} to get and set the element.</li>
 * <li>A &lt;domain:crDate&gt; element that contains the date and time of domain
 * creation. Use {@code getCreatedDate} and {@code setCreatedDate} to get and
 * set the element.</li>
 * <li>A &lt;domain:exDate&gt; element that contains the date and time
 * identifying the end of the domain's registration period. Use
 * {@code getExpirationDate} and {@code setExpirationDate} to get and set the
 * element.</li>
 * <li>A &lt;domain:upID&gt; element that contains the identifier of the client
 * that last updated the domain name. This element MUST NOT be present if the
 * domain has never been modified. Use {@code getLastUpdatedBy} and
 * {@code setLastUpdatedBy} to get and set the element.</li>
 * <li>A &lt;domain:upDate&gt; element that contains the date and time of the
 * most recent domain modification. This element MUST NOT be present if the
 * domain has never been modified. Use {@code getLastUpdatedDate} and
 * {@code setLastUpdatedDate} to get and set the element.</li>
 * <li>A &lt;domain:trDate&gt; elements that contains the date and time of the
 * most recent successful transfer. This element MUST NOT be provided if the
 * domain has never been transferred. Use {@code getLastTransferDate} and
 * {@code setLastTransferDate} to get and set the element.</li>
 * <li>An OPTIONAL &lt;domain:authInfo&gt; element that contains authorization
 * information associated with the domain object. This element MUST NOT be
 * provided if the querying client is not the current sponsoring client. Use
 * {@code getAuthInfo} and {@code setAuthInfo} to get and set the elements.</li>
 * </ul>
 *
 * @see com.verisign.epp.codec.domain.EPPDomainInfoCmd
 */
public class EPPDomainInfoResp extends EPPResponse {

  /**
   * XML local name for {@code EPPDomainInfoResp}.
   */
  public static final String ELM_LOCALNAME = "infData";

  /**
   * XML Element Name of {@code EPPDomainInfoResp} root element.
   */
  public static final String ELM_NAME = EPPDomainMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /**
   * Admin contact type constant.
   */
  public static final String CONTACT_TYPE_ADMIN = EPPDomainContact.TYPE_ADMINISTRATIVE;

  /**
   * Tech contact type constant.
   */
  public static final String CONTACT_TYPE_TECH = EPPDomainContact.TYPE_TECHNICAL;

  /**
   * Billing contact type constant.
   */
  public static final String CONTACT_TYPE_BILLING = EPPDomainContact.TYPE_BILLING;

  /**
   * XML tag name for the {@code name} attribute.
   */
  private final static String ELM_DOMAIN_NAME = "name";

  /**
   * XML tag name for the {@code statuses} attribute.
   */
  private final static String ELM_STATUS = "status";

  /**
   * XML tag name for the {@code contacts} attribute.
   */
  private final static String ELM_CONTACT = "contact";

  /**
   * XML tag name for the {@code clientId} attribute.
   */
  private final static String ELM_CLID = "clID";

  /**
   * XML tag name for the {@code createdDate} attribute.
   */
  private final static String ELM_CRDATE = "crDate";

  /**
   * XML tag name for the {@code createdBy} attribute.
   */
  private final static String ELM_CRID = "crID";

  /**
   * XML tag name for the {@code expirationDate} attribute.
   */
  private final static String ELM_EXDATE = "exDate";

  /**
   * XML tag name for the {@code servers} attribute.
   */
  private final static String ELM_HOST = "host";

  /**
   * XML tag name for the {@code childServers} attribute.
   */
  private final static String ELM_NS = "ns";

  /**
   * XML tag name for host object reference
   */
  private final static String ELM_HOST_OBJ = "hostObj";

  /** XML tag name for host attribute */
  private final static String ELM_HOST_ATTR = EPPHostAttr.ELM_NAME;

  /**
   * XML tag name for the {@code registrant} attribute.
   */
  private final static String ELM_REGISTRANT = "registrant";

  /**
   * XML tag name for the {@code roid} attribute.
   */
  private final static String ELM_ROID = "roid";

  /**
   * XML tag name for the {@code lastTransferDate} attribute.
   */
  private final static String ELM_TRDATE = "trDate";

  /**
   * XML tag name for the {@code lastUpdatedDate} attribute.
   */
  private final static String ELM_UPDATE = "upDate";

  /**
   * XML tag name for the {@code lastUpdatedBy} attribute.
   */
  private final static String ELM_UPID = "upID";

  /**
   * Category for logging
   */
      private static Logger cat = LoggerFactory.getLogger(EPPDomainInfoResp.class);
        

  /**
   * Fully qualified name of the domain
   */
  private String name = null;

  /**
   * Identifier of sponsoring client
   */
  private String clientId = null;

  /**
   * {@code Vector} of {@code EPPDomainContact} instances associated with
   * domain
   */
  private Vector<EPPDomainContact> contacts = new Vector<EPPDomainContact>();

  /**
   * Identifier of the client that created the domain name
   */
  private String createdBy = null;

  /**
   * Date and time of domain creation
   */
  private Date createdDate = null;

  /**
   * Date and time identifying the end of the domain's registration period
   */
  private Date expirationDate = null;

  /**
   * Identifier of the client that last updated the domain name
   */
  private String lastUpdatedBy = null;

  /**
   * Date and time of the most recent domain modification
   */
  private Date lastUpdatedDate = null;

  /**
   * Date and time of the most recent successful transfer
   */
  private Date lastTransferDate = null;

  /**
   * Authorization information.
   */
  private EPPAuthInfo authInfo = null;

  /**
   * Registrant.
   */
  private String registrant = null;

  /**
   * Names of host objects.
   */
  private Vector<String> hosts = new Vector<String>();

  /**
   * Names of name server object names.
   */
  private Vector<String> nses = new Vector<String>();

  /**
   * name server attributes.
   */
  private Vector<EPPHostAttr> nsAttrs = new Vector<EPPHostAttr>();

  /**
   * One or more current status descriptors.
   */
  private Vector<EPPDomainStatus> statuses = new Vector<EPPDomainStatus>();

  /**
   * Registry Object IDentifier (ROID).
   */
  private java.lang.String roid = null;

  /**
   * {@code EPPDomainInfoResp} default constructor. Must call required setter
   * methods before invoking {@code encode}, which include:<br>
   * <br>
   *
   * <ul>
   * <li>name - {@code setName}</li>
   * <li>roid - {@code setRoid}</li>
   * <li>client id - {@code setClientId}</li>
   * <li>statuses - {@code setStatuses}</li>
   * <li>created by - {@code setCreatedBy}</li>
   * <li>created date - {@code setCreatedDate}</li>
   * <li>transaction id - {@code setTransId}</li>
   * </ul>
   */
  public EPPDomainInfoResp() {
    // Default values set in attribute definitions.
  }

  /**
   * {@code EPPDomainInfoResp} constructor that takes the required attribute
   * values as parameters. The setter methods of the optional attributes can be
   * called before invoking {@code encode}.
   *
   * @param aTransId
   *           Transaction Id associated with response.
   * @param aRoid
   *           roid
   * @param aName
   *           Domain name
   * @param aClientId
   *           Owning Client Id
   * @param aStatuses
   *           Current status descriptors associated with the domain.
   * @param aCreatedBy
   *           Client Id of Registrar that created the domain
   * @param aCreatedDate
   *           Date the domain was created
   * @param aAuthInfo
   *           Authorization info for domain name.  Can be set to {@code null},
   */
  public EPPDomainInfoResp(EPPTransId aTransId, String aRoid, String aName, String aClientId,
        Vector<EPPDomainStatus> aStatuses, String aCreatedBy, Date aCreatedDate, EPPAuthInfo aAuthInfo) {
    super(aTransId);

    this.name = aName;
    this.roid = aRoid;
    this.clientId = aClientId;
    this.setStatuses(aStatuses);
    this.createdBy = aCreatedBy;
    this.createdDate = aCreatedDate;
    this.authInfo = aAuthInfo;
    if (this.authInfo != null) {
      this.authInfo.setRootName(EPPDomainMapFactory.NS, EPPDomainMapFactory.ELM_DOMAIN_AUTHINFO);
    }
  }

  /**
   * Is at least one subordinate host object defined?
   * 
   * @return {@code true} if at least one subordinate host object is defined; {@code false}
   *         otherwise.
   */
  public boolean hasHosts() {
    return !this.hosts.isEmpty();
  }
  
  /**
   * Gets the fully qualified names of the subordinate host objects that exist
   * under the superordinate domain object.
   *
   * @return {@code Vector} of host name {@code String} instances for fully
   *         qualified names of the subordinate host objects that exist under
   *         the superordinate domain object. A non-{@code null} {@code Vector}
   *         is always returned.
   */
  public Vector<String> getHosts() {
    return this.hosts;
  }

  /**
   * Sets the fully qualified names of the subordinate host objects that exist
   * under the superordinate domain object.
   *
   * @param aHosts
   *           {@code Vector} of host name {@code String} instances for fully
   *           qualified names of the subordinate host objects that exist under
   *           the superordinate domain object. Set to {@code null} if there
   *           are no subordinate host objects.
   */
  public void setHosts(Vector<String> aHosts) {
    if (aHosts == null) {
      this.hosts = new Vector<String>();
    }
    else {
      this.hosts = aHosts;
    }
  }

  /**
   * Adds a subordinate host to the list of subordinate hosts.
   * 
   * @param aHost
   *           Host name of the subordinate host
   */
  public void addHost(String aHost) {
    if (aHost != null) {
      this.hosts.add(aHost);
    }
  }
  
  
  /**
   * Is at least one name server defined?
   * 
   * @return {@code true} if at least one name server is defined; {@code false}
   *         otherwise.
   */
  public boolean hasNses() {
    return !this.nses.isEmpty();
  }

  /**
   * Gets the name servers. The name servers are the fully qualified name of a
   * known name server host object.
   *
   * @return {@code Vector} of name server {@code String} instances for host
   *         object references. A non-{@code null} {@code Vector} is always
   *         returned.
   */
  public Vector<String> getNses() {
    return this.nses;
  }

  /**
   * Sets the name servers. The name servers are the fully qualified name of a
   * known name server host object.
   *
   * @param aServers
   *           {@code Vector} of name server {@code String} instances for host
   *           object references.
   */
  public void setNses(Vector<String> aServers) {
    if (aServers == null) {
      this.nses = new Vector<String>();
    }
    else {
      this.nses = aServers;
    }
  }

  /**
   * Adds a name server object name to the list of name servers.
   * 
   * @param aNs
   *           Name server name
   */
  public void addNs(String aNs) {
    if (aNs != null) {
      this.nses.add(aNs);
    }
  }

  /**
   * Is at least one name server attribute defined?
   * 
   * @return {@code true} if at least one name server attribute is defined;
   *         {@code false} otherwise.
   */
  public boolean hasNsAttrs() {
    return !this.nsAttrs.isEmpty();
  }

  /**
   * Gets the name server attributes. The name server attributes are
   * {@code EPPHostAttr} instances containing the fully qualified name of a
   * host and optionally the host IP addresses.
   *
   * @return {@code Vector} of {@link EPPHostAttr} instances for host attribute
   *         values if exists. A non-{@code null} {@code Vector} is always
   *         returned.
   */
  public Vector<EPPHostAttr> getNsAttrs() {
    return this.nsAttrs;
  }

  /**
   * Sets the name server attributes. The name server attributes are
   * {@link EPPHostAttr} instances containing the fully qualified name of a
   * host and optionally the host IP addresses.
   *
   * @param aServers
   *           {@code Vector} of {@link EPPHostAttr} instances for host
   *           attribute values. Set to {@code null} if there are no name
   *           server attributes.
   */
  public void setNsAttrs(Vector<EPPHostAttr> aServers) {
    if (aServers == null) {
      this.nsAttrs = new Vector<EPPHostAttr>();
    }
    else {
      this.nsAttrs = aServers;
    }
  }

  /**
   * Adds a name server attribute to the list of name server attributes.
   * 
   * @param aNsAttr
   *           Name server attribute
   */
  public void addNsAttr(EPPHostAttr aNsAttr) {
    if (aNsAttr != null) {
      this.nsAttrs.add(aNsAttr);
    }
  }

  /**
   * Is at least one status defined?
   * 
   * @return {@code true} if at least one status is defined; {@code false}
   *         otherwise.
   */
  public boolean hasStatuses() {
    return !this.statuses.isEmpty();
  }

  /**
   * Get the current associated statuses
   *
   * @return {@code Vector} of {@link EPPDomainStatus} instances. A
   *         non-{@code null} {@code Vector} is always returned.
   */
  public Vector<EPPDomainStatus> getStatuses() {
    return this.statuses;
  }

  /**
   * Set associated statuses.
   *
   * @param aStatuses
   *           {@code Vector} of {@link EPPDomainStatus} instances. Set to
   *           {@code null} to clear the statuses.
   */
  public void setStatuses(Vector<EPPDomainStatus> aStatuses) {
    if (aStatuses == null) {
      this.statuses = new Vector<EPPDomainStatus>();
    }
    else {
      this.statuses = aStatuses;
    }
  }

  /**
   * Adds a status to the list of statuses.
   * 
   * @param aStatus
   *           Status to add to the list of statuses
   */
  public void addStatus(EPPDomainStatus aStatus) {
    if (aStatus != null) {
      this.statuses.add(aStatus);
    }
  }

  /**
   * Gets the EPP response type associated with {@code EPPDomainInfoResp} .
   *
   * @return {@code EPPDomainInfoResp.ELM_NAME}
   */
  @Override
  public String getType() {
    return ELM_NAME;
  }

  /**
   * Gets the EPP command namespace associated with {@code EPPDomainInfoResp}.
   *
   * @return {@code EPPDomainMapFactory.NS}
   */
  @Override
  public String getNamespace() {
    return EPPDomainMapFactory.NS;
  }

  /**
   * Compare an instance of {@code EPPDomainInfoResp} with this instance.
   *
   * @param aObject
   *           Object to compare with.
   *
   * @return {@code true} if equal; {@code false} otherwise.
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPDomainInfoResp)) {
      return false;
    }

    if (!super.equals(aObject)) {
      return false;
    }

    EPPDomainInfoResp theInfoData = (EPPDomainInfoResp) aObject;

    // Name
    if (!(this.name == null ? theInfoData.name == null : this.name.equals(theInfoData.name))) {
      return false;
    }

    // roid
    if (!(this.roid == null ? theInfoData.roid == null : this.roid.equals(theInfoData.roid))) {
      return false;
    }

    // Client Id
    if (!(this.clientId == null ? theInfoData.clientId == null : this.clientId.equals(theInfoData.clientId))) {
      return false;
    }

    // Statuses
    if (!EPPUtil.equalVectors(this.statuses, theInfoData.statuses)) {
      return false;
    }

    // registrant
    if (!(this.registrant == null ? theInfoData.registrant == null :
          this.registrant.equals(theInfoData.registrant))) {
      return false;
    }

    // Contacts
    if (EPPFactory.getInstance().hasService(EPPDomainMapFactory.NS_CONTACT)) {
      if (!EPPUtil.equalVectors(this.contacts, theInfoData.contacts)) {
        return false;
      }
    }

    // name servers
    if (!EPPUtil.equalVectors(this.nses, theInfoData.nses)) {
      return false;
    }
    if (!EPPUtil.equalVectors(this.nsAttrs, theInfoData.nsAttrs)) {
      return false;
    }

    // hosts
    if (!EPPUtil.equalVectors(this.hosts, theInfoData.hosts)) {
      return false;
    }

    // Created By
    if (!(this.createdBy == null ? theInfoData.createdBy == null : this.createdBy.equals(theInfoData.createdBy))) {
      return false;
    }

    // Created Date
    if (!(this.createdDate == null ? theInfoData.createdDate == null :
          this.createdDate.equals(theInfoData.createdDate))) {
      return false;
    }

    // Expiration Date
    if (!(this.expirationDate == null ? theInfoData.expirationDate == null :
          this.expirationDate.equals(theInfoData.expirationDate))) {
      return false;
    }

    // Last Updated By
    if (!(this.lastUpdatedBy == null ? theInfoData.lastUpdatedBy == null :
          this.lastUpdatedBy.equals(theInfoData.lastUpdatedBy))) {
      return false;
    }

    // Last Updated Date
    if (!(this.lastUpdatedDate == null ? theInfoData.lastUpdatedDate == null :
          this.lastUpdatedDate.equals(theInfoData.lastUpdatedDate))) {
      return false;
    }

    // Last Transfer Date
    if (!(this.lastTransferDate == null ? theInfoData.lastTransferDate == null :
          this.lastTransferDate.equals(theInfoData.lastTransferDate))) {
      return false;
    }

    // Authorization Info
    if (!(this.authInfo == null ? theInfoData.authInfo == null : this.authInfo.equals(theInfoData.authInfo))) {
      return false;
    }

    return true;
  }

  /**
   * Clone {@code EPPDomainInfoResp}.
   *
   * @return clone of {@code EPPDomainInfoResp}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPDomainInfoResp clone = (EPPDomainInfoResp) super.clone();

    clone.statuses = (Vector<EPPDomainStatus>) this.statuses.clone();

    if (this.contacts != null) {
      clone.contacts = (Vector<EPPDomainContact>) this.contacts.clone();
    }

    if (this.nses != null) {
      clone.nses = (Vector<String>) this.nses.clone();
    }

    if (this.nsAttrs != null) {
      clone.nsAttrs = (Vector<EPPHostAttr>) this.nsAttrs.clone();
    }

    if (this.hosts != null) {
      clone.hosts = (Vector<String>) this.hosts.clone();
    }

    if (this.authInfo != null) {
      clone.authInfo = (EPPAuthInfo) this.authInfo.clone();
    }

    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Gets the domain name
   *
   * @return Domain Name if defined; {@code null} otherwise.
   */
  public String getName() {
    return this.name;
  }

  /**
   * Sets the domain name.
   *
   * @param aName
   *           Domain Name
   */
  public void setName(String aName) {
    this.name = aName;
  }

  /**
   * Gets the domain owning Client Id.
   *
   * @return Client Id
   */
  public String getClientId() {
    return this.clientId;
  }

  /**
   * Sets the domain owning Client Id.
   *
   * @param aClientId
   *           Client Id
   */
  public void setClientId(String aClientId) {
    this.clientId = aClientId;
  }

  /**
   * Is at least one contact defined?
   * 
   * @return {@code true} if at least one contact is defined; {@code false}
   *         otherwise.
   */
  public boolean hasContacts() {
    return !this.contacts.isEmpty();
  }

  /**
   * Gets the Contacts
   *
   * @return {@code Vector} of {@code EPPDomainContact} instances if contacts
   *         exist; {@code null} otherwise.
   */
  public Vector<EPPDomainContact> getContacts() {
    if (this.contacts.isEmpty()) {
      return null; // Returned as null for backward compatibility
    }
    else {
      return this.contacts;
    }
  }

  /**
   * Gets a contact by type using one of the {@code CONTACT_TYPE} constants.
   *
   * @param aType
   *           Type of constant using one of the {@code CONTACT_TYPE}
   *           constants.
   *
   * @return Contact by type if found; {@code null} otherwise.
   */
  public EPPDomainContact getContactByType(String aType) {
    if (this.contacts != null) {
      Iterator<EPPDomainContact> contactIter = this.contacts.iterator();
      while (contactIter.hasNext()) {
        EPPDomainContact contact = contactIter.next();
        if (contact != null && contact.getType().equals(aType)) {
          // found
          return contact;
        }
      }
    }
    // not found
    return null;
  }

  /**
   * Has the admin contact been set?
   *
   * @return {@code true} if the admin contact has been set; {@code false}
   *         otherwise.
   */
  public boolean hasAdminContact() {
    return (this.getAdminContact() != null ? true : false);
  }

  /**
   * Gets the admin contact if defined.
   *
   * @return Admin contact if defined; {@code null} otherwise.
   */
  public EPPDomainContact getAdminContact() {
    return this.getContactByType(CONTACT_TYPE_ADMIN);
  }

  /**
   * Has the tech contact been set?
   *
   * @return {@code true} if the tech contact has been set; {@code false}
   *         otherwise.
   */
  public boolean hasTechContact() {
    return (this.getTechContact() != null ? true : false);
  }

  /**
   * Gets the tech contact if defined.
   *
   * @return Tech contact if defined; {@code null} otherwise.
   */
  public EPPDomainContact getTechContact() {
    return this.getContactByType(CONTACT_TYPE_TECH);
  }

  /**
   * Has the billing contact been set?
   *
   * @return {@code true} if the billing contact has been set; {@code false}
   *         otherwise.
   */
  public boolean hasBillingContact() {
    return (this.getBillingContact() != null ? true : false);
  }

  /**
   * Gets the billing contact if defined.
   *
   * @return Billing contact if defined; {@code null} otherwise.
   */
  public EPPDomainContact getBillingContact() {
    return this.getContactByType(CONTACT_TYPE_BILLING);
  }

  /**
   * Sets the Contacts. This method should only be called if the Contact
   * Namespace supported.
   *
   * @param aContacts
   *           - {@code Vector} of {@code EPPDomainContact} instances
   */
  public void setContacts(Vector<EPPDomainContact> aContacts) {
    if (EPPFactory.getInstance().hasService(EPPDomainMapFactory.NS_CONTACT)) {
      if (aContacts == null) {
        this.contacts = new Vector<EPPDomainContact>();
      }
      else {
        this.contacts = aContacts;
      }
    }
  }

  /**
   * Adds a contact to the list of contacts.
   * 
   * @param aContact
   *           Contact to add to the list of contacts
   */
  public void addContact(EPPDomainContact aContact) {
    if (EPPFactory.getInstance().hasService(EPPDomainMapFactory.NS_CONTACT)) {
      if (aContact != null) {
        this.contacts.add(aContact);
      }
    }
  }

  /**
   * Gets Client Id that created the domain.
   *
   * @return Client Id if defined; {@code null} otherwise.
   */
  public String getCreatedBy() {
    return this.createdBy;
  }

  /**
   * Sets Client Id that created the domain.
   *
   * @param aCreatedBy
   *           Client Id that created the domain.
   */
  public void setCreatedBy(String aCreatedBy) {
    this.createdBy = aCreatedBy;
  }

  /**
   * Gets the date and time the domain was created.
   *
   * @return Date and time the domain was created if defined; {@code null}
   *         otherwise.
   */
  public Date getCreatedDate() {
    return this.createdDate;
  }

  /**
   * Sets the date and time the domain was created.
   *
   * @param aDate
   *           Date and time the domain was created.
   */
  public void setCreatedDate(Date aDate) {
    this.createdDate = aDate;
  }

  /**
   * Gets the expiration date and time of the domain.
   *
   * @return Expiration date and time of the domain if defined; {@code null}
   *         otherwise.
   */
  public Date getExpirationDate() {
    return this.expirationDate;
  }

  /**
   * Sets the expiration date and time of the domain.
   *
   * @param aExpirationDate
   *           Expiration date and time of the domain.
   */
  public void setExpirationDate(Date aExpirationDate) {
    this.expirationDate = aExpirationDate;
  }

  /**
   * Has the last updated by been set?
   *
   * @return {@code true} if the last updated by has been set; {@code false}
   *         otherwise.
   */
  public boolean hasLastUpdatedBy() {
    return (this.lastUpdatedBy != null ? true : false);
  }

  /**
   * Gets the Client Id that last updated the domain. This will be null if the
   * domain has not been updated since creation.
   *
   * @return Client Id that last updated the domain has been updated;
   *         {@code null} otherwise.
   */
  public String getLastUpdatedBy() {
    return this.lastUpdatedBy;
  }

  /**
   * Sets the Client Id that last updated the domain.
   *
   * @param aLastUpdatedBy
   *           Client Id String that last updated the domain.
   */
  public void setLastUpdatedBy(String aLastUpdatedBy) {
    this.lastUpdatedBy = aLastUpdatedBy;
  }

  /**
   * Has the last updated date been set?
   *
   * @return {@code true} if the last updated date has been set; {@code false}
   *         otherwise.
   */
  public boolean hasLastUpdatedDate() {
    return (this.lastUpdatedDate != null ? true : false);
  }

  /**
   * Gets the date and time of the last domain update. This will be
   * {@code null} if the domain has not been updated since creation.
   *
   * @return date and time of the last domain update if defined; {@code null}
   *         otherwise.
   */
  public Date getLastUpdatedDate() {
    return this.lastUpdatedDate;
  }

  /**
   * Sets the last date and time the domain was updated.
   *
   * @param aLastUpdatedDate
   *           Date and time of the last domain update.
   */
  public void setLastUpdatedDate(Date aLastUpdatedDate) {
    this.lastUpdatedDate = aLastUpdatedDate;
  }

  /**
   * Has the last transfer date been set?
   *
   * @return {@code true} if the last transfer date has been set; {@code false}
   *         otherwise.
   */
  public boolean hasLastTransferDate() {
    return (this.lastTransferDate != null ? true : false);
  }

  /**
   * Gets the date and time of the last successful domain transfer. This will
   * be {@code null} if the domain has not been successfully transferred since
   * creation.
   *
   * @return date and time of the last successful transfer if defined;
   *         {@code null} otherwise.
   */
  public Date getLastTransferDate() {
    return this.lastTransferDate;
  }

  /**
   * Sets the last date and time the domain was successfully transferred.
   *
   * @param aLastTransferDate
   *           Date and time of the last succesful transfer
   */
  public void setLastTransferDate(Date aLastTransferDate) {
    this.lastTransferDate = aLastTransferDate;
  }

  /**
   * Has the authorization information been set?
   *
   * @return {@code true} if the authorization information has been set;
   *         {@code false} otherwise.
   */
  public boolean hasAuthInfo() {
    return (this.authInfo != null ? true : false);
  }

  /**
   * Get authorization information
   *
   * @return Authorization information if defined; {@code null} otherwise;
   */
  public EPPAuthInfo getAuthInfo() {
    return this.authInfo;
  }

  /**
   * Set authorization information
   *
   * @param aAuthInfo
   *           EPPAuthInfo
   */
  public void setAuthInfo(EPPAuthInfo aAuthInfo) {
    this.authInfo = aAuthInfo;
    if (aAuthInfo != null) {
      this.authInfo.setRootName(EPPDomainMapFactory.NS, EPPDomainMapFactory.ELM_DOMAIN_AUTHINFO);
    }
  }

  /**
   * Has the registrant been set?
   *
   * @return {@code true} if the registrant has been set; {@code false}
   *         otherwise.
   */
  public boolean hasRegistrant() {
    return (this.registrant != null ? true : false);
  }

  /**
   * Get registrant
   *
   * @return String
   */
  public String getRegistrant() {
    return this.registrant;
  }

  /**
   * Set registrants.
   *
   * @param newRegistrant
   *           String
   */
  public void setRegistrant(String newRegistrant) {
    this.registrant = newRegistrant;
  }

  /**
   * Has registry object identifier (roid) been set?
   *
   * @return {@code true} if the registry object identifier (roid) has been
   *         set; {@code false} otherwise.
   */
  public boolean hasRoid() {
    return (this.roid != null ? true : false);
  }

  /**
   * Get roid.
   *
   * @return registry object identifier (roid)
   */
  public String getRoid() {
    return this.roid;
  }

  /**
   * Set registry object identifier (roid).
   *
   * @param aRoid
   *           registry object identifier (roid) value
   */
  public void setRoid(String aRoid) {
    this.roid = aRoid;
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPDomainInfoResp} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the EPPDomainInfoResp
   *         instance.
   *
   * @exception EPPEncodeException
   *               Unable to encode EPPDomainInfoResp instance.
   */
  @Override
  protected Element doEncode(Document aDocument) throws EPPEncodeException {
    try {
      validateState();
    }
    catch (EPPCodecException e) {
      cat.error("EPPDomainInfoResp.doEncode(): Invalid state on encode: " + e);
      throw new EPPEncodeException("Invalid state on EPPDomainInfoResp.encode: " + e);
    }

    Element root = aDocument.createElementNS(EPPDomainMapFactory.NS, ELM_NAME);

    // Name
    EPPUtil.encodeString(aDocument, root, this.name, EPPDomainMapFactory.NS,
          EPPDomainMapFactory.NS_PREFIX + ":" + ELM_DOMAIN_NAME);

    // roid
    EPPUtil.encodeString(aDocument, root, this.roid, EPPDomainMapFactory.NS,
          EPPDomainMapFactory.NS_PREFIX + ":" + ELM_ROID);

    // Statuses
    EPPUtil.encodeCompVector(aDocument, root, this.statuses);

    // registrant
    if (this.registrant != null) {
      EPPUtil.encodeString(aDocument, root, this.registrant, EPPDomainMapFactory.NS,
            EPPDomainMapFactory.NS_PREFIX + ":" + ELM_REGISTRANT);
    }

    // Contacts
    if (this.hasContacts()) {
      if (EPPFactory.getInstance().hasService(EPPDomainMapFactory.NS_CONTACT)) {
        EPPUtil.encodeCompVector(aDocument, root, this.contacts);
      }
      else {
        throw new EPPEncodeException("Contacts specified when the Contact Mapping is not supported");
      }
    }

    // name servers
    if (this.hasNses() || this.hasNsAttrs()) {
      Element theServersElm = aDocument.createElementNS(EPPDomainMapFactory.NS,
            EPPDomainMapFactory.NS_PREFIX + ":" + ELM_NS);
      root.appendChild(theServersElm);

      if (this.hasNses()) {
        EPPUtil.encodeVector(aDocument, theServersElm, this.nses, EPPDomainMapFactory.NS,
              EPPDomainMapFactory.NS_PREFIX + ":" + ELM_HOST_OBJ);
      }
      else { // this.hasNsAttrs()
        EPPUtil.encodeCompVector(aDocument, theServersElm, this.nsAttrs);
      }

    }

    // end if (this.servers != null) && (this.servers.size()) > 0)
    // hosts
    if (this.hosts != null) {
      EPPUtil.encodeVector(aDocument, root, this.hosts, EPPDomainMapFactory.NS,
            EPPDomainMapFactory.NS_PREFIX + ":" + ELM_HOST);
    }

    // Client Id
    EPPUtil.encodeString(aDocument, root, this.clientId, EPPDomainMapFactory.NS,
          EPPDomainMapFactory.NS_PREFIX + ":" + ELM_CLID);

    // Created By
    EPPUtil.encodeString(aDocument, root, this.createdBy, EPPDomainMapFactory.NS,
          EPPDomainMapFactory.NS_PREFIX + ":" + ELM_CRID);

    // Created Date
    EPPUtil.encodeTimeInstant(aDocument, root, this.createdDate, EPPDomainMapFactory.NS,
          EPPDomainMapFactory.NS_PREFIX + ":" + ELM_CRDATE);

    // Last Updated By
    if (this.lastUpdatedBy != null) {
      EPPUtil.encodeString(aDocument, root, this.lastUpdatedBy, EPPDomainMapFactory.NS,
            EPPDomainMapFactory.NS_PREFIX + ":" + ELM_UPID);
    }

    // Last Updated Date
    if (this.lastUpdatedDate != null) {
      EPPUtil.encodeTimeInstant(aDocument, root, this.lastUpdatedDate, EPPDomainMapFactory.NS,
            EPPDomainMapFactory.NS_PREFIX + ":" + ELM_UPDATE);
    }

    // Expiration Date
    if (this.expirationDate != null) {
      EPPUtil.encodeTimeInstant(aDocument, root, this.expirationDate, EPPDomainMapFactory.NS,
            EPPDomainMapFactory.NS_PREFIX + ":" + ELM_EXDATE);
    }

    // Last Transfer Date
    if (this.lastTransferDate != null) {
      EPPUtil.encodeTimeInstant(aDocument, root, this.lastTransferDate, EPPDomainMapFactory.NS,
            EPPDomainMapFactory.NS_PREFIX + ":" + ELM_TRDATE);
    }

    // Authorization Info
    if (this.authInfo != null) {
      EPPUtil.encodeComp(aDocument, root, this.authInfo);
    }

    return root;
  }

  /**
   * Decode the {@code EPPDomainInfoResp} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPDomainInfoResp} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  protected void doDecode(Element aElement) throws EPPDecodeException {
    // Name
    this.name = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_DOMAIN_NAME);

    // roid
    this.roid = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_ROID);

    // Statuses
    this.statuses = EPPUtil.decodeCompVector(aElement, EPPDomainMapFactory.NS, ELM_STATUS, EPPDomainStatus.class);

    // registant
    this.registrant = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_REGISTRANT);

    // Contacts
    this.contacts = EPPUtil.decodeCompVector(aElement, EPPDomainMapFactory.NS, ELM_CONTACT, EPPDomainContact.class);

    // name servers
    Element theServersElm = EPPUtil.getElementByTagNameNS(aElement, EPPDomainMapFactory.NS, ELM_NS);

    if (theServersElm != null) {
      Element theServerElm = EPPUtil.getFirstElementChild(theServersElm);

      if (theServerElm != null) {
        if (theServerElm.getLocalName().equals(EPPUtil.getLocalName(ELM_HOST_OBJ))) {
          this.nses = EPPUtil.decodeVector(theServersElm, EPPDomainMapFactory.NS, ELM_HOST_OBJ);
        }
        else if (theServerElm.getLocalName().equals(EPPUtil.getLocalName(ELM_HOST_ATTR))) {
          this.nsAttrs = EPPUtil.decodeCompVector(theServersElm, EPPDomainMapFactory.NS, ELM_HOST_ATTR,
                EPPHostAttr.class);
        }
        else {
          throw new EPPDecodeException(
                "EPPDomainCreateCmd.doDecode: Invalid host child element " + theServersElm.getLocalName());
        }
      }

      // end if (theServerElm != null)
    }

    // end if (theServersElm != null)
    // Child Servers
    this.hosts = EPPUtil.decodeVector(aElement, EPPDomainMapFactory.NS, ELM_HOST);

    // Client Id
    this.clientId = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_CLID);

    // Created By
    this.createdBy = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_CRID);

    // Created Date
    this.createdDate = EPPUtil.decodeTimeInstant(aElement, EPPDomainMapFactory.NS, ELM_CRDATE);

    // Expiration Date
    this.expirationDate = EPPUtil.decodeTimeInstant(aElement, EPPDomainMapFactory.NS, ELM_EXDATE);

    // Last Updated By
    this.lastUpdatedBy = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_UPID);

    // Last Updated Date
    this.lastUpdatedDate = EPPUtil.decodeTimeInstant(aElement, EPPDomainMapFactory.NS, ELM_UPDATE);

    // Last Transfer Date
    this.lastTransferDate = EPPUtil.decodeTimeInstant(aElement, EPPDomainMapFactory.NS, ELM_TRDATE);

    // Authorization Info
    this.authInfo = (EPPAuthInfo) EPPUtil.decodeComp(aElement, EPPDomainMapFactory.NS,
          EPPDomainMapFactory.ELM_DOMAIN_AUTHINFO, EPPAuthInfo.class);
  }

  /**
   * Validate the state of the {@code EPPDomainInfoResp} instance. A valid
   * state means that all of the required attributes have been set. If
   * validateState returns without an exception, the state is valid. If the
   * state is not valid, the EPPCodecException will contain a description of
   * the error. throws EPPCodecException State error. This will contain the
   * name of the attribute that is not valid.
   *
   * @throws EPPCodecException
   *            With invalid state
   */
  void validateState() throws EPPCodecException {

    if (this.name == null) {
      throw new EPPCodecException("name required attribute is not set");
    }

    if (this.roid == null) {
      throw new EPPCodecException("roid required attribute is not set");
    }

    if (this.clientId == null) {
      throw new EPPCodecException("clientId required attribute is not set");
    }

    if (this.hasNses() && this.hasNsAttrs()) {
      throw new EPPCodecException("both name server objects and attributes are defined");
    }
  }

}
